# DX-13: Incremental Proof Verification

**Status:** Proposed
**Created:** 2025-12-21
**Problem:** `invar guard --prove` takes 6+ minutes, blocking Agent workflow

## Executive Summary

CrossHair symbolic verification is powerful but slow. This proposal implements **zero-configuration optimizations** that reduce `--prove` time from 6+ minutes to ~10 seconds while maintaining verification completeness.

**Key insight:** Agent-Native means the system decides what to optimize, not the Agent.

---

## Problem Analysis

### Current Behavior

```
$ invar guard --prove
Verification: --prove (static + doctests + CrossHair)
[Waiting 6+ minutes...]
✓ Verified: 40 files
```

### Root Causes

| Factor | Impact | Current | Optimized |
|--------|--------|---------|-----------|
| All files checked | 🔴 High | 40 files | 2 (changed only) |
| Sequential execution | 🔴 High | 1 worker | 4 workers |
| Fixed timeout | 🟡 Medium | 10s/condition | Adaptive iterations |
| No caching | 🟡 Medium | Re-verify all | Cache verified |
| Contract-less files | 🟢 Low | 0.1s waste/file | Skip (minor gain) |

### Time Breakdown (40 core files)

```
Current:
  Startup overhead:     40 × 0.1s  =   4s
  Verification:         40 × 10s   = 400s
  Total:                           = 404s (6.7 minutes)

Optimized (2 changed files, 4 workers):
  Contract detection:   40 × 0.02ms = 0.001s
  Verification:          2 × 4s / 1  =   8s  (parallel, fast mode)
  Cache lookup:         38 × 0.001s =  0.04s
  Total:                            ≈  8s
```

---

## Optimization Strategies

### Priority 1: Automatic Incremental Mode (8.5x speedup)

**Principle:** Only verify what changed. Zero flags needed.

```python
def get_files_to_prove(path: Path, all_core_files: list[Path]) -> list[Path]:
    """
    Agent-Native: Automatically select files for verification.

    Logic:
    1. If git repo with changes → only changed core files
    2. If no changes → skip (already verified in previous run)
    3. If not git repo → all core files (fallback)
    """
    if not is_git_repo(path):
        return all_core_files  # Fallback: verify all

    changed = get_changed_files(path).unwrap_or(set())
    if not changed:
        return []  # Nothing changed, skip proving

    # Only verify changed files that are in core/
    return [f for f in all_core_files if f in changed]
```

**Agent Experience:**
```
$ invar guard --prove
Verification: --prove (2 changed, 38 unchanged)
✓ CrossHair verified: 2 files in 8.2s
```

**Why this works:**
- If file unchanged → previous verification still valid
- Contract changes → file is "changed" by git
- Implementation changes → file is "changed" by git

---

### Priority 2: Parallel Execution (4x speedup)

**Principle:** Use all CPU cores. No configuration needed.

```python
from concurrent.futures import ProcessPoolExecutor, as_completed
import os

def run_crosshair_parallel(
    files: list[Path],
    timeout_per_file: int = 30,
) -> dict[Path, Result]:
    """
    Run CrossHair on multiple files concurrently.

    Agent-Native: Automatically uses optimal worker count.
    """
    max_workers = min(len(files), os.cpu_count() or 4)

    with ProcessPoolExecutor(max_workers=max_workers) as executor:
        futures = {
            executor.submit(_verify_single_file, f, timeout_per_file): f
            for f in files
        }

        results = {}
        for future in as_completed(futures):
            file_path = futures[future]
            try:
                results[file_path] = future.result()
            except Exception as e:
                results[file_path] = Failure(str(e))

        return results

def _verify_single_file(file: Path, timeout: int) -> Result[dict, str]:
    """Verify a single file with CrossHair."""
    cmd = [
        sys.executable, "-m", "crosshair", "check",
        str(file),
        "--max_uninteresting_iterations=5",  # Fast mode
        "--analysis_kind=deal",
    ]
    # ... subprocess execution
```

**Scaling:**
| Files | Workers | Sequential | Parallel | Speedup |
|-------|---------|------------|----------|---------|
| 4 | 4 | 16s | 4s | 4x |
| 8 | 4 | 32s | 8s | 4x |
| 17 | 4 | 68s | 17s | 4x |

---

### Priority 3: Fast Mode with Iterations (2x speedup)

**Principle:** Stop when done exploring, not after fixed time.

#### The Problem with `--per_condition_timeout`

```
Simple function (3 execution paths):
├── Path 1 ─────── 0.1s
├── Path 2 ─────── 0.2s
├── Path 3 ─────── 0.1s
└── [Waiting...] ─────────────────── 9.6s WASTED
    Total: 10s (fixed)

Complex function (100 paths):
├── Paths 1-50 ──────────── 5s
├── Paths 51-80 ─────────── 4s
└── [Timeout!] ──────────── 1s INCOMPLETE
    Total: 10s (may miss bugs!)
```

#### The Solution: `--max_uninteresting_iterations`

```
Simple function:
├── Iter 1: Path 1 ─── 0.1s (interesting: new!)
├── Iter 2: Path 2 ─── 0.2s (interesting: new!)
├── Iter 3: Path 3 ─── 0.1s (interesting: new!)
├── Iter 4: (repeat) ── 0.05s (uninteresting: count=1)
├── ...
└── Iter 8: (repeat) ── 0.05s (count=5 → STOP)
    Total: ~0.7s ✓

Complex function:
├── Iters 1-50: New paths ─────────── 15s
├── Iters 51-80: Slower but new ──── 20s
├── Iters 81-100: Starting repeats ── 5s
└── Iters 101-105: 5 consecutive repeats → STOP
    Total: ~40s (COMPLETE exploration)
```

**Key insight from CrossHair docs:**
> "This option can be more useful than --per_condition_timeout because the amount of time invested will scale with the complexity of the code under analysis."

**Benchmark (3 simple functions):**
| Strategy | Time | Result |
|----------|------|--------|
| `--per_condition_timeout=10` | 10.18s | Complete |
| `--max_uninteresting_iterations=5` | 3.97s | Complete |
| **Speedup** | **2.5x** | Same coverage |

---

### Priority 4: Verification Caching (∞x for unchanged)

**Principle:** Never re-verify unchanged code.

```
.invar/
├── cache/
│   └── prove/
│       ├── manifest.json           # Cache metadata
│       ├── a1b2c3_parser.json      # file_hash.json
│       └── d4e5f6_rules.json
```

**Cache entry format:**
```json
{
  "file_path": "src/invar/core/parser.py",
  "file_hash": "sha256:a1b2c3...",
  "verified_at": "2025-12-21T22:00:00Z",
  "crosshair_version": "0.0.98",
  "invar_version": "0.5.0",
  "result": "verified",
  "functions_checked": 10,
  "time_taken_ms": 4200
}
```

**Cache invalidation rules:**
1. File content changes (hash mismatch)
2. CrossHair version upgrade
3. Invar version upgrade
4. Manual `--no-cache` flag (escape hatch)

**Implementation:**
```python
def should_verify(file: Path, cache: ProveCache) -> bool:
    """Check if file needs verification."""
    current_hash = hash_file(file)
    cached = cache.get(file)

    if cached is None:
        return True  # Never verified

    if cached.file_hash != current_hash:
        return True  # File changed

    if cached.crosshair_version != get_crosshair_version():
        return True  # CrossHair upgraded

    return False  # Cache hit!
```

---

### Deprioritized: Skip Contract-less Files

**Analysis showed this optimization provides minimal benefit for well-structured projects:**

| Project Type | Contract-less Files | Benefit |
|--------------|---------------------|---------|
| Invar (high coverage) | 1/18 (5.5%) | 0.1s saved |
| Typical project (medium) | 10/40 (25%) | 1s saved |
| Legacy project (low) | 30/40 (75%) | 3s saved |

**Why deprioritized:**
1. Invar's `missing_contract = ERROR` ensures high coverage
2. Files without functions (e.g., `__init__.py`) are rare
3. Other optimizations provide 10-40x more benefit

**Still implemented** as defense-in-depth, but not a priority.

---

## Implementation Plan

| Phase | Feature | Files | Effort | Speedup |
|-------|---------|-------|--------|---------|
| 1 | Fast mode (iterations) | prove.py | 0.5h | 2x |
| 2 | Automatic incremental | cli.py, prove.py | 2h | 8.5x |
| 3 | Parallel execution | prove.py | 2h | 4x |
| 4 | Verification caching | prove.py, cache.py (new) | 4h | ∞x (cache hit) |
| 5 | Skip contract-less | prove.py | 0.5h | 1.01x |

**Total effort:** ~9 hours

---

## Expected Performance

### Before (Current)

```
$ time invar guard --prove
Verification: --prove (static + doctests + CrossHair)
...
✓ Verified: 40 files

real    6m44.123s
```

### After (DX-13)

```
$ time invar guard --prove
Verification: --prove (2 changed, 38 cached)
  ├── Parallel: 4 workers
  └── Mode: fast (iterations)
✓ CrossHair verified: 2 files in 8.2s

real    0m12.456s
```

### Performance Matrix

| Scenario | Current | DX-13 | Speedup |
|----------|---------|-------|---------|
| Full verify (40 files) | 6m 44s | 17s | 24x |
| Incremental (2 changed) | 6m 44s | 8s | 50x |
| Re-run (no changes) | 6m 44s | 0.5s | 800x |
| CI/pre-commit | 6m 44s | 8s | 50x |

---

## Agent-Native Design

### Zero Configuration

All optimizations are **automatic**. The Agent uses the same command:

```bash
invar guard --prove
```

The system decides:
- Which files to verify (changed only)
- How many workers (CPU count)
- Timeout strategy (iterations)
- Cache hits/misses

### Transparent Reporting

Agent sees what's happening:

```
Verification: --prove
  Files: 2 changed, 38 cached
  Workers: 4 parallel
  Mode: fast (max 5 uninteresting iterations)
✓ Verified in 8.2s
```

### Escape Hatches

For edge cases, explicit flags exist:

```bash
invar guard --prove --no-cache      # Force re-verification
invar guard --prove --all-files     # Verify everything
invar guard --prove --sequential    # Disable parallelism
```

But these are **never needed** in normal workflow.

---

## Technical Details

### CrossHair Integration

```python
# Current (slow)
cmd = ["crosshair", "check", file, "--per_condition_timeout=10"]

# DX-13 (fast)
cmd = [
    "crosshair", "check", file,
    "--max_uninteresting_iterations=5",  # Adaptive timeout
    "--analysis_kind=deal",               # Only deal contracts
]
```

### Cache Directory Structure

```
.invar/
├── cache/
│   └── prove/
│       ├── manifest.json
│       │   {
│       │     "version": "1.0",
│       │     "created": "2025-12-21",
│       │     "crosshair_version": "0.0.98",
│       │     "entries": 17
│       │   }
│       ├── a1b2c3d4.json  # SHA256 prefix
│       └── ...
└── ...
```

### Git Integration

```python
def get_changed_core_files(path: Path) -> list[Path]:
    """Get changed files that are in core/ directory."""
    changed = get_changed_files(path).unwrap_or(set())
    core_path = path / "src" / "invar" / "core"
    return [f for f in changed if core_path in f.parents]
```

---

## Risks and Mitigations

| Risk | Mitigation |
|------|------------|
| Cache corruption | SHA256 hash validation, auto-rebuild on mismatch |
| Parallel race conditions | Each file verified independently |
| Git not available | Fallback to full verification |
| CrossHair version mismatch | Cache includes version, auto-invalidate |

---

## Success Criteria

1. **Performance:** `--prove` completes in <30s for typical changes
2. **Correctness:** No false positives (cache hit when should re-verify)
3. **Agent-Native:** Zero new flags needed for normal workflow
4. **Transparency:** Agent understands what was verified vs cached

---

## References

- [CrossHair GitHub](https://github.com/pschanely/CrossHair)
- [CrossHair --max_uninteresting_iterations](https://crosshair.readthedocs.io/en/latest/introduction.html)
- DX-12: Hypothesis as CrossHair Fallback
- Invar Protocol v3.23: Three-Level Verification
