import hashlib
import hmac


class IPayDataV2:
    """
    This class is used to generate the URL that will be used to redirect the user to the iPay payment page.
    """
    def __init__(self, live, oid, inv, ttl, tel, eml, vid, curr, p1, p2, p3, p4, cbk, cst, crl, hsh,secret_key):
        """
        This is the constructor method for the IPay2Data class.
        Args:
            live (int): The live value to be used for the request. 0 for sandbox and 1 for live.
            oid (str): The oid value to be used for the request. This is the order id. It should be unique for each. You can use BaseUtils.generate_random_alphanumeric_string() to generate this.
            inv (str): The inv value to be used for the request. This is the invoice number. It should be unique for each. You can use BaseUtils.generate_random_alphanumeric_string() to generate this.
            ttl (int): The ttl value to be used for the request. This is the total amount to be paid.
            tel (str): The tel value to be used for the request. This is the phone number of the customer.
            eml (str): The eml value to be used for the request. This is the email of the customer.
            vid (str): The vid value to be used for the request. This is the vendor id. It is provided by iPay.
            curr (str): The curr value to be used for the request. This is the currency to be used for the transaction. It should be KES if in Kenya. It is provided by iPay.
            p1 (str): The p1 value to be used for the request. This is the first parameter. It is optional.
            p2 (str): The p2 value to be used for the request. This is the second parameter. It is optional.
            p3 (str): The p3 value to be used for the request. This is the third parameter. It is optional.
            p4 (str): The p4 value to be used for the request. This is the fourth parameter. It is optional.
            cbk (str): The cbk value to be used for the request. This is the callback url. It is the url that iPay will send the response to after the transaction is complete.
            cst (str): The cst value to be used for the request. This is the customer value. It is optional.
            crl (str): The crl value to be used for the request. This is the cancel url. It is the url that iPay will send the response to if the transaction is cancelled.
            hsh (str): The hsh value to be used for the request. This is the hash value. It is generated by hashing the other values using the secret key.
        """
        self.live = live
        self.oid = oid
        self.inv = inv
        self.ttl = ttl
        self.tel = tel
        self.eml = eml
        self.vid = vid
        self.curr = curr
        self.p1 = p1
        self.p2 = p2
        self.p3 = p3
        self.p4 = p4
        self.cbk = cbk
        self.cst = cst
        self.crl = crl
        self.hsh = hsh
        self.secret_key = secret_key

    def to_map(self):
        """
        This method is used to convert the IPay2Data object to a map.
        Returns:
            dict: The IPay2Data object as a map.
        """
        return {
            "live": self.live,
            "oid": self.oid,
            "inv": self.inv,
            "ttl": self.ttl,
            "tel": self.tel,
            "eml": self.eml,
            "vid": self.vid,
            "curr": self.curr,
            "p1": self.p1,
            "p2": self.p2,
            "p3": self.p3,
            "p4": self.p4,
            "cbk": self.cbk,
            "cst": self.cst,
            "crl": self.crl,
            "hsh": self.hsh
        }

    def attach_hash(self, i_pay_secret):
        """
        This method is used to attach the hash value to the IPay2Data object.
        Args:
            i_pay_secret (str): The secret key provided by iPay.
        Returns:
            IPay2Data: The IPay2Data object with the hash value attached.
        """
        hash_string = ''.join(str(value) for key, value in self.to_map().items() if key != "hsh").encode('utf-8')
        secret_key = i_pay_secret.encode('utf-8')
        hmac_value = hmac.new(secret_key, hash_string, hashlib.sha1).hexdigest()
        self.hsh = hmac_value
        return self

    def attach_more_fields(self):
        """
        This method is used to attach more fields to the IPay2Data object.
        Returns:
            dict: The IPay2Data object with the additional fields attached.
        """
        additional_fields = {
            "mpesa": "1",
            "airtel": "1",
            "equity": "1",
            "mobilebanking": "1",
            "creditcard": "1",
            "unionpay": "1",
            "mvisa": "1",
            "vooma": "1",
            "pesalink": "1",
            "autopay": "0"
        }
        additional_fields.update(self.attach_hash(self.secret_key).to_map())
        return additional_fields

    def generate_url(self):
        """
        This method is used to generate the url that will be used to redirect the user to the iPay payment page.
        Returns:
            str: The url that will be used to redirect the user to the iPay payment page.
        """
        url = "https://payments.ipayafrica.com/v3/ke?"
        url += '&'.join(f'{key}={value}' for key, value in self.attach_more_fields().items())
        return url


if __name__ == '__main__':
    print(IPayDataV2(
        live=0,
        oid="123456789",
        inv="123456789",
        ttl=100,
        tel="254712345678",
        eml="name@email.com",
        vid="demo",
        curr="KES",
        p1="",
        p2="",
        p3="",
        p4="",
        cbk="http://localhost:8000/callback",
        cst="1",
        crl="0",
        secret_key="demoCHANGED",
        hsh=""
    ).generate_url())