# noqa: D100
from __future__ import annotations

import itertools
import json
from typing import TYPE_CHECKING
from typing import Any
from typing import AsyncIterable
from typing import Iterable
from typing import List
from typing import Mapping
from typing import Optional
from typing import Sequence
from typing import TextIO
from typing import Tuple
from typing import TypeVar
from typing import Union

from .match import FilterContextVars
from .match import JSONPathMatch

if TYPE_CHECKING:
    from .env import JSONPathEnvironment
    from .selectors import JSONPathSelector


class JSONPath:
    """A compiled JSONPath ready to be applied to a JSON string or Python object.

    Arguments:
        env: The `JSONPathEnvironment` this path is bound to.
        selectors: An iterable of `JSONPathSelector` objects, as generated by
            a `Parser`.
    """

    __slots__ = ("env", "_selectors")

    def __init__(
        self,
        *,
        env: JSONPathEnvironment,
        selectors: Iterable[JSONPathSelector],
    ) -> None:
        self.env = env
        self._selectors = tuple(selectors)

    def __str__(self) -> str:
        return self.env.root_token + "".join(
            str(selector) for selector in self._selectors
        )

    def findall(
        self,
        data: Union[str, TextIO, Sequence[Any], Mapping[str, Any]],
        *,
        filter_context: Optional[FilterContextVars] = None,
    ) -> List[object]:
        """Find all objects in `data` matching the given JSONPath `path`.

        If `data` is a string or a file-like objects, it will be loaded
        using `json.loads()` and the default `JSONDecoder`.

        Arguments:
            data: A JSON document or Python object implementing the `Sequence`
                or `Mapping` interfaces.
            filter_context: Arbitrary data made available to filters using
                the _filter context_ selector.

        Returns:
            A list of matched objects. If there are no matches, the list will
            be empty.

        Raises:
            JSONPathSyntaxError: If the path is invalid.
            JSONPathTypeError: If a filter expression attempts to use types in
                an incompatible way.
        """
        if isinstance(data, str):
            _data = json.loads(data)
        elif isinstance(data, TextIO):
            _data = json.loads(data.read())
        else:
            _data = data
        return [
            match.obj for match in self.finditer(_data, filter_context=filter_context)
        ]

    def finditer(
        self,
        data: Union[str, TextIO, Sequence[Any], Mapping[str, Any]],
        *,
        filter_context: Optional[FilterContextVars] = None,
    ) -> Iterable[JSONPathMatch]:
        """Generate `JSONPathMatch` objects for each match.

        If `data` is a string or a file-like objects, it will be loaded
        using `json.loads()` and the default `JSONDecoder`.

        Arguments:
            data: A JSON document or Python object implementing the `Sequence`
                or `Mapping` interfaces.
            filter_context: Arbitrary data made available to filters using
                the _filter context_ selector.

        Returns:
            An iterator yielding `JSONPathMatch` objects for each match.

        JSONPathSyntaxError: If the path is invalid.
            JSONPathTypeError: If a filter expression attempts to use types in
                an incompatible way.
        """
        if isinstance(data, str):
            _data = json.loads(data)
        elif isinstance(data, TextIO):
            _data = json.loads(data.read())
        else:
            _data = data

        matches: Iterable[JSONPathMatch] = [
            JSONPathMatch(
                filter_context=filter_context or {},
                obj=_data,
                parent=None,
                path=self.env.root_token,
                parts=(),
                root=_data,
            )
        ]

        for selector in self._selectors:
            matches = selector.resolve(matches)

        return matches

    async def findall_async(
        self,
        data: Union[str, TextIO, Sequence[Any], Mapping[str, Any]],
        *,
        filter_context: Optional[FilterContextVars] = None,
    ) -> List[object]:
        """An async version of `findall()`."""
        if isinstance(data, str):
            _data = json.loads(data)
        elif isinstance(data, TextIO):
            _data = json.loads(data.read())
        else:
            _data = data
        return [
            match.obj
            async for match in await self.finditer_async(
                _data, filter_context=filter_context
            )
        ]

    async def finditer_async(
        self,
        data: Union[str, TextIO, Sequence[Any], Mapping[str, Any]],
        *,
        filter_context: Optional[FilterContextVars] = None,
    ) -> AsyncIterable[JSONPathMatch]:
        """An async version of `finditer()`."""
        if isinstance(data, str):
            _data = json.loads(data)
        elif isinstance(data, TextIO):
            _data = json.loads(data.read())
        else:
            _data = data

        async def root_iter() -> AsyncIterable[JSONPathMatch]:
            yield JSONPathMatch(
                filter_context=filter_context or {},
                obj=_data,
                parent=None,
                path=self.env.root_token,
                parts=(),
                root=_data,
            )

        matches: AsyncIterable[JSONPathMatch] = root_iter()

        for selector in self._selectors:
            matches = selector.resolve_async(matches)

        return matches

    def empty(self) -> bool:
        """Return `True` if this path has no selectors."""
        return bool(self._selectors)


class CompoundJSONPath:
    """Multiple `JSONPath`s combined."""

    __slots__ = ("env", "path", "paths")

    def __init__(
        self,
        *,
        env: JSONPathEnvironment,
        path: Union[JSONPath, CompoundJSONPath],
    ) -> None:
        self.env = env
        self.path = path
        self.paths: List[Tuple[(str, JSONPath)]] = []

    def __str__(self) -> str:
        buf: List[str] = [str(self.path)]
        for op, path in self.paths:
            buf.append(f" {op} ")
            buf.append(str(path))
        return "".join(buf)

    def findall(
        self,
        data: Union[str, TextIO, Sequence[Any], Mapping[str, Any]],
        filter_context: Optional[FilterContextVars] = None,
    ) -> List[object]:
        """Find all objects in `data` matching the given JSONPath `path`.

        If `data` is a string or a file-like objects, it will be loaded
        using `json.loads()` and the default `JSONDecoder`.

        Arguments:
            data: A JSON document or Python object implementing the `Sequence`
                or `Mapping` interfaces.
            filter_context: Arbitrary data made available to filters using
                the _filter context_ selector.

        Returns:
            A list of matched objects. If there are no matches, the list will
                be empty.

        Raises:
            JSONPathSyntaxError: If the path is invalid.
            JSONPathTypeError: If a filter expression attempts to use types in
                an incompatible way.
        """
        objs = self.path.findall(data, filter_context=filter_context)

        for op, path in self.paths:
            _objs = path.findall(data, filter_context=filter_context)
            if op == self.env.union_token:
                objs.extend(_objs)
            else:
                assert op == self.env.intersection_token, op
                objs = [obj for obj in objs if obj in _objs]

        return objs

    def finditer(
        self,
        data: Union[str, TextIO, Sequence[Any], Mapping[str, Any]],
        filter_context: Optional[FilterContextVars] = None,
    ) -> Iterable[JSONPathMatch]:
        """Generate `JSONPathMatch` objects for each match.

        If `data` is a string or a file-like objects, it will be loaded
        using `json.loads()` and the default `JSONDecoder`.

        Arguments:
            data: A JSON document or Python object implementing the `Sequence`
                or `Mapping` interfaces.
            filter_context: Arbitrary data made available to filters using
                the _filter context_ selector.

        Returns:
            An iterator yielding `JSONPathMatch` objects for each match.

        Raises:
            JSONPathSyntaxError: If the path is invalid.
            JSONPathTypeError: If a filter expression attempts to use types in
                an incompatible way.
        """
        matches = self.path.finditer(data, filter_context=filter_context)

        for op, path in self.paths:
            _matches = path.finditer(data, filter_context=filter_context)
            if op == self.env.union_token:
                matches = itertools.chain(matches, _matches)
            else:
                assert op == self.env.intersection_token
                _objs = [match.obj for match in _matches]
                matches = (match for match in matches if match.obj in _objs)

        return matches

    async def findall_async(
        self,
        data: Union[str, TextIO, Sequence[Any], Mapping[str, Any]],
        filter_context: Optional[FilterContextVars] = None,
    ) -> List[object]:
        """An async version of `findall()`."""
        objs = await self.path.findall_async(data, filter_context=filter_context)

        for op, path in self.paths:
            _objs = await path.findall_async(data, filter_context=filter_context)
            if op == self.env.union_token:
                objs.extend(_objs)
            else:
                assert op == self.env.intersection_token
                objs = [obj for obj in objs if obj in _objs]

        return objs

    async def finditer_async(
        self,
        data: Union[str, TextIO, Sequence[Any], Mapping[str, Any]],
        filter_context: Optional[FilterContextVars] = None,
    ) -> AsyncIterable[JSONPathMatch]:
        """An async version of `finditer()`."""
        matches = await self.path.finditer_async(data, filter_context=filter_context)

        for op, path in self.paths:
            _matches = await path.finditer_async(data, filter_context=filter_context)
            if op == self.env.union_token:
                matches = _achain(matches, _matches)
            else:
                assert op == self.env.intersection_token
                _objs = [match.obj async for match in _matches]
                matches = (match async for match in matches if match.obj in _objs)

        return matches

    def union(self, path: JSONPath) -> CompoundJSONPath:
        """In-place union of this path and another path."""
        self.paths.append((self.env.union_token, path))
        return self

    def intersection(self, path: JSONPath) -> CompoundJSONPath:
        """In-place intersection of this path and another path."""
        self.paths.append((self.env.intersection_token, path))
        return self


T = TypeVar("T")


async def _achain(*iterables: AsyncIterable[T]) -> AsyncIterable[T]:
    for it in iterables:
        async for element in it:
            yield element
