# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['kasa', 'kasa.tests']

package_data = \
{'': ['*'], 'kasa.tests': ['fixtures/*']}

install_requires = \
['asyncclick>=7,<8', 'importlib-metadata']

entry_points = \
{'console_scripts': ['kasa = kasa.cli:cli']}

setup_kwargs = {
    'name': 'python-kasa',
    'version': '0.4.0.dev0',
    'description': 'Python API for TP-Link Kasa Smarthome devices',
    'long_description': '# python-kasa\n\n[![PyPI version](https://badge.fury.io/py/python-kasa.svg)](https://badge.fury.io/py/python-kasa)\n[![Build Status](https://dev.azure.com/python-kasa/python-kasa/_apis/build/status/python-kasa.python-kasa?branchName=master)](https://dev.azure.com/python-kasa/python-kasa/_build/latest?definitionId=2&branchName=master)\n[![Coverage Status](https://coveralls.io/repos/github/python-kasa/python-kasa/badge.svg?branch=master)](https://coveralls.io/github/python-kasa/python-kasa?branch=master)\n\npython-kasa is a Python library to control TPLink smart home devices (plugs, wall switches, power strips, and bulbs) using asyncio.\nThis project is a maintainer-made fork of [pyHS100](https://github.com/GadgetReactor/pyHS100) project.\n\n\n**Supported devices**\n\n* Plugs\n  * HS100\n  * HS103\n  * HS105\n  * HS107\n  * HS110\n* Power Strips\n  * HS300\n  * KP303\n* Wall switches\n  * HS200\n  * HS210\n  * HS220\n* Bulbs\n  * LB100\n  * LB110\n  * LB120\n  * LB130\n  * LB230\n  * KL60\n  * KL110\n  * KL120\n  * KL130\n\n**Contributions (be it adding missing features, fixing bugs or improving documentation) are more than welcome, feel free to submit pull requests! See below for instructions for setting up a development environment.**\n\n\n# Usage\n\nThe package is shipped with a console tool named kasa, please refer to ```kasa --help``` for detailed usage.\nThe device to which the commands are sent is chosen by `KASA_HOST` environment variable or passing `--host <address>` as an option.\nTo see what is being sent to and received from the device, specify option `--debug`.\n\nTo avoid discovering the devices when executing commands its type can be passed by specifying either `--plug` or `--bulb`,\nif no type is given its type will be discovered automatically with a small delay.\nSome commands (such as reading energy meter values and setting color of bulbs) additional parameters are required,\nwhich you can find by adding `--help` after the command, e.g. `kasa emeter --help` or `kasa hsv --help`.\n\nIf no command is given, the `state` command will be executed to query the device state.\n\n## Initial Setup\n\nYou can provision your device without any extra apps by using the `kasa wifi` command:\n1. If the device is unprovisioned, connect to its open network\n2. Use `kasa discover` (or check the routes) to locate the IP address of the device (likely 192.168.0.1)\n3. Scan for available networks using `kasa wifi scan`\n4. Join/change the network using `kasa wifi join` command, see `--help` for details.\n## Discovering devices\n\nThe devices can be discovered either by using `kasa discover` or by calling `kasa` without any parameters.\nIn both cases supported devices are discovered from the same broadcast domain, and their current state will be queried and printed out.\n\n```\n$ kasa\nNo --bulb nor --plug given, discovering..\nDiscovering devices for 3 seconds\n== My Smart Plug - HS110(EU) ==\nDevice state: ON\nIP address: 192.168.x.x\nLED state: False\nOn since: 2017-03-26 18:29:17.242219\n== Generic information ==\nTime:         1970-06-22 02:39:41\nHardware:     1.0\nSoftware:     1.0.8 Build 151101 Rel.24452\nMAC (rssi):   50:C7:BF:XX:XX:XX (-77)\nLocation:     {\'latitude\': XXXX, \'longitude\': XXXX}\n== Emeter ==\nCurrent state: {\'total\': 133.082, \'power\': 100.418681, \'current\': 0.510967, \'voltage\': 225.600477}\n```\n\n## Basic controls\n\nAll devices support a variety of common commands, including:\n * `state` which returns state information\n * `on` and `off` for turning the device on or off\n * `emeter` (where applicable) to return energy consumption information\n * `sysinfo` to return raw system information\n\n## Energy meter\n\nPassing no options to `emeter` command will return the current consumption.\nPossible options include `--year` and `--month` for retrieving historical state,\nand reseting the counters is done with `--erase`.\n\n```\n$ kasa emeter\n== Emeter ==\nCurrent state: {\'total\': 133.105, \'power\': 108.223577, \'current\': 0.54463, \'voltage\': 225.296283}\n```\n\n## Bulb-specific commands\n\nAt the moment setting brightness, color temperature and color (in HSV) are supported depending on the device.\nThe commands are straightforward, so feel free to check `--help` for instructions how to use them.\n\n# Library usage\n\nThe property accesses use the data obtained before by awaiting `update()`.\nThe values are cached until the next update call. In practice this means that property accesses do no I/O and are dependent, while I/O producing methods need to be awaited.\n\nMethods changing the state of the device do not invalidate the cache (i.e., there is no implicit `update()`).\nYou can assume that the operation has succeeded if no exception is raised.\nThese methods will return the device response, which can be useful for some use cases.\n\nErrors are raised as `SmartDeviceException` instances for the library user to handle.\n\n## Discovering devices\n\n`Discover.discover()` can be used to discover supported devices in the local network.\nThe return value is a dictionary keyed with the IP address and the value holds a ready-to-use instance of the detected device type.\n\nExample:\n```python\nimport asyncio\nfrom kasa import Discover\n\ndevices = asyncio.run(Discover.discover())\nfor addr, dev in devices.items():\n    asyncio.run(dev.update())\n    print(f"{addr} >> {dev}")\n```\n```\n$ python example.py\n<SmartPlug at 192.168.XXX.XXX (My Smart Plug), is_on: True - dev specific: {\'LED state\': True, \'On since\': datetime.datetime(2017, 3, 26, 18, 29, 17, 52073)}>\n```\n\n## Querying basic information\n\n```python\nimport asyncio\nfrom kasa import SmartPlug\nfrom pprint import pformat as pf\n\nplug = SmartPlug("192.168.XXX.XXX")\nasyncio.run(plug.update())\nprint("Hardware: %s" % pf(plug.hw_info))\nprint("Full sysinfo: %s" % pf(plug.sys_info))\n```\n\nThe rest of the examples assume that you have initialized an instance.\n\n## State & switching\n\nDevices can be turned on and off by either calling appropriate methods on the device object.\n\n```python\nprint("Current state: %s" % plug.is_on)\nawait plug.turn_off()\nawait plug.turn_on()\n```\n\n## Getting emeter status (if applicable)\nThe `update()` call will automatically fetch the following emeter information:\n* Current consumption (accessed through `emeter_realtime` property)\n* Today\'s consumption (`emeter_today`)\n* This month\'s consumption (`emeter_this_month`)\n\nYou can also request this information separately:\n\n```python\nprint("Current consumption: %s" % await plug.get_emeter_realtime())\nprint("Per day: %s" % await plug.get_emeter_daily(year=2016, month=12))\nprint("Per month: %s" % await plug.get_emeter_monthly(year=2016))\n```\n\n## Bulb and dimmer-specific APIs\n\nThe bulb API is likewise straightforward, so please refer to its API documentation.\nInformation about supported features can be queried by using properties prefixed with `is_`, e.g. `is_dimmable`.\n\n### Setting the brightness\n\n```python\nimport asyncio\nfrom kasa import SmartBulb\n\nbulb = SmartBulb("192.168.1.123")\nasyncio.run(bulb.update())\n\nif bulb.is_dimmable:\n    asyncio.run(bulb.set_brightness(100))\n    asyncio.run(bulb.update())\n    print(bulb.brightness)\n```\n\n### Setting the color temperature\n```python\nif bulb.is_variable_color_temp:\n    await bulb.set_color_temp(3000)\n    await bulb.update()\n    print(bulb.color_temp)\n```\n\n### Setting the color\n\nHue is given in degrees (0-360) and saturation and value in percentage.\n\n```python\nif bulb.is_color:\n    await bulb.set_hsv(180, 100, 100) # set to cyan\n    await bulb.update()\n    print(bulb.hsv)\n```\n\n## Contributing\n\nContributions are very welcome! To simplify the process, we are leveraging automated checks and tests for contributions.\n\n### Resources\n\n* [softScheck\'s github contains lot of information and wireshark dissector](https://github.com/softScheck/tplink-smartplug#wireshark-dissector)\n* [https://github.com/plasticrake/tplink-smarthome-simulator](tplink-smarthome-simulator)\n\n### Setting up development environment\n\n```bash\npoetry install\npre-commit install\n```\n\n### Code-style checks\n\nWe use several tools to automatically check all contributions, which are run automatically when you commit your code.\n\nIf you want to manually execute the checks, you can run `tox -e lint` to do the linting checks or `tox` to also execute the tests.\n',
    'author': 'Your Name',
    'author_email': 'you@example.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/python-kasa/python-kasa',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
