"""A collection of plot functions"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/02_plot.ipynb.

# %% auto 0
__all__ = ['set_sns', 'get_color_dict', 'logo_func', 'get_logo', 'get_logo2', 'plot_rank', 'plot_hist', 'plot_heatmap', 'plot_2d',
           'plot_cluster', 'plot_bokeh', 'plot_count', 'plot_bar', 'plot_group_bar', 'plot_box', 'plot_corr',
           'draw_corr', 'get_AUCDF', 'plot_confusion_matrix']

# %% ../nbs/02_plot.ipynb 4
import joblib,logomaker
import fastcore.all as fc, pandas as pd, numpy as np, seaborn as sns
from adjustText import adjust_text
from pathlib import Path

from scipy.stats import spearmanr, pearsonr
from sklearn.metrics import confusion_matrix
from matplotlib import pyplot as plt
from matplotlib.ticker import MultipleLocator
from numpy import trapz


# Katlas
from .feature import *
from .core import *

# Bokeh
from bokeh.io import output_notebook, show
from bokeh.plotting import figure, ColumnDataSource
from bokeh.models import HoverTool, AutocompleteInput, CustomJS
from bokeh.layouts import column
from bokeh.palettes import Category20_20
from itertools import cycle

# %% ../nbs/02_plot.ipynb 6
def set_sns():
    "Set seaborn resolution for notebook display"
    sns.set(rc={"figure.dpi":300, 'savefig.dpi':300})
    sns.set_context('notebook')
    sns.set_style("ticks")

# %% ../nbs/02_plot.ipynb 7
def get_color_dict(categories, # list of names to assign color
                   palette: str='tab20', # choose from sns.color_palette
                   ):
    "Assign colors to a list of names (allow duplicates), returns a dictionary of unique name with corresponding color"
    p=sns.color_palette(palette)
    color_cycle = cycle(p)
    color_map = {category: next(color_cycle) for category in categories}
    return color_map

# %% ../nbs/02_plot.ipynb 11
def logo_func(df:pd.DataFrame, # a dataframe that contains ratios for each amino acid at each position
              title: str='logo', # title of the motif logo
             ):
    "Use logomaker plot motif logos given a df matrix "
    
    # Indicates color scheme of the amino acid
    aa = {
        'AG': '#037f04',
        'DEsty': '#da143e', # sty seems to be the same color as big ST&Y even though we set it here
        'F': '#84380b',
        'HQN': '#8d2be1',
        'LMIFWTVC': '#d9a41c',
        'P': '#000000',
        'RK': '#0000ff',
        'ST': '#8d008d', # STY overwrites the previous s,t,y as logomaker does not distingusih capital and lower case
        'Y': '#84380b',
        
        # some old settings
        # 'st':'#8d2be1',
        # 'y':'#8d2be1'
        # 'pS/pT':'#8d2be1',# logomaker does not support double letters like pS or pT
        # 'pY':'#8d2be1'
    }
    
    # Use logomaker to plot
    logo = logomaker.Logo(df,color_scheme = aa,flip_below=False,figsize=(7,2.5)) #5.5,2.5
    
    logo.style_xticks(fmt='%d')
    logo.ax.set_yticks([])
    logo.ax.set_title(title)

# %% ../nbs/02_plot.ipynb 12
def get_logo(df: pd.DataFrame, # stacked Dataframe with kinase as index, substrates as columns
             kinase: str, # a specific kinase name in index
             ):
    "Given stacked df (index as kinase, columns as substrates), get a specific kinase's logo"
    
    
    # get raw kinase to calculate S/T
    pp = get_one_kinase(df,kinase,normalize=False)
    
    # get S/T ratio value
    ss = pp['S'].sum()
    st = pp['T'].sum()

    S_ctrl = 0.75*ss - 0.25*st
    T_ctrl = 0.75*st - 0.25*ss

    S0 = S_ctrl / max(S_ctrl, T_ctrl)
    T0 = T_ctrl / max(S_ctrl, T_ctrl)

    S_ratio = S0/(S0+T0)
    T_ratio = T0/(S0+T0)
    
    # get normalized kinase
    norm_p = get_one_kinase(df,kinase, normalize=True)
    
    # calculate ratio, divide values by median, followed by log2 transformation
    ratio =norm_p.apply(lambda r: r/r.median(),axis=1)
    ratio = np.log2(ratio)

    m = ratio.apply(lambda row: row[row > 0].sum(), axis=1).max()

    new_row = pd.DataFrame({'S': S_ratio*m, 'T':T_ratio*m}, index=[0]) 

    ratio2 = pd.concat([ratio, new_row], ignore_index=False).fillna(0)
    
    # plot logo
    logo_func(ratio2, kinase)

# %% ../nbs/02_plot.ipynb 16
def get_logo2(full: pd.DataFrame, # a dataframe that contains the full matrix of a kinase, with index as amino acid, and columns as positions
              title: str = 'logo', # title of the graph
              ):
    
    "Plot logo from a full freqency matrix of a kinase"
    
    # get S,T,Y ratio
    S_ratio,T_ratio,Y_ratio = full[0][['s','t','y']]/full[0][['s','t','y']].sum()
    
    # drop position 0 
    full = full.drop(columns=[0])
    
    # identify the minimum value other than 0
    min_val = full[full > 0].min().min()
    
    # replace 0s with the identified minimum value
    full = full.replace(0, min_val)
    
    norm_p = full.T

    # calculate ratio, use substraction
    ratio =norm_p.apply(lambda r: r-r.median(),axis=1)
    
    # calculate ratio based on previous method, divide values by median, followed by log2
    # ratio =norm_p.apply(lambda r: r/r.median(),axis=1)
    # ratio = np.log2(ratio)
    
    # get the max value for a position
    m = ratio.apply(lambda row: row[row > 0].sum(), axis=1).max()
    
    # get the relative height of S,T,Y relative to the max value
    new_row = pd.DataFrame({'S': S_ratio*m, 'T':T_ratio*m,'Y':Y_ratio*m}, index=[0]) 
    
    # prepare the matrix for logomaker
    ratio2 = pd.concat([ratio, new_row], ignore_index=False).fillna(0)

    logo_func(ratio2,title)

# %% ../nbs/02_plot.ipynb 19
@fc.delegates(sns.scatterplot)
def plot_rank(sorted_df: pd.DataFrame, # a sorted dataframe
              x: str, # column name for x axis
              y: str, # column name for y aixs
              n_hi: int=10, # if not None, show the head n names
              n_lo: int=10, # if not None, show the tail n names
              figsize: tuple=(10,8), # figure size
              **kwargs # arguments for sns.scatterplot()
              ):
    
    "Plot rank from a sorted dataframe"

    plt.figure(figsize=figsize)
    
    sorted_df = sorted_df.reset_index(drop=True) # drop customized index
    
    sns_plot = sns.scatterplot(data=sorted_df, 
                               x = x,
                               y = y, **kwargs)

    sns_plot.set_xticks([])
    
    
    texts = []
    
    # Annotate the highest n values
    if n_hi is not None:
        
        for i, row in sorted_df.head(n_hi).iterrows():
            texts.append(plt.text(i, row[y], row[x], ha='center', va='bottom'))

    if n_lo is not None:
        # Annotate the lowest n values
        n_lowest = n_lo
        for i, row in sorted_df.tail(n_lowest).iterrows():
            texts.append(plt.text(i, row[y], row[x], ha='center', va='bottom'))
            
    if len(texts)>0:
        # Use adjustText to adjust text positions
        adjust_text(texts, arrowprops=dict(arrowstyle='-', color='black'))

    plt.tight_layout()

# %% ../nbs/02_plot.ipynb 23
@fc.delegates(sns.histplot)
def plot_hist(df: pd.DataFrame, # a dataframe that contain values for plot
              x: str, # column name of values
              figsize: tuple=(6,2),
              **kwargs, # arguments for sns.histplot()
             ):
    
    hist_params = {'element':'poly',
              'edgecolor': None,
              'alpha':0.5,
              'bins':100,
              'kde':True}
    
    plt.figure(figsize=figsize)
    sns.histplot(data=df,x=x,**hist_params,**kwargs)

# %% ../nbs/02_plot.ipynb 27
@fc.delegates(sns.heatmap)
def plot_heatmap(matrix, # a matrix of values
                 title: str='heatmap', # title of the heatmap
                 figsize: tuple=(6,10), # figure size of the heatmap
                 cmap: str='binary', # color map, default is dark&white
                 **kwargs, # arguments for sns.heatmap()
                 ):
    
    "Plot heatmap based on a matrix of values"
    
    plt.figure(figsize=figsize)
    sns.heatmap(matrix, cmap=cmap, annot=False,**kwargs)
    plt.title(title)

# %% ../nbs/02_plot.ipynb 31
@fc.delegates(sns.scatterplot)
def plot_2d(X: pd.DataFrame, # a dataframe that has first column to be x, and second column to be y
            **kwargs, # arguments for sns.scatterplot
            ):
    "Make 2D plot from a dataframe that has first column to be x, and second column to be y"
    plt.figure(figsize=(7,7))
    sns.scatterplot(data = X,x=X.columns[0],y=X.columns[1],alpha=0.7,**kwargs)

# %% ../nbs/02_plot.ipynb 33
def plot_cluster(df: pd.DataFrame, # a dataframe of values that is waited for dimensionality reduction
                 method: str='pca', # dimensionality reduction method, choose from pca, umap, and tsne
                 hue: str=None, # colname of color
                 complexity: int=30, # recommend 30 for tsne, 15 for umap, none for pca
                 palette: str='tab20', # color scheme, could be tab10 if less categories
                 legend: bool=False, # whether or not add the legend on the side
                 name_list=None, # a list of names to annotate each dot in the plot
                 seed: int=123, # seed for dimensionality reduction
                 s: int=50, # size of the dot
                 **kwargs # arguments for dimensional reduction method to be used
                 ):
    
    "Given a dataframe of values, plot it in 2d, method could be pca, tsne, or umap"
    
    embedding_df = reduce_feature(df, method=method, seed=seed, complexity = complexity,**kwargs)
    # x_col, y_col = [col for col in embedding_df.columns if col.startswith(method.upper())]
    x_col, y_col = embedding_df.columns
    sns.relplot(data=embedding_df, x=x_col, y=y_col, hue=hue, palette=palette, s=s, alpha=0.8, legend=legend)
    plt.xticks([])
    plt.yticks([])
    if name_list is not None:
        texts = [plt.text(embedding_df[x_col][i], embedding_df[y_col][i], name_list[i],fontsize=8) for i in range(len(embedding_df))]
        adjust_text(texts, arrowprops=dict(arrowstyle='-', color='black'))

# %% ../nbs/02_plot.ipynb 37
def plot_bokeh(X:pd.DataFrame, # a dataframe of two columns from dimensionality reduction
               idx, # pd.Series or list that indicates identities for searching box
               hue:None, # pd.Series or list that indicates category for each sample
               s: int=3, # dot size
               **kwargs # key:args format for information to include in the dot information box
               ):
    
    "Make interactive 2D plot with a searching box and window of dot information when pointing "
        
    output_notebook()
    
    idx = list(idx)
    hue = list(hue)
    
    def assign_colors(categories, palette):
        "assign each unique name in a list with a color, returns a color list of same length"
        color_cycle = cycle(palette)
        color_map = {category: next(color_cycle) for category in categories}
        return [color_map[category] for category in categories]
    
    if hue is not None:
        colors  = assign_colors(hue, Category20_20) 
    else:
        colors = ['navy'] * len(X)
    
    data_dict={
    'x': X.iloc[:,0],
    'y': X.iloc[:,1],
    'identity': idx,
    'color': colors,
    'original_color': colors,
    'size': [s] * len(X), 
    'highlighted': ['no'] * len(X)  # To keep track of which dot is highlighted
    }
    
    for key, value in kwargs.items():
        data_dict[key] = value
    
    source = ColumnDataSource(data=data_dict)
    
    p = figure(tools="pan,box_zoom,wheel_zoom,reset")
    p.scatter('x', 'y', source=source, alpha=0.6, color='color', size='size')

    # Disable grid lines
    p.xgrid.visible = False
    p.ygrid.visible = False
    
    # Add hover tool
    hover = HoverTool()
    
    tooltips = [("Identity", "@identity")]

    for key in kwargs.keys():
        tooltips.append((key.capitalize(), f"@{key}"))

    
    hover.tooltips = tooltips
    p.add_tools(hover)
    
    
    autocomplete = AutocompleteInput(title="Search by Identity:", completions=idx)

    callback = CustomJS(args=dict(source=source, plot=p), code="""
        const data = source.data;
        const search_val = cb_obj.value.toLowerCase();
        const x = data['x'];
        const y = data['y'];
        const identity = data['identity'];
        const color = data['color'];
        const original_color = data['original_color'];
        const size = data['size'];
        const highlighted = data['highlighted'];

        for (let i = 0; i < identity.length; i++) {
            if (highlighted[i] === 'yes') {
                color[i] = original_color[i];
                size[i] = 10;
                highlighted[i] = 'no';
            }
            if (identity[i].toLowerCase() === search_val) {
                plot.x_range.start = x[i] - 5;
                plot.x_range.end = x[i] + 5;
                plot.y_range.start = y[i] - 5;
                plot.y_range.end = y[i] + 5;
                color[i] = 'red';
                size[i] = 15;
                highlighted[i] = 'yes';
            }
        }
        source.change.emit();
    """)
    autocomplete.js_on_change('value', callback)

    # Show layout
    layout = column(autocomplete, p)
    show(layout)

# %% ../nbs/02_plot.ipynb 40
def plot_count(cnt, # from df['x'].value_counts()
               tick_spacing: float= None, # tick spacing for x axis
               palette: str='tab20'):
    
    "Make bar plot from df['x'].value_counts()"
    
    c = sns.color_palette(palette)
    ax = cnt.plot.barh(color = c)

    for index, value in enumerate(cnt):
        plt.text(value, index, str(value),fontsize=10,rotation=-90, va='center')
        # Set x-ticks at regular intervals
    if tick_spacing is not None:
        ax.xaxis.set_major_locator(MultipleLocator(tick_spacing))

# %% ../nbs/02_plot.ipynb 42
@fc.delegates(sns.barplot)
def plot_bar(df, 
             value, # colname of value
             group, # colname of group
             title = None,
             figsize = (12,5),
             fontsize=14,
             dots = True, # whether or not add dots in the graph
             rotation=90,
             ascending=False,
             **kwargs
              ):
    
    "Plot bar graph from unstacked dataframe; need to indicate columns of values and categories"
    
    plt.figure(figsize=figsize)
    
    idx = df.groupby(group)[value].mean().sort_values(ascending=ascending).index
    
    sns.barplot(data=df, x=group, y=value, order=idx, **kwargs)
    
    if dots:
        marker = {'marker': 'o', 
                  'color': 'white', 
                  'edgecolor': 'black', 
                  'linewidth': 1.5, 
                  'jitter':True,
                  's': 5}

        sns.stripplot(data=df, 
                      x=group, 
                      y=value,
                      order=idx,
                      alpha=0.8,
                      # ax=g.ax,
                      **marker)
        
    # Increase font size for the x-axis and y-axis tick labels
    plt.tick_params(axis='x', labelsize=fontsize)  # Increase x-axis label size
    plt.tick_params(axis='y', labelsize=fontsize)  # Increase y-axis label size
    
    # Modify x and y label and increase font size
    plt.xlabel('', fontsize=fontsize)
    plt.ylabel(value, fontsize=fontsize)
    
    # Rotate X labels
    plt.xticks(rotation=rotation)
    
    # Plot titles
    if title is not None:
        plt.title(title,fontsize=fontsize)
    
    plt.gca().spines[['right', 'top']].set_visible(False)

# %% ../nbs/02_plot.ipynb 45
@fc.delegates(sns.barplot)
def plot_group_bar(df, 
                   value_cols,  # list of column names for values, the order depends on the first item
                   group,       # column name of group (e.g., 'kinase')
                   figsize=(12, 5),
                   order=None,
                   title=None,
                   fontsize=14,
                   rotation=90,
                   **kwargs):
    
    " Plot grouped bar graph from dataframe. "

    # Prepare the dataframe for plotting
    # Melt the dataframe to go from wide to long format
    df_melted = df.melt(id_vars=group, value_vars=value_cols, var_name='Ranking', value_name='Value')

    plt.figure(figsize=figsize)
    
    # Create the bar plot
    sns.barplot(data=df_melted, x=group, y='Value', hue='Ranking', order=order, 
                capsize=0.1,errwidth=1.5,errcolor='gray', # adjust the error bar settings
                **kwargs)
    
    # Increase font size for the x-axis and y-axis tick labels
    plt.tick_params(axis='x', labelsize=fontsize)  # Increase x-axis label size
    plt.tick_params(axis='y', labelsize=fontsize)  # Increase y-axis label size
    
    # Modify x and y label and increase font size
    plt.xlabel('', fontsize=fontsize)
    plt.ylabel('Value', fontsize=fontsize)
    
    # Rotate X labels
    plt.xticks(rotation=rotation)
    
    # Plot titles
    if title is not None:
        plt.title(title, fontsize=fontsize)
    
    plt.gca().spines[['right', 'top']].set_visible(False)
    plt.legend(fontsize=fontsize) # if change legend location, use loc='upper right'

# %% ../nbs/02_plot.ipynb 48
@fc.delegates(sns.boxplot)
def plot_box(df,
             value, # colname of value
             group, # colname of group
             title=None, 
             figsize=(6,3),
             fontsize=14,
             dots=True, 
             rotation=90,
             **kwargs
            ):
    
    "Plot box plot."
    
    plt.figure(figsize=figsize)
    
    idx = df[[group,value]].groupby(group).median().sort_values(value,ascending=False).index
    
    
    sns.boxplot(data=df, x=group, y=value, order=idx, **kwargs)
    
    if dots:
        sns.stripplot(x=group, y=value, data=df, order=idx, jitter=True, color='black', size=3)
        

    # Increase font size for the x-axis and y-axis tick labels
    plt.tick_params(axis='x', labelsize=fontsize)  # Increase x-axis label size
    plt.tick_params(axis='y', labelsize=fontsize)  # Increase y-axis label size

    plt.xlabel('', fontsize=fontsize)
    plt.ylabel(value, fontsize=fontsize)

    plt.xticks(rotation=rotation)
    
    if title is not None:
        plt.title(title,fontsize=fontsize)
    
    # Remove right and top spines 
    # plt.gca().spines[['right', 'top']].set_visible(False)
    

# %% ../nbs/02_plot.ipynb 51
@fc.delegates(sns.regplot)
def plot_corr(x, # x axis values, or colname of x axis
              y, # y axis values, or colname of y axis
              xlabel=None,# x axis label
              ylabel=None,# y axis label
              data = None, # dataframe that contains data
              text_location = [0.8,0.1],
              **kwargs
              ):
    "Given a dataframe and the name of two columns, plot the two columns' correlation"
    if data is not None:
        x=data[x]
        y=data[y]
        
    pear, pvalue = pearsonr(x, y)
        
    sns.regplot(
                x=x,
                y=y,
                line_kws={'color': 'gray'}, **kwargs
           )
    
    if xlabel is not None:
        plt.xlabel(xlabel)
        
    if ylabel is not None:
        plt.ylabel(ylabel)
    
    # correlation_text = f'Spearman: {correlation:.2f}' if method == 'spearman' else f'Pearson: {correlation:.2f}'

    # plt.text(x=0.8, y=0.1, s=correlation_text, transform=plt.gca().transAxes, ha='center', va='center')
    plt.text(s=f'Pearson = {round(pear,2)}\n   p = {"{:.2e}".format(pvalue)}',
             x=text_location[0],y=text_location[1],
            transform=plt.gca().transAxes, 
             ha='center', va='center')

# %% ../nbs/02_plot.ipynb 55
def draw_corr(corr):
    
    "plot heatmap from df.corr()"
    
    # Mask for the upper triangle
    mask = np.triu(np.ones_like(corr, dtype=bool))
    
    # Plotting the heatmap
    plt.figure(figsize=(20, 16))  # Set the figure size
    sns.heatmap(corr, annot=True, cmap='coolwarm', vmin=-1, vmax=1, mask=mask, fmt='.2f')

# %% ../nbs/02_plot.ipynb 59
def get_AUCDF(df,col, reverse=False,plot=True,xlabel='Rank of reported kinase'):
    
    "Plot CDF curve and get relative area under the curve"
    
    # sort col values as x values
    x_values = df[col].sort_values().values
    
    # get y_values evenly distributed from 0 to 1
    # y_values = np.arange(1, len(x_values) + 1) / len(x_values) # this method assumes equal distribution of each x value
    y_values = pd.Series(x_values).rank(method='average', pct=True).values # this method takes duplicates into account
    
    if reverse:
        y_values = 1 - y_values + y_values.min()  # Adjust for reverse while keeping the distribution's integrity
    # calculate the area under the curve using the trapezoidal rule
    area_under_curve = trapz(y_values, x_values)
    
    # calculate total area
    total_area = (x_values[-1] - x_values[0]) * (y_values[-1] - y_values[0])
    

    AUCDF = area_under_curve / total_area
    if reverse:
        AUCDF = -AUCDF
    
    if plot:
        # Create a figure and a primary axis
        fig, ax1 = plt.subplots(figsize=(7,5))
        
        # fontsize
        fontsize=17
        
        # Plot the histogram on the primary axis
        sns.histplot(x_values,bins=20,ax=ax1)
        ax1.set_xlabel(xlabel,fontsize=fontsize)
        ax1.set_ylabel('Substrates',color='darkblue',fontsize=fontsize)
        ax1.tick_params(axis='y', labelcolor='darkblue',labelsize=fontsize)
        ax1.tick_params(axis='x', labelcolor='black',labelsize=fontsize)
        ax1.set_xlim(min(x_values),max(x_values))

        # Create a secondary axis for the CDF
        ax2 = ax1.twinx()

        # Plot the CDF on the secondary axis
        # ax2.plot(bin_edges[:-1], cumulative_data, color='red', linestyle='-', linewidth=2.0)
        ax2.plot(x_values, y_values, color='darkred', linestyle='-', linewidth=2.0)
        if reverse:
            ax2.plot([max(x_values),0],[0, max(y_values)], 'k--')  # 'k--' is for a black dashed line
        else:
            ax2.plot([0, max(x_values)], [0, max(y_values)], 'k--')  # 'k--' is for a black dashed line

        ax2.set_ylabel('Probability', color='darkred',fontsize=fontsize,rotation=270,labelpad=18)
        if reverse:
            ax2.text(0.45, 0.3, f"AUCDF:{AUCDF.round(4)}", transform=plt.gca().transAxes, ha='right', va='bottom',fontsize=fontsize)
        else:
            ax2.text(0.95, 0.3, f"AUCDF:{AUCDF.round(4)}", transform=plt.gca().transAxes, ha='right', va='bottom',fontsize=fontsize)
        ax2.tick_params(axis='y', labelcolor='darkred',labelsize=fontsize)
        ax2.set_ylim(0, 1)  # Probabilities range from 0 to 1

        # Show the plot
        plt.title(f'{len(x_values)} kinase-substrate pairs',fontsize=fontsize)
        plt.show()
        
    return AUCDF

# %% ../nbs/02_plot.ipynb 62
def plot_confusion_matrix(target, # pd.Series 
                          pred, # pd.Series
                          class_names:list=['0','1'],
                          normalize=False,
                          title='Confusion matrix',
                          cmap=plt.cm.Blues):
    
    "Plot the confusion matrix."
    
    cm = confusion_matrix(target, pred)
    
    if normalize:
        cm = cm.astype('float') / cm.sum(axis=1)[:, np.newaxis]
        print("Normalized confusion matrix")
    else:
        print('Confusion matrix, without normalization')


    plt.figure(figsize=(6,6))
    sns.heatmap(cm, annot=True, cmap=cmap)  # Plot the heatmap
    plt.title(title)
    plt.ylabel('True label')
    plt.xlabel('Predicted label')
    plt.xticks(np.arange(len(class_names)) + 0.5, class_names)
    plt.yticks(np.arange(len(class_names)) + 0.5, class_names, rotation=0)
