"""A collection of tools to extract features from SMILES, proteins, etc."""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_feature.ipynb.

# %% auto 0
__all__ = ['get_rdkit', 'get_morgan', 'get_esm', 'get_t5', 'get_t5_bfd', 'reduce_feature', 'remove_hi_corr', 'preprocess']

# %% ../nbs/01_feature.ipynb 4
from fastbook import *
import torch,re,joblib,gc,esm
from tqdm.notebook import tqdm; tqdm.pandas()
from .core import Data

# Rdkit
from rdkit import Chem
from rdkit.ML.Descriptors import MoleculeDescriptors
from rdkit.Chem import Draw,Descriptors,AllChem,rdFingerprintGenerator

# Models
from fairscale.nn.data_parallel import FullyShardedDataParallel as FSDP
from fairscale.nn.wrap import enable_wrap, wrap
from transformers import T5Tokenizer, T5EncoderModel, T5Model

# Dimension Reduction
from sklearn import set_config
from sklearn.decomposition import PCA
from sklearn.manifold import TSNE
from sklearn.preprocessing import StandardScaler
from umap.umap_ import UMAP

set_config(transform_output="pandas")

# %% ../nbs/01_feature.ipynb 7
def get_rdkit(df: pd.DataFrame, # a dataframe that contains smiles
              col:str = "SMILES", # colname of smile
              normalize: bool = True, # normalize features using StandardScaler()
              ):
    "Extract chemical features from smiles via rdkit.Chem.Descriptors; if normalize, apply StandardScaler"
    
    mols = [Chem.MolFromSmiles(smi) for smi in df[col]]
    desc_names = [desc_name[0] for desc_name in Descriptors.descList]
    desc_calc = MoleculeDescriptors.MolecularDescriptorCalculator(desc_names)
    desc_values = [desc_calc.CalcDescriptors(mol) for mol in mols]
    feature_df = pd.DataFrame(np.stack(desc_values), index=df.index,columns=desc_names)
    
    if normalize:
        feature_df = StandardScaler().fit_transform(feature_df)
        
    # feature_df = feature_df.reset_index()
    return feature_df

# %% ../nbs/01_feature.ipynb 11
def get_morgan(df: pd.DataFrame, # a dataframe that contains smiles
               col: str = "SMILES", # colname of smile
               radius=3
              ):
    "Get 2048 morgan fingerprint (binary feature) from smiles in a dataframe"
    mols = [Chem.MolFromSmiles(smi) for smi in df[col]]

    mfpgen = rdFingerprintGenerator.GetMorganGenerator(radius=radius,fpSize=2048)
    morgan_fps = [mfpgen.GetFingerprint(mol) for mol in mols]
    
    fp_df = pd.DataFrame(np.array(morgan_fps), index=df.index)
    fp_df.columns = "morgan_" + fp_df.columns.astype(str)
    return fp_df

# %% ../nbs/01_feature.ipynb 16
def get_esm(df:pd.DataFrame, # a dataframe that contains amino acid sequence
            col: str = 'sequence', # colname of amino acid sequence
            model_name: str = "esm2_t33_650M_UR50D", # Name of the ESM model to use for the embeddings.
            ):
    
    "Extract esmfold2 embeddings from protein sequence in a dataframe"
    
    # Initialize distributed world with world_size 1
    if not torch.distributed.is_initialized():
        url = "tcp://localhost:23456"
        torch.distributed.init_process_group(backend="nccl", init_method=url, world_size=1, rank=0)
    
    #get number of repr layers
    match = re.search(r'_t(\d+)_', model_name)
    number = int(match.group(1))
    print(f"repr_layers number for model {model_name} is {number}.")
    print("You can also choose other esm2 models:",
          "\nesm2_t48_15B_UR50D\nesm2_t36_3B_UR50D\nesm2_t33_650M_UR50D\nesm2_t30_150M_UR50D\nesm2_t12_35M_UR50D\nesm2_t6_8M_UR50D\n")

    # Download model data from the hub
    model_data, regression_data = esm.pretrained._download_model_and_regression_data(model_name)

    # Initialize the model with FSDP wrapper
    fsdp_params = dict(
        mixed_precision=True,
        flatten_parameters=True,
        state_dict_device=torch.device("cpu"),  # reduce GPU mem usage
        cpu_offload=True,  # enable cpu offloading
    )

    with enable_wrap(wrapper_cls=FSDP, **fsdp_params):
        model, vocab = esm.pretrained.load_model_and_alphabet_core(
            model_name, model_data, regression_data
        )
        batch_converter = vocab.get_batch_converter()
        model.eval()

        # Wrap each layer in FSDP separately
        for name, child in model.named_children():
            if name == "layers":
                for layer_name, layer in child.named_children():
                    wrapped_layer = wrap(layer)
                    setattr(child, layer_name, wrapped_layer)
        model = wrap(model)

        # Define the feature extraction function
        def esm_embeddings(r, colname=col):
            data = [('protein', r[colname])]
            labels, strs, tokens = batch_converter(data)
            with torch.no_grad():
                results = model(tokens.cuda(), repr_layers=[number], return_contacts=False)
            rpr = results["representations"][number].squeeze()
            rpr = rpr[1 : len(r[colname]) + 1].mean(0).detach().cpu().numpy()

            del results, labels, strs, tokens, data #especially need to delete those on cuda: tokens, results
            gc.collect()

            return rpr
        
        # Apply the feature extraction function to each row in the DataFrame
        series = df.progress_apply(esm_embeddings, axis=1)
        df_feature = pd.DataFrame(series.tolist(), index=df.index)
        df_feature.columns = 'esm_' + df_feature.columns.astype(str)

        return df_feature

# %% ../nbs/01_feature.ipynb 20
def get_t5(df: pd.DataFrame, 
           col: str = 'sequence'
           ):
    "Extract ProtT5-XL-uniref50 embeddings from protein sequence in a dataframe"
    
    # Reference: https://github.com/agemagician/ProtTrans/tree/master/Embedding/PyTorch/Advanced
    # Load the tokenizer
    tokenizer = T5Tokenizer.from_pretrained('Rostlab/prot_t5_xl_half_uniref50-enc', do_lower_case=False)

    # Load the model
    model = T5EncoderModel.from_pretrained("Rostlab/prot_t5_xl_half_uniref50-enc").to('cuda')

    # Set the model precision based on the device
    model.half()
    
    def T5_embeddings(sequence):
        seq_len = len(sequence)
        # Prepare the protein sequences as a list
        sequence = [" ".join(list(re.sub(r"[UZOB]", "X", sequence)))]

        # Tokenize sequences and pad up to the longest sequence in the batch
        ids = tokenizer.batch_encode_plus(sequence, add_special_tokens=True, padding="longest")
        input_ids = torch.tensor(ids['input_ids']).to('cuda')
        attention_mask = torch.tensor(ids['attention_mask']).to('cuda')

        # Generate embeddings
        with torch.no_grad():
            embedding_rpr = model(input_ids=input_ids, attention_mask=attention_mask)

        emb_mean = embedding_rpr.last_hidden_state[0][:seq_len].detach().cpu().numpy().mean(axis=0)

        return emb_mean

    series = df[col].progress_apply(T5_embeddings)
        

    T5_feature = pd.DataFrame(series.tolist(),index=df.index)
    T5_feature.columns = 'T5_' + T5_feature.columns.astype(str)
    
    return T5_feature

# %% ../nbs/01_feature.ipynb 23
def get_t5_bfd(df:pd.DataFrame, 
               col: str = 'sequence'
               ):
    
    "Extract ProtT5-XL-BFD embeddings from protein sequence in a dataframe"
    # Reference: https://github.com/agemagician/ProtTrans/tree/master/Embedding/PyTorch/Advanced
    # Load the tokenizer
    tokenizer = T5Tokenizer.from_pretrained('Rostlab/prot_t5_xl_bfd', do_lower_case=False)

    model = T5Model.from_pretrained("Rostlab/prot_t5_xl_bfd").to('cuda')

    model.eval()
    
    def T5_embeddings_bfd(sequence, device = 'cuda'):

        seq_len = len(sequence)

        # Prepare the protein sequences as a list
        sequence = [" ".join(list(re.sub(r"[UZOB]", "X", sequence)))]

        # Tokenize sequences and pad up to the longest sequence in the batch
        ids = tokenizer.batch_encode_plus(sequence, add_special_tokens=True, padding="longest")
        input_ids = torch.tensor(ids['input_ids']).to(device)
        attention_mask = torch.tensor(ids['attention_mask']).to(device)

        # Generate embeddings
        with torch.no_grad():
            embedding_rpr = model(input_ids=input_ids, attention_mask=attention_mask, decoder_input_ids = input_ids)

        emb_mean = embedding_rpr.last_hidden_state[0][:seq_len].detach().cpu().numpy().mean(axis=0)

        return emb_mean

    series = df[col].progress_apply(T5_embeddings_bfd)
        

    T5_feature = pd.DataFrame(series.tolist(),index=df.index)
    T5_feature.columns = 'T5bfd_' + T5_feature.columns.astype(str)
    
    return T5_feature

# %% ../nbs/01_feature.ipynb 27
def reduce_feature(df: pd.DataFrame, 
                   method: str='pca', # dimensionality reduction method, accept both capital and lower case
                   complexity: int=20, # None for PCA; perfplexity for TSNE, recommend: 30; n_neigbors for UMAP, recommend: 15
                   n: int=2, # n_components
                   load: str=None, # load a previous model, e.g. model.pkl
                   save: str=None, # pkl file to be saved, e.g. pca_model.pkl
                   seed: int=123, # seed for random_state
                   **kwargs, # arguments from PCA, TSNE, or UMAP depends on which method to use
                  ):
    
    "Reduce the dimensionality given a dataframe of values"
    
    method = method.lower()
    assert method in ['pca','tsne','umap'], "Please choose a method among PCA, TSNE, and UMAP"
    
    if load is not None:
        reducer = joblib.load(load)
    else:
        if method == 'pca':
            reducer = PCA(n_components=n, random_state=seed,**kwargs)
        elif method == 'tsne':
            reducer = TSNE(n_components=n,
                           random_state=seed, 
                           perplexity = complexity, # default from official is 30 
                          **kwargs)
        elif method == 'umap':
            reducer = UMAP(n_components=n, 
                           random_state=seed, 
                           n_neighbors=complexity, # default from official is 15, try 15-200
                          **kwargs)
        else:
            raise ValueError('Invalid method specified')

    proj = reducer.fit_transform(df)
    embedding_df = pd.DataFrame(proj).set_index(df.index)
    embedding_df.columns = [f"{method.upper()}{i}" for i in range(1, n + 1)]

    if save is not None:
        path = Path(save)
        path.parent.mkdir(exist_ok=True)
        
        joblib.dump(reducer, save)

    return embedding_df

# %% ../nbs/01_feature.ipynb 30
def remove_hi_corr(df: pd.DataFrame, 
                   thr: float=0.98 # threshold
                   ):
    "Remove highly correlated features in a dataframe given a pearson threshold"
    
    # Create correlation matrix
    corr_matrix = df.corr().abs()
    
    # Select upper triangle of correlation matrix
    upper = corr_matrix.where(np.triu(np.ones(corr_matrix.shape), k=1).astype(bool))
    
    # Find index of feature columns with correlation greater than threshold
    to_drop = [column for column in upper.columns if any(upper[column] > thr)]
    
    # Drop features 
    df = df.drop(to_drop, axis=1)
    
    return df

# %% ../nbs/01_feature.ipynb 34
def preprocess(df: pd.DataFrame,
               thr: float=0.98):
    
    "Remove features with no variance, and highly correlated features based on threshold"
    
    df_original = df.copy()
    df = df.loc[:,df.std() != 0]
    df = remove_hi_corr(df, thr)
    dropping_col = set(df_original.columns) - set(df.columns)
    print(f'removing columns: {dropping_col}')
    return df
