"""A collection of machine learning tools"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/03_ML.ipynb.

# %% auto 0
__all__ = ['get_splits', 'split_data', 'score_each', 'train_ml', 'train_ml_cv', 'predict_ml']

# %% ../nbs/03_ML.ipynb 4
# katlas
from .core import Data
from .feature import *

# essentials
import pandas as pd, numpy as np
from joblib import dump, load
import math,matplotlib.pyplot as plt
from pathlib import Path

# scipy
from scipy.stats import spearmanr, pearsonr
from scipy.stats import spearmanr,pearsonr

# sklearn
from sklearn.model_selection import *
from sklearn.metrics import mean_squared_error
from sklearn.linear_model import *
from sklearn.svm import *
from sklearn.ensemble import *
from sklearn import set_config
set_config(transform_output="pandas")

# %% ../nbs/03_ML.ipynb 7
def get_splits(df: pd.DataFrame, # df contains info for split
               stratified: str=None, # colname to make stratified kfold; sampling from different groups
               group: str=None, # colname to make group kfold; test and train are from different groups
               nfold: int=5,
               seed: int=123):
    
    "Split samples in a dataframe based on Stratified, Group, or StratifiedGroup Kfold method"
    def _log(colname):
        print(kf)
        split=splits[0]
        print(f'# kinase {colname} in train set: {df.loc[split[0]][colname].unique().shape[0]}')
        print(f'# kinase {colname} in test set: {df.loc[split[1]][colname].unique().shape[0]}')
        
    splits = []
    if stratified is not None and group is None:
        kf = StratifiedKFold(nfold, shuffle=True, random_state=seed)
        for split in kf.split(df.index, df[stratified]):
            splits.append(split)
            
        _log(stratified)
        
    elif group is not None and stratified is None:
        kf = GroupKFold(nfold)
        for split in kf.split(df.index, groups=df[group]):
            splits.append(split)
            
        _log(group)
        
    elif stratified is not None and group is not None:
        kf = StratifiedGroupKFold(nfold, shuffle=True, random_state=seed)
        for split in kf.split(df.index, groups=df[group], y=df[stratified]):
            splits.append(split)
            
        _log(stratified)

    else:
        raise ValueError("Either 'stratified' or 'group' argument must be provided.")
        
        
    print('---------------------------')
    print(f'# kinase in train set: {df.loc[split[0]].kinase.unique().shape[0]}')
    
    print('---------------------------')
    print(f'# kinase in test set: {df.loc[split[1]].kinase.unique().shape[0]}')
    print('---------------------------')
    print(f'test set: {df.loc[split[1]].kinase.unique()}')
    
    return splits

# %% ../nbs/03_ML.ipynb 12
def split_data(df: pd.DataFrame, # dataframe of values
               feat_col: list, # feature columns
               target_col: list, # target columns
               split: tuple # one of the split in splits
               ):
    "Given split tuple, split dataframe into X_train, y_train, X_test, y_test"
    
    X_train = df.loc[split[0]][feat_col]
    y_train = df.loc[split[0]][target_col]
    
    X_test = df.loc[split[1]][feat_col]
    y_test = df.loc[split[1]][target_col]
    
    return X_train, y_train, X_test, y_test

# %% ../nbs/03_ML.ipynb 16
def score_each(target: pd.DataFrame, # target dataframe
              pred: pd.DataFrame, # predicted dataframe
              absolute = True, # if absolute, take average with absolute values for pearson/spearman
              verbose=True, # whether or not display the error value
              
             ):
    
    "Calculate the overall MSE and average Pearson (per row) between two dataframes."
    
    # Calculate overall MSE
    mse = mean_squared_error(target, pred)
    
    # Calculate RMSE
    # rmse = math.sqrt(mse)

    # Calculate the Spearman correlation coefficient
    # spearman, _ = spearmanr(target.values.ravel(), pred.values.ravel())

    # Calculate the Pearson correlation coefficient
    # pearson, _ = pearsonr(target.values.ravel(), pred.values.ravel())
    
    # Calculate Spearman and Pearson per row 
    # spearman_all = target.corrwith(pred,axis=1,method='spearman')
    pearson_all = target.corrwith(pred,axis=1,method='pearson')
    
    # Calculate mean
    # spearman_mean = spearman_all.abs().mean()
    pearson_mean = pearson_all.abs().mean()
    
    metrics_df = pd.DataFrame({'Pearson':pearson_all})
    
    if verbose:
        print(f'overall MSE: {mse:.4f}')
        # print(f"Average Spearman:  {spearman_mean:.4f}")
        print(f"Average Pearson: {pearson_mean:.4f} ")
    
    return mse,pearson_mean, metrics_df

# %% ../nbs/03_ML.ipynb 21
def train_ml(df, # dataframe of values
             feat_col, # feature columns
             target_col, # target columns
             split, # one split in splits
             model,  # a sklearn models
             save = None, # file (.joblib) to save, e.g. 'model.joblib'
             params={}, # parameters for model.fit from sklearn
            ):
    
    "Fit and predict using sklearn model format, return target and pred of valid dataset."
    
    # split data
    X_train, y_train, X_test, y_test = split_data(df, feat_col, target_col, split)
    
    # Fit the model
    model.fit(X_train, y_train, **params) # better convert y_train to numpy array and flatten
    print(model)
    
    if save is not None:
        # Save the model to a file
        # joblib.dump(model, save)
        dump(model, save)
        
    # Predict train
    y_train_pred = model.predict(X_train) # X_test is dataframe, y_pred is numpy array
    
    # Predict test
    y_pred = model.predict(X_test) # X_test is dataframe, y_pred is numpy array

    # Make dataframe
    y_pred = pd.DataFrame(y_pred,index=y_test.index, columns = y_test.columns)
    
    return y_test, y_pred

# %% ../nbs/03_ML.ipynb 24
def train_ml_cv( df, # dataframe of values
                 feat_col, # feature columns
                 target_col,  # target columns
                 splits, # splits
                 model, # sklearn model
                 save = None, # model name to be saved, e.g., 'LR'
                 params = {}, # act as kwargs, for model.fit
                ):
    
    "Cross-validation through the given splits"
    
    OOF = []
    metrics = []
    
    for fold, split in enumerate(splits):
        print(f'------ fold: {fold} --------')
        
        if save is not None: 
            save = f'models/{save}_{fold}.joblib'
            
        target, pred = train_ml(df, feat_col, target_col, split, model,save,params=params)

        mse, pearson_avg, _ = score_each(target,pred)
        
        # Store metrics in a dictionary for the current fold
        fold_metrics = {
            'fold': fold,
            'mse': mse,
            'pearson_avg': pearson_avg
        }
        
        metrics.append(fold_metrics)

        OOF.append(pred)
        
    # Concatenate OOF from each fold to a new dataframe
    oof = pd.concat(OOF).sort_index()
    
    # Get metrics into a dataframe
    metrics = pd.DataFrame(metrics)
    
    return oof, metrics

# %% ../nbs/03_ML.ipynb 31
def predict_ml(df, # Dataframe that contains features
               feat_col, # feature columns
               target_col=None,
               model_pth = 'model.joblib'
              ):
    
    "Make predictions based on trained model."
    
    test = df[feat_col]
    
    model = load(model_pth)
    
    pred = model.predict(test)
    
    pred_df = pd.DataFrame(pred,index=df.index,columns=target_col)
    
    return pred_df
