"""A collection of tools to extract features from SMILES, proteins, etc."""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/04_feature.ipynb.

# %% auto 0
__all__ = ['remove_hi_corr', 'preprocess', 'standardize', 'get_rdkit', 'get_rdkit_3d', 'get_rdkit_all', 'get_rdkit_df',
           'get_morgan', 'onehot_encode', 'onehot_encode_df', 'kmeans', 'filter_range_columns', 'get_clusters_elbow',
           'get_esm', 'get_t5', 'get_t5_bfd']

# %% ../nbs/04_feature.ipynb 3
import pandas as pd, numpy as np
import torch,re,joblib,gc,esm
from tqdm.notebook import tqdm; tqdm.pandas()
from .data import Data
from sklearn.preprocessing import OneHotEncoder,StandardScaler

# Rdkit
from rdkit import Chem
from rdkit.ML.Descriptors import MoleculeDescriptors
from rdkit.Chem import Draw,Descriptors,Descriptors3D, AllChem,rdFingerprintGenerator

# Models
from fairscale.nn.data_parallel import FullyShardedDataParallel as FSDP
from fairscale.nn.wrap import enable_wrap, wrap
from transformers import T5Tokenizer, T5EncoderModel, T5Model

# Elbow
from sklearn.cluster import KMeans
from matplotlib import pyplot as plt

from sklearn import set_config
set_config(transform_output="pandas")

# %% ../nbs/04_feature.ipynb 5
def remove_hi_corr(df: pd.DataFrame, 
                   thr: float=0.98 # threshold
                   ):
    "Remove highly correlated features in a dataframe given a pearson threshold"
    
    # Create correlation matrix
    corr_matrix = df.corr().abs()
    
    # Select upper triangle of correlation matrix
    upper = corr_matrix.where(np.triu(np.ones(corr_matrix.shape), k=1).astype(bool))
    
    # Find index of feature columns with correlation greater than threshold
    to_drop = [column for column in upper.columns if any(upper[column] > thr)]
    
    # Drop features 
    df = df.drop(to_drop, axis=1)
    
    return df

# %% ../nbs/04_feature.ipynb 9
def preprocess(df: pd.DataFrame,
               thr: float=0.98):
    
    "Remove features with no variance, and highly correlated features based on threshold"
    
    col_ori = df.columns
    df = df.loc[:,df.std() != 0].copy()
    df = remove_hi_corr(df, thr)
    dropping_col = set(col_ori) - set(df.columns)
    print(f'removing columns: {dropping_col}')
    return df

# %% ../nbs/04_feature.ipynb 12
def standardize(df): 
    "Standardize features from a df"
    return StandardScaler().fit_transform(df.copy())

# %% ../nbs/04_feature.ipynb 15
def get_rdkit(SMILES):
    """
    Extract chemical features from SMILES
    Reference: https://greglandrum.github.io/rdkit-blog/posts/2022-12-23-descriptor-tutorial.html
    """
    mol = Chem.MolFromSmiles(SMILES)
    return Descriptors.CalcMolDescriptors(mol)

# %% ../nbs/04_feature.ipynb 16
def get_rdkit_3d(SMILES):
    """
    Extract 3d features from SMILES
    """
    mol = Chem.MolFromSmiles(SMILES)
    mol = Chem.AddHs(mol)
    AllChem.EmbedMolecule(mol, AllChem.ETKDG())
    AllChem.UFFOptimizeMolecule(mol)
    return Descriptors3D.CalcMolDescriptors3D(mol)

# %% ../nbs/04_feature.ipynb 17
def get_rdkit_all(SMILES):
    "Extract chemical features and 3d features from SMILES"
    feat = get_rdkit(SMILES)
    feat_3d = get_rdkit_3d(SMILES)
    return feat|feat_3d

# %% ../nbs/04_feature.ipynb 18
def get_rdkit_df(df,
                 col, # column of SMILES
                 postprocess=True, # remove redundant columns and standardize features for dimension reduction
                 ):
    "Extract rdkit features (including 3d) from SMILES in a df"
    out = df[col].apply(get_rdkit_all).apply(pd.Series)

    if postprocess:
        out = preprocess(out) # remove redundant
        out = standardize(out)
    return out

# %% ../nbs/04_feature.ipynb 22
def get_morgan(df: pd.DataFrame, # a dataframe that contains smiles
               col: str = "SMILES", # colname of smile
               radius=3
              ):
    "Get 2048 morgan fingerprint (binary feature) from smiles in a dataframe"
    mols = [Chem.MolFromSmiles(smi) for smi in df[col]]

    mfpgen = rdFingerprintGenerator.GetMorganGenerator(radius=radius,fpSize=2048)
    morgan_fps = [mfpgen.GetFingerprint(mol) for mol in mols]
    
    fp_df = pd.DataFrame(np.array(morgan_fps), index=df.index)
    fp_df.columns = "morgan_" + fp_df.columns.astype(str)
    return fp_df

# %% ../nbs/04_feature.ipynb 27
def onehot_encode(sequences, transform_colname=True, n=20):
    encoder = OneHotEncoder(handle_unknown='ignore', sparse_output=False)
    encoded_array = encoder.fit_transform([list(seq) for seq in sequences])
    colnames = [x[1:] for x in encoder.get_feature_names_out()]
    if transform_colname:
        colnames = [f"{int(item.split('_', 1)[0]) - 20}{item.split('_', 1)[1]}" for item in colnames]
    encoded_df = pd.DataFrame(encoded_array)
    encoded_df.columns=colnames
    return encoded_df

# %% ../nbs/04_feature.ipynb 28
def onehot_encode_df(df,seq_col='site_seq', **kwargs):
    return onehot_encode(df[seq_col],**kwargs)

# %% ../nbs/04_feature.ipynb 33
def kmeans(onehot,n=2,seed=42):
    "Take onehot encoded and regurn the cluster number."
    kmeans = KMeans(n_clusters=n, random_state=seed,n_init='auto')
    return kmeans.fit_predict(onehot)

# %% ../nbs/04_feature.ipynb 36
def filter_range_columns(df, # df need to have column names of position + aa
                         low=-10,high=10):
    positions = df.columns.str[:-1].astype(int)
    mask = (positions >= low) & (positions <= high)
    return df.loc[:,mask]

# %% ../nbs/04_feature.ipynb 42
def get_clusters_elbow(encoded_data,max_cluster=400, interval=50):

    wcss = []
    for i in range(1, max_cluster,interval):
        kmeans = KMeans(n_clusters=i, random_state=42)
        kmeans.fit(encoded_data)
        wcss.append(kmeans.inertia_)

    # Plot the Elbow graph
    plt.figure(figsize=(5, 3))
    plt.plot(range(1, max_cluster,interval), wcss)
    plt.title(f'Elbow Method (n={len(encoded_data)})')
    plt.xlabel('# Clusters')
    plt.ylabel('WCSS')

# %% ../nbs/04_feature.ipynb 45
def get_esm(df:pd.DataFrame, # a dataframe that contains amino acid sequence
            col: str = 'sequence', # colname of amino acid sequence
            model_name: str = "esm2_t33_650M_UR50D", # Name of the ESM model to use for the embeddings.
            ):
    
    "Extract esmfold2 embeddings from protein sequence in a dataframe"
    
    # Initialize distributed world with world_size 1
    if not torch.distributed.is_initialized():
        url = "tcp://localhost:23456"
        torch.distributed.init_process_group(backend="nccl", init_method=url, world_size=1, rank=0)
    
    #get number of repr layers
    match = re.search(r'_t(\d+)_', model_name)
    number = int(match.group(1))
    print(f"repr_layers number for model {model_name} is {number}.")
    print("You can also choose other esm2 models:",
          "\nesm2_t48_15B_UR50D\nesm2_t36_3B_UR50D\nesm2_t33_650M_UR50D\nesm2_t30_150M_UR50D\nesm2_t12_35M_UR50D\nesm2_t6_8M_UR50D\n")

    # Download model data from the hub
    model_data, regression_data = esm.pretrained._download_model_and_regression_data(model_name)

    # Initialize the model with FSDP wrapper
    fsdp_params = dict(
        mixed_precision=True,
        flatten_parameters=True,
        state_dict_device=torch.device("cpu"),  # reduce GPU mem usage
        cpu_offload=True,  # enable cpu offloading
    )

    with enable_wrap(wrapper_cls=FSDP, **fsdp_params):
        model, vocab = esm.pretrained.load_model_and_alphabet_core(
            model_name, model_data, regression_data
        )
        batch_converter = vocab.get_batch_converter()
        model.eval()

        # Wrap each layer in FSDP separately
        for name, child in model.named_children():
            if name == "layers":
                for layer_name, layer in child.named_children():
                    wrapped_layer = wrap(layer)
                    setattr(child, layer_name, wrapped_layer)
        model = wrap(model)

        # Define the feature extraction function
        def esm_embeddings(r, colname=col):
            data = [('protein', r[colname])]
            labels, strs, tokens = batch_converter(data)
            with torch.no_grad():
                results = model(tokens.cuda(), repr_layers=[number], return_contacts=False)
            rpr = results["representations"][number].squeeze()
            rpr = rpr[1 : len(r[colname]) + 1].mean(0).detach().cpu().numpy()

            del results, labels, strs, tokens, data #especially need to delete those on cuda: tokens, results
            gc.collect()

            return rpr
        
        # Apply the feature extraction function to each row in the DataFrame
        series = df.progress_apply(esm_embeddings, axis=1)
        df_feature = pd.DataFrame(series.tolist(), index=df.index)
        df_feature.columns = 'esm_' + df_feature.columns.astype(str)

        return df_feature

# %% ../nbs/04_feature.ipynb 49
def get_t5(df: pd.DataFrame, 
           col: str = 'sequence'
           ):
    "Extract ProtT5-XL-uniref50 embeddings from protein sequence in a dataframe"
    
    # Reference: https://github.com/agemagician/ProtTrans/tree/master/Embedding/PyTorch/Advanced
    # Load the tokenizer
    tokenizer = T5Tokenizer.from_pretrained('Rostlab/prot_t5_xl_half_uniref50-enc', do_lower_case=False)

    # Load the model
    model = T5EncoderModel.from_pretrained("Rostlab/prot_t5_xl_half_uniref50-enc").to('cuda')

    # Set the model precision based on the device
    model.half()
    
    def T5_embeddings(sequence):
        seq_len = len(sequence)
        # Prepare the protein sequences as a list
        sequence = [" ".join(list(re.sub(r"[UZOB]", "X", sequence)))]

        # Tokenize sequences and pad up to the longest sequence in the batch
        ids = tokenizer.batch_encode_plus(sequence, add_special_tokens=True, padding="longest")
        input_ids = torch.tensor(ids['input_ids']).to('cuda')
        attention_mask = torch.tensor(ids['attention_mask']).to('cuda')

        # Generate embeddings
        with torch.no_grad():
            embedding_rpr = model(input_ids=input_ids, attention_mask=attention_mask)

        emb_mean = embedding_rpr.last_hidden_state[0][:seq_len].detach().cpu().numpy().mean(axis=0)

        return emb_mean

    series = df[col].progress_apply(T5_embeddings)
        

    T5_feature = pd.DataFrame(series.tolist(),index=df.index)
    T5_feature.columns = 'T5_' + T5_feature.columns.astype(str)
    
    return T5_feature

# %% ../nbs/04_feature.ipynb 52
def get_t5_bfd(df:pd.DataFrame, 
               col: str = 'sequence'
               ):
    
    "Extract ProtT5-XL-BFD embeddings from protein sequence in a dataframe"
    # Reference: https://github.com/agemagician/ProtTrans/tree/master/Embedding/PyTorch/Advanced
    # Load the tokenizer
    tokenizer = T5Tokenizer.from_pretrained('Rostlab/prot_t5_xl_bfd', do_lower_case=False)

    model = T5Model.from_pretrained("Rostlab/prot_t5_xl_bfd").to('cuda')

    model.eval()
    
    def T5_embeddings_bfd(sequence, device = 'cuda'):

        seq_len = len(sequence)

        # Prepare the protein sequences as a list
        sequence = [" ".join(list(re.sub(r"[UZOB]", "X", sequence)))]

        # Tokenize sequences and pad up to the longest sequence in the batch
        ids = tokenizer.batch_encode_plus(sequence, add_special_tokens=True, padding="longest")
        input_ids = torch.tensor(ids['input_ids']).to(device)
        attention_mask = torch.tensor(ids['attention_mask']).to(device)

        # Generate embeddings
        with torch.no_grad():
            embedding_rpr = model(input_ids=input_ids, attention_mask=attention_mask, decoder_input_ids = input_ids)

        emb_mean = embedding_rpr.last_hidden_state[0][:seq_len].detach().cpu().numpy().mean(axis=0)

        return emb_mean

    series = df[col].progress_apply(T5_embeddings_bfd)
        

    T5_feature = pd.DataFrame(series.tolist(),index=df.index)
    T5_feature.columns = 'T5bfd_' + T5_feature.columns.astype(str)
    
    return T5_feature
