"""Functions related with PSSMs"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/02_pssm.ipynb.

# %% auto 0
__all__ = ['EPSILON', 'get_prob', 'flatten_pssm', 'recover_pssm', 'clean_zero_normalize', 'get_pssm_LO', 'get_pssm_LO_flat',
           'get_cluster_pssms', 'get_entropy', 'get_entropy_flat', 'get_IC', 'get_IC_flat', 'get_specificity',
           'get_specificity_flat', 'plot_heatmap_simple', 'sty2pSTY_df', 'plot_heatmap', 'plot_two_heatmaps',
           'plot_logo_raw', 'change_center_name', 'get_pos_min_max', 'scale_zero_position', 'scale_pos_neg_values',
           'convert_logo_df', 'get_logo_IC', 'plot_logo', 'plot_logo_LO', 'plot_logos_idx', 'plot_logos',
           'plot_logo_heatmap', 'plot_logo_heatmap_LO', 'preprocess_pspa', 'plot_logo_pspa', 'plot_logo_heatmap_pspa',
           'raw2norm', 'get_one_kinase', 'get_logo', 'kl_divergence', 'kl_divergence_flat', 'js_divergence',
           'js_divergence_flat', 'js_similarity', 'js_similarity_flat', 'cosine_similarity', 'cosine_overall_flat']

# %% ../nbs/02_pssm.ipynb 3
import numpy as np, pandas as pd
from .data import *
from .utils import *
from fastcore.meta import delegates
from tqdm import tqdm
from tqdm.contrib.concurrent import process_map
from functools import partial
from fastcore.meta import delegates

# for plot
from matplotlib import pyplot as plt
import logomaker,math
import seaborn as sns

from matplotlib.colors import TwoSlopeNorm

# for plot two heatmaps
from matplotlib.patches import Rectangle
from mpl_toolkits.axes_grid1 import make_axes_locatable
from matplotlib.ticker import FuncFormatter

# hierarchical clustering
from scipy.cluster.hierarchy import linkage, fcluster,dendrogram

# for type
from typing import Sequence

# %% ../nbs/02_pssm.ipynb 6
EPSILON = 1e-8

# %% ../nbs/02_pssm.ipynb 9
def get_prob(data: pd.DataFrame | pd.Series | Sequence[str], # input data, list or df
             col: str='site_seq', # column name if input is df
             ):
    "Get the probability matrix of PSSM from phosphorylation site sequences."

    aa_order=[i for i in 'PGACSTVILMFYWHKRQNDEsty']

    if isinstance(data, pd.DataFrame):
        if col not in data.columns:
            raise ValueError(f"Column '{col}' not found in DataFrame.")
        site = data[col]
    else:
        try:
            site = pd.Series(data,copy=False)
        except Exception:
            raise TypeError("Input must be a DataFrame, Series, or list of sequences.")
    
    
    site = check_seqs(site)
    
    site_array = np.array(site.apply(list).tolist())
    seq_len = site_array.shape[1]
    
    position = list(range(-(seq_len // 2), (seq_len // 2)+1)) # add 1 because range do not include the final num
    
    site_df = pd.DataFrame(site_array, columns=position)
    melted = site_df.melt(var_name='Position', value_name='aa')
    
    grouped = melted.groupby(['Position', 'aa']).size().reset_index(name='Count')
    grouped = grouped[grouped.aa.isin(aa_order)].reset_index(drop=True)
    
    pivot_df = grouped.pivot(index='aa', columns='Position', values='Count').fillna(0)
    pssm_df = pivot_df / pivot_df.sum()
    
    pssm_df = pssm_df.reindex(index=aa_order, columns=position, fill_value=0)
    # pssm_df = pssm_df.rename(index={'s': 'pS', 't': 'pT', 'y': 'pY'}) # convert sty to pS,pT,pY
    
    return pssm_df

# %% ../nbs/02_pssm.ipynb 14
def flatten_pssm(pssm_df,
                 column_wise=True, # if True, column major flatten; else row wise flatten (for pytorch training)
                ):
    "Flatten PSSM dataframe to dictionary"
    
    pssm_df=pssm_df.copy()

    # Flatten a pssm_df
    # Column wise
    if column_wise: 
        pssm = pssm_df.unstack().reset_index(name='value')
        # Combine position column and residue identity column as new column for keys
        pssm['position_residue']=pssm.iloc[:,0].astype(str)+pssm.iloc[:,1]
        
    # Row wise
    else: 
        pssm = pssm_df.T.unstack().reset_index(name='value')
        pssm['position_residue']=pssm.iloc[:,1].astype(str)+pssm.iloc[:,0].astype(str)
    
    # Set index to be position+residue
    return pssm.set_index('position_residue')['value'].to_dict()

# %% ../nbs/02_pssm.ipynb 17
def recover_pssm(flat_pssm: pd.Series):
    "Recover 2D PSSM from flattened PSSM Series."
    df = flat_pssm.reset_index()
    df.columns=['index', 'value']
    df['Position'] = df['index'].str[:-1].astype(int)
    df['aa'] = df['index'].str[-1]

    df = df.pivot(index='aa', columns='Position', values='value').fillna(0)
    aa_order=tuple('PGACSTVILMFYWHKRQNDEsty')
    order = [aa for aa in aa_order if aa in df.index]
    return df.reindex(index=order)

# %% ../nbs/02_pssm.ipynb 25
def _clean_zero(pssm_df):
    "Zero out non-last three values in position 0 (keep only s,t,y values at center)"
    pssm_df = pssm_df.copy()
    standard_aa = list(set(pssm_df.index)-set(['s','t','y']))
    pssm_df.loc[standard_aa, 0] = 0
    return pssm_df

# %% ../nbs/02_pssm.ipynb 27
def clean_zero_normalize(pssm_df):
    "Zero out non-last three values in position 0 (keep only s,t,y values at center), and normalize per position"
    pssm_df=pssm_df.copy()
    pssm_df.columns= pssm_df.columns.astype(int)
    pssm_df = _clean_zero(pssm_df)
    return pssm_df/pssm_df.sum()

# %% ../nbs/02_pssm.ipynb 31
def get_pssm_LO(pssm_df,
                site_type, # S, T, Y, ST, or STY
               ):
    "Get log odds PSSM: log2 (freq pssm/background pssm)."
    bg_pssms = Data.get_ks_background()
    flat_bg = bg_pssms.loc[f'ks_{site_type}']
    pssm_bg = recover_pssm(flat_bg)
    pssm_odds = ((pssm_df+EPSILON)/(pssm_bg+EPSILON)).dropna(axis=0,how='all').dropna(axis=1, how='all')
    # make sure all columns and index matched
    assert pssm_odds.shape == pssm_df.shape
    return np.log2(pssm_odds).replace([np.inf, -np.inf], 0).fillna(0)

# %% ../nbs/02_pssm.ipynb 38
def get_pssm_LO_flat(flat_pssm,
                    site_type, # S, T, Y, ST, or STY
                    ):
    pssm_df = recover_pssm(flat_pssm)
    return get_pssm_LO(pssm_df,site_type)

# %% ../nbs/02_pssm.ipynb 41
def get_cluster_pssms(df, 
                    cluster_col, 
                    seq_col='site_seq', 
                    id_col = 'sub_site',
                    count_thr=10, # if less than the count threshold, not include in the return
                    valid_thr=None, # percentage of not-nan values in pssm
                    IC_thr = None,
                      plot=False):
    "Extract motifs from clusters in a dataframe"
    pssms = []
    ids = []
    # drop duplicates based both on cluster column and substrate seq id column
    value_counts = df.drop_duplicates(subset=[cluster_col,id_col])[cluster_col].value_counts()
    for cluster_id, counts in tqdm(value_counts.items(),total=len(value_counts)):
        if count_thr is not None and counts < count_thr:
            continue

        if id_col is not None:
            df_cluster = df[df[cluster_col] == cluster_id].drop_duplicates(id_col)
        else:
            df_cluster = df[df[cluster_col] == cluster_id]
        
        n= len(df_cluster)
        pssm = get_prob(df_cluster, seq_col)
        valid_score = (pssm != 0).sum().sum() / (pssm.shape[0] * pssm.shape[1])

        if valid_thr is not None and valid_score <= valid_thr:
            continue

        pssms.append(flatten_pssm(pssm))
        ids.append(cluster_id)

        if plot:
            plot_logo(pssm, title=f'Cluster {cluster_id} (n={n})', figsize=(14, 1))
            plt.show()
            plt.close()

    pssm_df = pd.DataFrame(pssms, index=ids)
    return pssm_df

# %% ../nbs/02_pssm.ipynb 44
def get_entropy(pssm_df,# a dataframe of pssm with index as aa and column as position
            return_min=False, # return min entropy as a single value or return all entropy as a pd.series
            exclude_zero=False, # exclude the column of 0 (center position) in the entropy calculation
            clean_zero=True, # if true, zero out non-last three values in position 0 (keep only s,t,y values at center)
            ): 
    "Calculate entropy per position of a PSSM surrounding 0. The less entropy the more information it contains."
    pssm_df = pssm_df.copy()
    pssm_df.columns= pssm_df.columns.astype(int)
    if 0 in pssm_df.columns:
        if clean_zero:                       
            pssm_df = _clean_zero(pssm_df)
        if exclude_zero:
            # remove columns starts with zero and columns with interger name 0
            cols_to_drop = [col for col in pssm_df.columns 
                            if col == 0 or (isinstance(col, str) and col.startswith('0'))]
            if cols_to_drop: pssm_df = pssm_df.drop(columns=cols_to_drop)

    pssm_df = pssm_df/pssm_df.sum()
    per_position = -np.sum(pssm_df * np.log2(pssm_df + EPSILON), axis=0)
    per_position[pssm_df.sum() == 0] = 0
    return float(per_position.min()) if return_min else per_position

# %% ../nbs/02_pssm.ipynb 48
@delegates(get_entropy)
def get_entropy_flat(flat_pssm:pd.Series,**kwargs): 
    "Calculate entropy per position of a flat PSSM surrounding 0"
    pssm_df = recover_pssm(flat_pssm)
    return get_entropy(pssm_df,**kwargs)

# %% ../nbs/02_pssm.ipynb 53
@delegates(get_entropy)
def get_IC(pssm_df,**kwargs):
    """
    Calculate the information content (bits) from a frequency matrix,
    using log2(3) for the middle position and log2(len(pssm_df)) for others.
    The higher the more information it contains.
    """
    
    entropy_position = get_entropy(pssm_df,**kwargs)
    
    max_entropy_array = pd.Series(np.log2(len(pssm_df)), index=entropy_position.index)

    # set exclude_zero to False
    exclude_zero = kwargs.get('exclude_zero', False)
    if exclude_zero is False: max_entropy_array[0] = np.log2(3)

    # information_content = max_entropy - entropy --> log2(N) - entropy
    IC_position = max_entropy_array - entropy_position

    # if entropy is zero, set to zero as there's no value
    IC_position[entropy_position == 0] = 0
    return IC_position

# %% ../nbs/02_pssm.ipynb 60
@delegates(get_IC)
def get_IC_flat(flat_pssm:pd.Series,**kwargs):
    """Calculate the information content (bits) from a flattened pssm pd.Series,
    using log2(3) for the middle position and log2(len(pssm_df)) for others."""
    
    pssm_df = recover_pssm(flat_pssm)
    return get_IC(pssm_df,**kwargs)

# %% ../nbs/02_pssm.ipynb 64
def get_specificity(pssm_df):
    "Get specificity score of a pssm, excluding zero position."
    ICs = get_IC(pssm_df, exclude_zero=True)
    # only consider IC with values
    ICs= ICs[ICs > 0]
    return float(2*ICs.max()+ICs.var())

# %% ../nbs/02_pssm.ipynb 67
def get_specificity_flat(flat_pssm):
    "Get specificity score of a pssm, excluding zero position."
    ICs = get_IC_flat(flat_pssm, exclude_zero=True)
    return float(2*ICs.max()+ICs.var())

# %% ../nbs/02_pssm.ipynb 71
@delegates(sns.heatmap)
def plot_heatmap_simple(matrix, # a matrix of values
                 title: str='heatmap', # title of the heatmap
                 figsize: tuple=(6,7), # (width, height)
                 cmap: str='binary', # color map, default is dark&white
                 **kwargs, # arguments for sns.heatmap()
                 ):
    
    "Plot heatmap based on a matrix of values"
    
    plt.figure(figsize=figsize)
    sns.heatmap(matrix, square=True,cmap=cmap, annot=False,**kwargs)
    plt.title(title)
    plt.ylabel('')
    plt.xlabel('')
    plt.yticks(rotation=0)

# %% ../nbs/02_pssm.ipynb 73
def sty2pSTY_df(df):
    df=df.copy()
    df.index = df.index.map(sty2pSTY)
    return df

# %% ../nbs/02_pssm.ipynb 74
def plot_heatmap(heatmap_df, ax=None, position_label=True, figsize=(5, 6), include_zero=True,scale_pos_neg=False, colorbar_title='Prob.'):
    "Plot a heatmap of pssm."
    if ax is None:
        fig, ax = plt.subplots(figsize=figsize)

    heatmap_df = sty2pSTY_df(heatmap_df)
    mask = np.zeros_like(heatmap_df, dtype=bool)
    zero_position = len(heatmap_df.columns) // 2
    second_position = math.ceil(len(heatmap_df.columns) / 2)
    # If they overlap, move the second line one step to the right (if possible)
    if second_position == zero_position: second_position = second_position + 1

    if not include_zero:
        mask[:, zero_position] = True  # Mask position 0 if include_zero is False

    if scale_pos_neg:
        vmin,vmax = heatmap_df.min().min(),heatmap_df.max().max()
        norm = TwoSlopeNorm(vmin=vmin, vcenter=0, vmax=vmax)
        sns.heatmap(
            heatmap_df,
            cmap='coolwarm',
            norm=norm,
            linewidth=0.3,
            ax=ax,
            mask=mask
        )
    else:
        sns.heatmap(
            heatmap_df,
            cmap='coolwarm',
            center=0,  # Center for diverging colormap
            linewidth=0.3,
            ax=ax,
            mask=mask
        )


    # Access and format the color bar
    colorbar = ax.collections[0].colorbar
    colorbar.ax.set_title(colorbar_title, loc='center')

    # Add vertical lines
    ax.axvline(zero_position, color='black', linewidth=0.5)
    ax.axvline(second_position, color='black', linewidth=0.5)

    # Format the heatmap border
    ax.patch.set_edgecolor("black")
    ax.patch.set_linewidth(1.5)

    # Hide axis labels
    ax.set_ylabel("")
    ax.set_xlabel("")
    ax.xaxis.set_ticks_position('top')
    ax.set_yticklabels(ax.get_yticklabels(), rotation=0)
    if not position_label:
        ax.set_xticklabels([])

    return ax

# %% ../nbs/02_pssm.ipynb 79
def plot_two_heatmaps(pssm1, pssm2, 
                      kinase_name="Kinase", title1='CDDM',title2='PSPA',
                      figsize=(4,4.5), cbar=True,scale_01=False,
                      **kwargs):
    """
    Plot two side-by-side heatmaps with black rectangle borders,
    titles on top, shared kinase label below, and only left plot showing y-axis labels.
    """
    fig, axes = plt.subplots(1, 2, figsize=figsize, gridspec_kw={'wspace': 0.05})
    pssm1 = pssm1.drop(columns=0)
    pssm2 = pssm2.drop(columns=0)

    pssm1 = sty2pSTY_df(pssm1)
    # pssm2 = sty2pSTY_df(pssm2)

    # Left heatmap (with y labels)
    im1=sns.heatmap(pssm1, square=False, cmap="Reds", 
                    **{'vmin':0, 'vmax':1} if scale_01 else {},
                annot=False, cbar=False, ax=axes[0], **kwargs)
    axes[0].set_title(title1, fontsize=12, pad=10)
    axes[0].tick_params(left=True, bottom=True)
    axes[0].tick_params(axis="y", rotation=0) 

    # Add rectangle border to left heatmap
    axes[0].add_patch(Rectangle((0,0), pssm1.shape[1], pssm1.shape[0], 
                                fill=False, edgecolor='black', lw=1.5))

    # Right heatmap (no y labels)
    im2=sns.heatmap(pssm2, square=False, cmap="Blues",
                    **{'vmin':0, 'vmax':1} if scale_01 else {},
                annot=False, cbar=False, ax=axes[1], **kwargs)
    axes[1].set_title(title2, fontsize=12, pad=10)
    axes[1].tick_params(left=False, labelleft=False, bottom=True)
    axes[1].tick_params(axis="y", rotation=0) 

    # Add rectangle border to right heatmap
    axes[1].add_patch(Rectangle((0,0), pssm2.shape[1], pssm2.shape[0], 
                                fill=False, edgecolor='black', lw=1.5))

    # Make y tick horizontal
    axes[0].set_title(title1, fontsize=12, pad=0)
    axes[1].set_title(title2, fontsize=12, pad=0)

    # ---- Add vertical separator lines ----
    xpos = (list(pssm1.columns).index(-1) + list(pssm1.columns).index(1)) / 2 + 0.5
    for ax in axes:
        ax.axvline(xpos, color='black', lw=0.75)
        ax.set_xticks([])
        # ax.set_xlabel("")
        ax.set_ylabel("")

    if cbar:
        for ax, im, label in zip(axes, [im1, im2], [title1, title2]):
            divider = make_axes_locatable(ax)
            cax = divider.append_axes("bottom", size="5%", pad=0.25)
            cbar = plt.colorbar(ax.collections[0], cax=cax, orientation="horizontal")
            
            # remove trailing zeros like 0.00 → 0, 1.00 → 1
            cbar.ax.xaxis.set_major_formatter(FuncFormatter(lambda x, _: f"{x:g}"))
            cax.tick_params(labelsize=8)

    # Shared kinase label below
    fig.suptitle(kinase_name, fontsize=14, x=0.52,y=0.96)

# %% ../nbs/02_pssm.ipynb 83
def plot_logo_raw(pssm_df,ax=None,title='Motif',ytitle='Bits',figsize=(10,2)):
    "Plot logo motif using Logomaker."
    if ax is None:
        fig, ax = plt.subplots(figsize=figsize)

    # convert s,t,y to pS,pT,pY for visualization
    pssm_df=sty2pSTY_df(pssm_df)

    logo = logomaker.Logo(pssm_df.T, color_scheme='kinase_protein', flip_below=False, ax=ax)
    logo.ax.set_ylabel(ytitle)
    logo.style_xticks(fmt='%d')
    ax.set_title(title)

# %% ../nbs/02_pssm.ipynb 86
def change_center_name(df):
    "Transfer the middle s,t,y to S,T,Y for plot if s,t,y have values; otherwise keep the original."
    df=df.copy()
    
    if not (df.loc[['s','t','y'],0]==0).all():
        df.loc['S', 0] = df.loc['s', 0]
        df.loc['T', 0] = df.loc['t', 0]
        df.loc['Y', 0] = df.loc['y', 0]
        df.loc[['s', 't', 'y'], 0] = 0
    return df

# %% ../nbs/02_pssm.ipynb 89
def get_pos_min_max(pssm_df):
    """
    Get min and max value of sum of positive and negative values across each position.
    """
    pssm_df = pssm_df.copy()
    pssm_neighbor = pssm_df.drop(columns=0)
    
    max_sum_pos = pssm_neighbor[pssm_neighbor>0].sum().max()
    max_sum_neg = pssm_neighbor[pssm_neighbor<0].sum().min()
    return max_sum_neg,max_sum_pos

# %% ../nbs/02_pssm.ipynb 90
def scale_zero_position(pssm_df):
    """
    Scale position 0 so that:
    - Positive values match the max positive column sum of other positions
    - Negative values match the min (most negative) column sum of other positions
    """
    max_sum_neg,max_sum_pos = get_pos_min_max(pssm_df)

    zero_col = pssm_df[0]
    zero_col_pos = zero_col[zero_col>0]
    zero_col_neg = zero_col[zero_col<0]
    
    scaled_col = zero_col.copy()
    if not zero_col_pos.empty and zero_col_pos.sum() != 0:
        scaled_col.loc[zero_col_pos.index] = max_sum_pos * (zero_col_pos / zero_col_pos.sum())
    if not zero_col_neg.empty and zero_col_neg.sum() != 0:
        scaled_col.loc[zero_col_neg.index] = max_sum_neg * (zero_col_neg / zero_col_neg.sum())

    pssm_df[0] = scaled_col
    return pssm_df
    

# %% ../nbs/02_pssm.ipynb 92
def scale_pos_neg_values(pssm_df):
    """
    Globally scale all positive values by max positive column sum,
    and negative values by min negative column sum (preserving sign).
    """
    pssm_df = pssm_df.copy()
    max_sum_neg, max_sum_pos = get_pos_min_max(pssm_df)

    pos_part = pssm_df.clip(lower=0)
    neg_part = pssm_df.clip(upper=0)

    if max_sum_pos != 0: pos_part = pos_part / max_sum_pos
    if max_sum_neg != 0: neg_part = neg_part / abs(max_sum_neg)  # make sure sign is correct

    return pos_part + neg_part

# %% ../nbs/02_pssm.ipynb 93
def convert_logo_df(pssm_df,scale_zero=True,scale_pos_neg=False):
    "Change center name from s,t,y to S, T, Y in a pssm and scaled zero position to the max of neigbors."
    pssm_df = change_center_name(pssm_df)
    if scale_zero: pssm_df = scale_zero_position(pssm_df)
    if scale_pos_neg: pssm_df = scale_pos_neg_values(pssm_df)
    return pssm_df

# %% ../nbs/02_pssm.ipynb 94
def get_logo_IC(pssm_df):
    """
    For plotting purpose, calculate the scaled information content (bits) from a frequency matrix,
    using log2(3) for the middle position and log2(len(pssm_df)) for others.
    """
    IC_position = get_IC(pssm_df)
    
    return pssm_df.mul(IC_position, axis=1) # total_IC = pssm_df.sum().sum().round(2)

# %% ../nbs/02_pssm.ipynb 97
def plot_logo(pssm_df,title='Motif', scale_zero=True,ax=None,figsize=(10,1)):
    "Plot logo of information content given a frequency PSSM."
    pssm_df = get_logo_IC(pssm_df)
    pssm_df= convert_logo_df(pssm_df,scale_zero=scale_zero)
    plot_logo_raw(pssm_df,ax=ax,title=title,ytitle='IC (bits)',figsize=figsize)

# %% ../nbs/02_pssm.ipynb 103
def plot_logo_LO(pssm_LO,title='Motif', acceptor=None, scale_zero=True,scale_pos_neg=True,ax=None,figsize=(10,1)):
    "Plot logo of log-odds given a frequency PSSM."
    if acceptor is not None: 
        acceptor = acceptor.upper()
        assert acceptor in ['S','T','Y']
        pssm_LO= pssm_LO.copy()
        pssm_LO.loc[acceptor,0]=0.1 # give it a value so that it can be shown on the motif

    # return pssm_LO
    pssm_LO= convert_logo_df(pssm_LO,scale_zero=scale_zero,scale_pos_neg=scale_pos_neg)
    ytitle = "Scaled Log-Odds" if scale_pos_neg else "Log-Odds (bits)"
    plot_logo_raw(pssm_LO,ax=ax,title=title,ytitle=ytitle,figsize=figsize)

# %% ../nbs/02_pssm.ipynb 112
def plot_logos_idx(pssms_df,*idxs,figsize=(14,1)):
    "Plot logos of a dataframe with flattened PSSMs with index ad IDs."
    for idx in idxs:
        pssm = recover_pssm(pssms_df.loc[idx])
        plot_logo(pssm,title=f'Motif {idx}',figsize=figsize)
        plt.show()
        plt.close()

# %% ../nbs/02_pssm.ipynb 116
def plot_logos(pssms_df, 
               count_dict=None, # used to display n in motif title
               path=None,
               prefix='Motif',
               figsize=(14,1)
               ):
    """
    Plot all logos from a dataframe of flattened PSSMs as subplots in a single figure.
    """
    n = len(pssms_df)
    hspace=0.7
    # 14 is width, 1 is height for each logo
    width,height=figsize
    fig, axes = plt.subplots(nrows=n, figsize=(width, n * (height+hspace)),gridspec_kw={'hspace': hspace+0.1})

    if n == 1:
        axes = [axes]  # ensure axes is iterable

    for ax, idx in zip(axes, pssms_df.index):
        pssm = recover_pssm(pssms_df.loc[idx])
        if count_dict is not None:
            plot_logo(pssm, title=f"{prefix or ''} {idx} (n={count_dict[idx]:,})",ax=ax)
        else:
            plot_logo(pssm, title=f"{prefix or ''} {idx}",ax=ax)

# %% ../nbs/02_pssm.ipynb 120
def plot_logo_heatmap(pssm_df, # column is position, index is aa
                       title='Motif',
                       figsize=(17,10),
                       include_zero=False
                      ):
    
    """Plot logo and heatmap vertically"""
    
    fig = plt.figure(figsize=figsize)
    gs = fig.add_gridspec(2, 2, height_ratios=[1, 5], width_ratios=[4, 1], hspace=0.11, wspace=0)

    ax_logo = fig.add_subplot(gs[0, 0])
    plot_logo(pssm_df,ax=ax_logo,title=title)

    ax_heatmap = fig.add_subplot(gs[1, :])
    plot_heatmap(pssm_df,ax=ax_heatmap,position_label=False,include_zero=include_zero)

# %% ../nbs/02_pssm.ipynb 122
def plot_logo_heatmap_LO(pssm_LO, # pssm of log-odds
                             title='Motif',
                         acceptor=None,
                             figsize=(17,10),
                             include_zero=False,
                         scale_pos_neg=True
                      ):
    
    """Plot logo and heatmap of enrichment bits vertically"""
    
    fig = plt.figure(figsize=figsize)
    gs = fig.add_gridspec(2, 2, height_ratios=[1, 5], width_ratios=[4, 1], hspace=0.11, wspace=0)

    ax_logo = fig.add_subplot(gs[0, 0])
    plot_logo_LO(pssm_LO,acceptor=acceptor,ax=ax_logo,title=title)

    ax_heatmap = fig.add_subplot(gs[1, :])
    plot_heatmap(pssm_LO,ax=ax_heatmap,position_label=False,include_zero=include_zero,scale_pos_neg=scale_pos_neg,colorbar_title='bits')

# %% ../nbs/02_pssm.ipynb 128
def preprocess_pspa(pssm):
    pssm = change_center_name(pssm)
    pssm = pssm.drop(index='s')
    pssm.index = pssm.index.map(lambda x: x.replace('t','pS/pT').replace('y','pY'))
    # pssm = np.log2(pssm/pssm.median()) # have to do it without position 0
    non_zero_cols = pssm.columns != 0
    pssm.loc[:, non_zero_cols] = np.log2(
        pssm.loc[:, non_zero_cols] / pssm.loc[:, non_zero_cols].median()
    )
    pssm=scale_zero_position(pssm)
    return pssm

# %% ../nbs/02_pssm.ipynb 130
def plot_logo_pspa(row,title='Motif',figsize=(5,2)):
    pssm = recover_pssm(row.dropna())
    logo_pssm = preprocess_pspa(pssm)
    plot_logo_raw(logo_pssm,ytitle='log₂(Value / Median)',title=title,figsize=figsize)

# %% ../nbs/02_pssm.ipynb 132
def plot_logo_heatmap_pspa(row, # row of Data.get_pspa()
                       title='Motif',
                       figsize=(6,10),
                       include_zero=False
                      ):

    """Plot logo and heatmap vertically"""
    pssm = recover_pssm(row.dropna())
    
    fig = plt.figure(figsize=figsize)
    gs = fig.add_gridspec(2, 2, height_ratios=[1, 5], width_ratios=[4, 1], hspace=0.11, wspace=0)

    ax_logo = fig.add_subplot(gs[0, 0])
    
    logo_pssm = preprocess_pspa(pssm)
    plot_logo_raw(logo_pssm,ax=ax_logo, ytitle='log₂(Value / Median)',title=title)

    ax_heatmap = fig.add_subplot(gs[1, :])
    plot_heatmap(pssm,ax=ax_heatmap,position_label=False,include_zero=include_zero,colorbar_title='Value')

# %% ../nbs/02_pssm.ipynb 135
def raw2norm(df: pd.DataFrame, # single kinase's df has position as index, and single amino acid as columns
             PDHK: bool=False, # whether this kinase belongs to PDHK family 
            ):
    
    "Normalize single ST kinase data"
    columns_to_exclude = ['S', 'T', 'C', 't', 'y']
    
    if PDHK:
        columns_to_exclude.append('Y')
        divisor = 16
    else:
        divisor = 17
    
    s = df.drop(columns=columns_to_exclude).sum(1)
    df2 = df.div(s, axis=0)
    df2.C = df2.C / (df2.C.median() * divisor)
    df2['S'] = df2.drop(columns=columns_to_exclude).median(1)
    df2['T'] = df2.drop(columns=columns_to_exclude).median(1)
    df2 = round(df2, 4)
    
    return df2

# %% ../nbs/02_pssm.ipynb 137
def get_one_kinase(df: pd.DataFrame, #stacked dataframe (paper's raw data)
                   kinase:str, # a specific kinase
                   normalize: bool=False, # normalize according to the paper; special for PDHK1/4
                   drop_s: bool= True, # drop s as s is a duplicates of t in PSPA
                  ):
    "Obtain a specific kinase data from stacked dataframe"
    
    p = pd.DataFrame(df.loc[kinase],columns = [kinase]).reset_index().rename(columns={'index':'substrate'})
    p['position'] = p.substrate.str.extract(r'(-?\d+)')
    p['aa'] = p.substrate.str[-1]
    p.position = p.position.astype(int)
    pp = p.pivot(index='position', columns='aa', values=kinase)
    if drop_s:
        if 's' in pp.columns:
            pp = pp.drop(columns=['s'])

    if normalize:
        pp = raw2norm(pp, PDHK=True if kinase == 'PDHK1' or kinase == 'PDHK4' else False)
    return pp

# %% ../nbs/02_pssm.ipynb 142
def get_logo(df: pd.DataFrame, # stacked Dataframe with kinase as index, substrates as columns
             kinase: str, # a specific kinase name in index
             ):
    "Given stacked df (index as kinase, columns as substrates), get a specific kinase's logo"
    
    
    # get raw kinase to calculate S/T
    pp = get_one_kinase(df,kinase,normalize=False)
    
    # get S/T ratio value
    ss = pp['S'].sum()
    st = pp['T'].sum()

    S_ctrl = 0.75*ss - 0.25*st
    T_ctrl = 0.75*st - 0.25*ss

    S0 = S_ctrl / max(S_ctrl, T_ctrl)
    T0 = T_ctrl / max(S_ctrl, T_ctrl)

    S_ratio = S0/(S0+T0)
    T_ratio = T0/(S0+T0)
    
    # get normalized kinase
    norm_p = get_one_kinase(df,kinase, normalize=True)
    
    # calculate ratio, divide values by median, followed by log2 transformation
    ratio =norm_p.apply(lambda r: r/r.median(),axis=1)
    ratio = np.log2(ratio)

    m = ratio.apply(lambda row: row[row > 0].sum(), axis=1).max()

    new_row = pd.DataFrame({'S': S_ratio*m, 'T':T_ratio*m}, index=[0]) 

    ratio2 = pd.concat([ratio, new_row], ignore_index=False).fillna(0)
    
    # plot logo
    # logo_func(ratio2, kinase)
    plot_logo_raw(ratio2.T,title=kinase,ytitle='log₂(Value / Median)')

# %% ../nbs/02_pssm.ipynb 149
def kl_divergence(p1,  # target pssm p (array-like, shape: (AA, positions))
                  p2,  # pred pssm q (array-like, same shape as p1)
                 ):
    """
    KL divergence D_KL(p1 || p2) over positions.
    
    p1 and p2 are arrays (df or np) with index as aa and column as position.
    Returns average divergence across positions if mean=True, else per-position.
    """
    assert p1.shape == p2.shape
    p1, p2 = p1.align(p2, join='inner', axis=None)
    # Mask invalid positions (both zero)
    valid = (p1 + p2) > 0
    p1 = np.where(valid, p1, 0.0)
    p2 = np.where(valid, p2, 0.0)

    # KL divergence: sum_x p1(x) log(p1(x)/p2(x))
    kl = np.sum(p1 * np.log((p1 + EPSILON) / (p2 + EPSILON)), axis=0)

    return kl

# %% ../nbs/02_pssm.ipynb 153
def kl_divergence_flat(p1_flat, # pd.Series of target flattened pssm p
                       p2_flat, # pd.Series of pred flattened pssm q
                       ):

    "p1 and p2 are two flattened pd.Series with index as aa and column as position"
    kld = kl_divergence(p1_flat,p2_flat) # do not do js.mean() because it's 1d
    total_position = len(p1_flat.index.str.extract(r'(-?\d+)').drop_duplicates())
    return float(kld/total_position)

# %% ../nbs/02_pssm.ipynb 156
def js_divergence(p1, # pssm 
                  p2, # pssm
                  index=True,
                 ):
    "p1 and p2 are two arrays (df or np) with index as aa and column as position"
    assert p1.shape==p2.shape
    p1, p2 = p1.align(p2, join='inner', axis=None)
    if index: positions=p1.columns
    valid = (p1 + p2) > 0
    p1 = np.where(valid, p1, 0.0)
    p2 = np.where(valid, p2, 0.0)
    
    m = 0.5 * (p1 + p2)
    
    js = 0.5 * np.sum(p1 * np.log((p1+ EPSILON) / (m + EPSILON)), axis=0) + \
         0.5 * np.sum(p2 * np.log((p2+ EPSILON) / (m + EPSILON)), axis=0)
    return pd.Series(js,index=positions) if index else js

# %% ../nbs/02_pssm.ipynb 160
def js_divergence_flat(p1_flat, # pd.Series of flattened pssm
                       p2_flat, # pd.Series of flattened pssm
                       ):

    "p1 and p2 are two flattened pd.Series with index as aa and column as position"
    js = js_divergence(p1_flat,p2_flat,index=False)
    total_position = len(p1_flat.index.str.extract(r'(-?\d+)').drop_duplicates())
    return float(js/total_position)

# %% ../nbs/02_pssm.ipynb 164
def js_similarity(pssm1,pssm2):
    "Convert JSD to bits to be in range (0,1) then 1-JSD."
    distance = js_divergence(pssm1,pssm2)/np.log(2)
    similarity = 1-distance
    return similarity

# %% ../nbs/02_pssm.ipynb 166
def js_similarity_flat(p1_flat,p2_flat):
    "Convert JSD to bits to be in range (0,1) then 1-JSD. "
    return 1-(js_divergence_flat(p1_flat,p2_flat)/np.log(2))

# %% ../nbs/02_pssm.ipynb 169
def cosine_similarity(pssm1: pd.DataFrame, pssm2: pd.DataFrame) -> pd.Series:
    "Compute cosine similarity per position (column) between two PSSMs."
    
    assert pssm1.shape == pssm2.shape, "PSSMs must have the same shape"
    
    sims = {}
    for pos in pssm1.columns:
        v1 = pssm1[pos]
        v2 = pssm2[pos]
        v1,v2 = v1.align(v2, join='inner') # make sure the aa index match with each other

        norm1 = np.linalg.norm(v1)
        norm2 = np.linalg.norm(v2)

        if norm1 == 0 or norm2 == 0:
            sims[pos] = 0.0
        else:
            dot_product = sum(v1*v2) # np.dot(v1,v2)
            sims[pos] = dot_product / (norm1 * norm2)

    return pd.Series(sims)

# %% ../nbs/02_pssm.ipynb 174
def cosine_overall_flat(pssm1_flat, pssm2_flat):
    """Compute overall cosine similarity between two PSSMs (flattened)."""
    # match index for dot product
    pssm1_flat, pssm2_flat = pssm1_flat.align(pssm2_flat, join='inner')
    norm1 = np.linalg.norm(pssm1_flat)
    norm2 = np.linalg.norm(pssm2_flat)
    if norm1 == 0 or norm2 == 0: return 0.0
    dot_product = sum(pssm1_flat*pssm2_flat) # np.dot(pssm1_flat, pssm2_flat)
    return  dot_product/ (norm1 * norm2)
