# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/03_hierarchical.ipynb.

# %% auto 0
__all__ = ['get_1d_distance', 'get_1d_js', 'get_distance', 'get_1d_distance_parallel', 'get_1d_js_parallel', 'get_Z',
           'plot_dendrogram', 'pssm_to_seq', 'get_pssm_seq_labels']

# %% ../nbs/03_hierarchical.ipynb 3
import pandas as pd,numpy as np
from .data import *
from .pssm import *
from .plot import set_sns
from tqdm import tqdm
from functools import partial
from tqdm.contrib.concurrent import process_map
from fastcore.meta import delegates
from scipy.cluster.hierarchy import linkage, fcluster,dendrogram
from matplotlib import pyplot as plt

# %% ../nbs/03_hierarchical.ipynb 6
def get_1d_distance(df,func_flat):
    "Compute 1D distance for each row in a dataframe given a distance function "
    n = len(df)
    dist = []
    for i in tqdm(range(n)):
        for j in range(i+1, n):
            d = func_flat(df.iloc[i], df.iloc[j])
            dist.append(d)
    return np.array(dist)

# %% ../nbs/03_hierarchical.ipynb 8
def get_1d_js(df): 
    "Compute 1D distance using JS divergence."
    return get_1d_distance(df,js_divergence_flat)

# %% ../nbs/03_hierarchical.ipynb 11
def get_distance(pair, df, func):
    i, j = pair
    return func(df.iloc[i], df.iloc[j])

# %% ../nbs/03_hierarchical.ipynb 12
def get_1d_distance_parallel(df, func_flat, max_workers=4, chunksize=100):
    "Parallel compute 1D distance for each row in a dataframe given a distance function "
    n = len(df)
    index_pairs = [(i, j) for i in range(n) for j in range(i + 1, n)]

    bound_worker = partial(get_distance, df=df, func=func_flat)

    dist = process_map(bound_worker, index_pairs, max_workers=max_workers, chunksize=chunksize)
    return np.array(dist)

# %% ../nbs/03_hierarchical.ipynb 14
@delegates(get_1d_distance_parallel)
def get_1d_js_parallel(df, func_flat=js_divergence_flat, **kwargs): 
    "Compute 1D distance matrix using JS divergence."
    return get_1d_distance_parallel(df, func_flat=func_flat, **kwargs)

# %% ../nbs/03_hierarchical.ipynb 16
def get_Z(pssms,func_flat=js_divergence_flat,parallel=True):
    "Get linkage matrix Z from pssms dataframe"
    distance = get_1d_distance_parallel(pssms,func_flat=func_flat) if parallel else get_1d_distance(pssms,func_flat=func_flat)
    Z = linkage(distance, method='ward')
    return Z

# %% ../nbs/03_hierarchical.ipynb 19
def plot_dendrogram(Z,
                    color_thr=0.07,
                    dense=7, # the higher the more dense for each row
                    line_width=1,
                    title='Hierarchical Clustering Dendrogram',
                    scale=1,
                    **kwargs):
    length = (len(Z) + 1) // dense
    
    plt.figure(figsize=(5*scale,length*scale))
    with plt.rc_context({'lines.linewidth': line_width}):
        dendrogram(
            Z,
            orientation='left',
            leaf_font_size=7,
            color_threshold=color_thr,
            **kwargs
        )
    plt.title(title)
    plt.ylabel('Distance')
    # plt.savefig(output, bbox_inches='tight')
    # plt.close()
    ax = plt.gca()
    for spine in ['top', 'right', 'left', 'bottom']:
        ax.spines[spine].set_visible(False)

# %% ../nbs/03_hierarchical.ipynb 23
def pssm_to_seq(pssm_df, 
                thr=0.2, # threshold of probability to show in sequence
                clean_center=True, # if true, zero out non-last three values in position 0 (keep only s,t,y values at center)
                ):
    "Represent PSSM in string sequence of amino acids"
    
    pssm_df = pssm_df.copy()
    if clean_center:
        pssm_df.loc[pssm_df.index[:-3], 0] = 0  # keep only s,t,y in center 0 position

    pssm_df.index = pssm_df.index.map(lambda x: x.replace('pS', 's').replace('pT', 't').replace('pY', 'y'))

    consensus = []
    for i, col in enumerate(pssm_df.columns):
        # consider the case where sum for the position is 0
        column_vals = pssm_df[col]
        if column_vals.sum() == 0:
            symbol = '_'
        else:
            top = column_vals.nlargest(3)
            passing = [aa for aa, prob in zip(top.index, top.values) if prob > thr]

            if not passing:
                symbol = '.'
            elif len(passing) == 1:
                symbol = passing[0]
            else:
                symbol = f"[{'/'.join(passing)}]"
                
        if col == 0:  # center position
            if symbol.startswith('['):
                symbol = symbol[:-1] + ']*'
            else:
                symbol += '*'

        consensus.append(symbol)

    return ''.join(consensus)

# %% ../nbs/03_hierarchical.ipynb 26
def get_pssm_seq_labels(pssms,
                        count_map=None, # df index as key, counts as value
                        thr=0.3, # threshold of probability to show in sequence
                       ):
    "Use index of pssms and the pssm to seq to represent pssm."
    if count_map is not None:
        labels=[str(i)+f' (n={count_map[i]:,})' + ': '+pssm_to_seq(recover_pssm(r),thr=thr) for i,r in pssms.iterrows()]
    else:
        labels=[str(i)+ ': '+pssm_to_seq(recover_pssm(r),thr) for i,r in pssms.iterrows()]

    return labels
