"""Load various kinase-relatd datasets"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_data.ipynb.

# %% auto 0
__all__ = ['Data', 'CPTAC']

# %% ../nbs/00_data.ipynb 3
import pandas as pd
from functools import lru_cache
from fastcore.all import patch,patch_to

import gdown,zipfile,shutil,tempfile
from pathlib import Path

# %% ../nbs/00_data.ipynb 9
class Data:
    "A class for fetching various datasets."
    DATASET_DIR = Path(tempfile.gettempdir()) / 'katlas_dataset'

# %% ../nbs/00_data.ipynb 12
@patch_to(Data)
def download(download_dir=None, # dest directory of downloaded folder
             force=False, # if force, will overwrite the current dataset folder
             verbose=True, # print existing dataset folder
            ):
    "Download dataset zip and extract them in tmp folder if dataset_dir is not given."
    path = 'https://drive.google.com/uc?id=17wIl0DbdoHV036Z3xgaT_0H3LlM_W47l'
    if download_dir is not None: Data.DATASET_DIR=Path(download_dir)/'katlas_dataset'
    
    # 🧹 If old extracted folder exists, remove it (so we overwrite cleanly)
    if Data.DATASET_DIR.exists():
        if force:
            print(f"♻️ Removing existing folder: {Data.DATASET_DIR}")
            shutil.rmtree(Data.DATASET_DIR)
        else:
            if verbose: print(f"✅ Dataset exists at: {Data.DATASET_DIR}")
            return

    # ⬇️ Download zip (always fresh)
    print(f"⬇️ Downloading katlas_dataset.zip ...")
    downloaded_file = gdown.download(path)

    # 📦 Extract zip to folder
    print(f"📂 Extracting to {Data.DATASET_DIR} ...")
    with zipfile.ZipFile(downloaded_file, 'r') as zip_ref:
        zip_ref.extractall(Data.DATASET_DIR)
    # 🧹 Remove the zip after extraction
    try:
        print(f"🧹 Removing zip file: {downloaded_file}")
        Path(downloaded_file).unlink()
    except Exception as e:
        print(f"⚠️ Could not remove {downloaded_file}: {e}")

    print(f"✅ Done! Extracted dataset is at: {Data.DATASET_DIR}")

# %% ../nbs/00_data.ipynb 16
@patch_to(Data)
def read_file(rel_path):
    """
    Load a CSV or Parquet file from the local dataset folder.

    Automatically infers file type from the filename extension.
    Renames 'Unnamed: 0' column to 'kinase' if present.
    """
    Data.download(verbose=False)
    path = Data.DATASET_DIR / rel_path
    ext = path.suffix.lower()

    if ext == '.csv': df = pd.read_csv(path)
    elif ext == '.parquet': 
        # df = pd.read_parquet(path)
        try:
            df = pd.read_parquet(path, engine="fastparquet")
        except Exception:
            try:
                df = pd.read_parquet(path, engine="pyarrow")
            except Exception as e:
                print(f"Failed to read parquet file {path}: {e}")
                return None
    else: raise ValueError(f"❌ Unsupported file type: {ext}")

    if "Unnamed: 0" in df.columns:
        df = df.rename(columns={"Unnamed: 0": "kinase"})

    return df

# %% ../nbs/00_data.ipynb 19
@patch_to(Data)
def get_kinase_info():
    """
    Get information of 523 human kinases on kinome tree. 
    Group, family, and subfamily classifications are sourced from Coral; 
    full protein sequences are retrieved using UniProt IDs; 
    kinase domain sequences are obtained from KinaseDomain.com; 
    and cellular localization data is extracted from published literature.
    """
    return Data.read_file("kinase_info.csv")

# %% ../nbs/00_data.ipynb 21
@patch_to(Data)
def get_kinase_uniprot() -> pd.DataFrame:
    """
    Get information of 672 uniprot human kinases, which were retrieved from UniProt by filtering all human protein entries using the keyword 'kinase'. 
    It includes additional pseudokinases and lipid kinases.
    """
    path = "uniprot_human_keyword_kinase.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 23
@patch_to(Data)
def get_kd_uniprot():
    "Kinase domains extracted from UniProt database. "
    path = "uniprot_kd_labeled.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 26
@patch_to(Data)
@lru_cache
def get_pspa_tyr_norm():
    """Get PSPA normalized data of tyrosine kinase."""
    path = "PSPA/pspa_tyr_norm.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 28
@patch_to(Data)
@lru_cache
def get_pspa_st_norm():
    """Get PSPA normalized data of serine/threonine kinase."""
    path = "PSPA/pspa_st_norm.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 30
@patch_to(Data)
@lru_cache
def get_pspa_all_norm() -> pd.DataFrame:
    """Get PSPA normalized data of serine/threonine and tyrosine kinases."""
    path = "PSPA/pspa_all_norm.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 32
@patch_to(Data)
@lru_cache
def get_pspa_all_scale():
    """
    Get PSPA (-5 to +4) scaled data from PSPA normalized data. 
    Each position (including both pS/pT and pS=pT) are normalized to 1.
    """
    path = "PSPA/pspa_all_scale.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 34
@patch_to(Data)
@lru_cache
def get_pspa_st_pct():
    """Get PSPA reference score to calculate percentile for serine/threonine kinases."""
    path = "PSPA/pspa_pct_st.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 36
@patch_to(Data)
@lru_cache
def get_pspa_tyr_pct():
    """Get PSPA reference score to calculate percentile for tyrosine kinases."""
    path = "PSPA/pspa_pct_tyr.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 38
@patch_to(Data)
@lru_cache
def get_num_dict() -> dict:
    """Get a dictionary mapping kinase to number of random amino acids in PSPA."""
    path = "PSPA/pspa_divide_num.csv"
    return Data.read_file(path).set_index("kinase")["num_random_aa"].to_dict()

# %% ../nbs/00_data.ipynb 41
@patch_to(Data)
def get_ks_unique():
    """Get kinase substrate dataset with unique sub site ID."""
    path = "CDDM/unique_ks_sites.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 43
@patch_to(Data)
def get_ks_dataset(add_kinase_info=True):
    """
    Get kinase–substrate dataset collected from public resources,
    with the option of enriching with kinase info.
    """
    # --- 1️⃣ Load and preprocess dataset ---
    path = "CDDM/ks_datasets_20250407.parquet"
    df = Data.read_file(path)

    # Convert numeric-looking column names once
    df.columns = [
        int(c) if isinstance(c, str) and c.lstrip("-").isdigit() else c
        for c in df.columns
    ]

    if "substrate_phosphoseq" in df.columns:
        df["substrate_sequence"] = df["substrate_phosphoseq"].str.upper()

    if not add_kinase_info:
        return df

    # --- 2️⃣ Prepare kinase info (dedup, indexed maps) ---
    info = (
        Data.get_kinase_info()
        .sort_values("kinase")
        .drop_duplicates("uniprot")
        .set_index("uniprot")
    )

    # Extract clean UniProt IDs (remove isoforms)
    df["uniprot_clean"] = df["kinase_uniprot"].str.split("-", n=1).str[0]

    # Create mapping dicts once
    maps = {
        "kinase_group": info["modi_group"].to_dict(),
        "kinase_family": info["family"].to_dict(),
        "kinase_subfamily": info["subfamily"].to_dict(),
        "kinase_pspa_big": info["pspa_category_big"].to_dict(),
        "kinase_pspa_small": info["pspa_category_small"].to_dict(),
        "kinase_coral_ID": info["ID_coral"].to_dict(),
        "kinase_protein": info["kinase"].to_dict(),
    }

    # Preload gene name map once
    gene_map = Data.get_kinase_uniprot().set_index("Entry")["Gene Names"].to_dict()

    # --- 3️⃣ Vectorized assignment ---
    df["kinase_on_tree"] = df["uniprot_clean"].isin(info.index).astype(int)
    df["kinase_genes"] = df["uniprot_clean"].map(gene_map)

    # Assign all kinase attributes in a loop instead of repeated `.map()`
    for col, mapping in maps.items():
        df[col] = df["uniprot_clean"].map(mapping)

    # --- 4️⃣ Add num_kin from ks_unique ---
    site_info = Data.get_ks_unique()[["sub_site", "num_kin"]].set_index("sub_site")
    df["num_kin"] = df["sub_site"].map(site_info["num_kin"])

    # Clean up
    df.drop(columns="uniprot_clean", inplace=True)
    return df

# %% ../nbs/00_data.ipynb 45
@patch_to(Data)
@lru_cache
def get_ks_background():
    """Get kinase substrate dataset with unique sub site ID."""
    path = "CDDM/ks_background.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 47
@patch_to(Data)
@lru_cache
def get_cddm():
    """Get the CDDM dataset."""
    path = "CDDM/pssms.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 49
@patch_to(Data)
@lru_cache
def get_cddm_upper():
    """Get the CDDM dataset of all uppercase sequence."""
    path = "CDDM/pssms_upper.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 51
@patch_to(Data)
@lru_cache
def get_cddm_LO():
    """Get CDDM Log-odds data with 'STY' background."""
    path = "CDDM/pssms_LO.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 53
@patch_to(Data)
@lru_cache
def get_cddm_LO_upper():
    """Get CDDM Log-odds data of all-uppercase sequence with 'STY' background."""
    path = "CDDM/pssms_LO_upper.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 56
@patch_to(Data)
def get_aa_info():
    """Get amino acid information."""
    path = f"amino_acids/aa_info.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 58
@patch_to(Data)
def get_aa_rdkit():
    """Get RDKit representations of amino acids."""
    path = "amino_acids/aa_rdkit.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 60
@patch_to(Data)
def get_aa_morgan():
    """Get Morgan fingerprint representations of amino acids."""
    path = "amino_acids/aa_morgan.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 63
@patch_to(Data)
def get_cptac_ensembl_site():
    """Get CPTAC dataset with unique EnsemblProteinID+site."""
    path = "phosphosites/linkedOmicsKB_ref_pan.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 65
@patch_to(Data)
def get_cptac_unique_site():
    """Get CPTAC dataset with unique site sequences."""
    path = "phosphosites/cptac_unique_site.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 67
@patch_to(Data)
def get_cptac_gene_site():
    """Get CPTAC dataset with unique Gene+site."""
    path = "phosphosites/linkedOmics_ref_pan.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 69
@patch_to(Data)
def get_psp_human_site():
    """Get PhosphoSitePlus human dataset (Gene+site)."""
    path = "phosphosites/psp_human.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 71
@patch_to(Data)
def get_ochoa_site():
    """Get phosphoproteomics dataset from Ochoa et al."""
    path = "phosphosites/ochoa_site.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 73
@patch_to(Data)
def get_combine_site_psp_ochoa() -> pd.DataFrame:
    """
    Get the combined dataset from Ochoa and PhosphoSitePlus.
    """
    path = "phosphosites/combine_site_psp_ochoa.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 75
@patch_to(Data)
def get_combine_site_phosphorylated():
    """
    Get the combined phosphorylated dataset from Ochoa and PhosphoSitePlus.
    """
    path = "phosphosites/phosphorylated_combine_site.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 77
@patch_to(Data)
@lru_cache
def get_human_site():
    """
    Get the combined phosphorylated dataset from Ochoa and PhosphoSitePlus (20-length version).
    """
    path = "phosphosites/phosphorylated_combine_site20.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 80
@patch_to(Data)
@lru_cache
def get_reactome_pathway_lo() -> pd.DataFrame:
    """
    Get lowest reactome pathways with Uniprot ID as identifier.
    """
    path = "reactome_lowest_level.parquet"
    return Data.read_file(path)

# %% ../nbs/00_data.ipynb 82
@patch_to(Data)
@lru_cache
def get_reactome_pathway() -> pd.DataFrame:
    """
    Get all level reactome pathways with Uniprot ID as identifier.
    """
    path = "reactome_all_levels.parquet"
    path_all = Data.read_file(path)
    # path_lo = Data.get_reactome_pathway_lo()
    # path_all['lowest'] = path_all.reactome_id.isin(path_lo.reactome_id).astype(int)
    return path_all

# %% ../nbs/00_data.ipynb 86
class CPTAC:
    
    "A class for fetching CPTAC phosphoproteomics data."
    @staticmethod
    def _read_file(cancer: str, # cancer type CPTAC
                    is_Tumor: bool=True, # tumor tissue or normal
                    is_KB: bool=False, # whether it is for LinkedOmicsKB or LinkedOmics
                   ):
        "Fetches the data from the given path and returns a DataFrame"
        
        # path of ID and data
        sample_type = "Tumor" if is_Tumor else "Normal"
        ID_URL = f"https://zenodo.org/records/8196130/files/bcm-{cancer.lower()}-mapping-gencode.v34.basic.annotation-mapping.txt.gz"
        DATA_URL = f"https://cptac-pancancer-data.s3.us-west-2.amazonaws.com/data_freeze_v1.2_reorganized/{cancer.upper()}/{cancer.upper()}_phospho_site_abundance_log2_reference_intensity_normalized_{sample_type}.txt"

        # Load ID data
        ref = pd.read_csv(ID_URL, compression='gzip', sep='\t')[['protein','gene','gene_name']].drop_duplicates().reset_index(drop=True)
        
        # Load CPTAC phosphoproteomics data
        try:
            raw = pd.read_csv(DATA_URL, sep='\t')
        except Exception as e:
            print(f'{cancer} has {e}')
        else:
            info = pd.DataFrame({'gene':raw.idx.str.split('|').str[0],
                                 'site':raw.idx.str.split('|').str[2],
                                 'site_seq':raw.idx.str.split('|').str[3]})

            print(f'the {cancer} dataset length is: {info.shape[0]}')

            # Merge ensembl ID with gene name
            info = info.merge(ref,'left')
            print(f'after id mapping, the length is {info.shape[0]}')

            print(f'{info.gene_name.isna().sum()} sites does not have a mapped gene name')

            info['gene_site'] = info['gene_name'] + '_' + info['site']
            info['protein_site'] = info['protein'].str.split('.').str[0] + '_' + info['site']
            
            info = info.drop_duplicates(subset="protein_site" if is_KB else "gene_site").reset_index(drop=True)
            print(f'after removing duplicates of protein_site, the length is {info.shape[0]}')

            return info
    

# %% ../nbs/00_data.ipynb 87
@patch_to(CPTAC)
def list_cancer():
    "List available CPTAC cancer type"
    return ['HNSCC','GBM','COAD','CCRCC','LSCC','BRCA','UCEC','LUAD','PDAC','OV']

# %% ../nbs/00_data.ipynb 89
@patch_to(CPTAC)
def get_id(cancer_type: str,
           is_Tumor: bool=True, # tumor tissue or normal
           is_KB: bool=False, # whether it is for LinkedOmicsKB or LinkedOmics
          ):
    "Get CPTAC phosphorylation sites information given a cancer type"
    assert cancer_type in CPTAC.list_cancer(), "cancer type is not included, check available cancer types from CPTAC.list_cancer()"
    return CPTAC._read_file(cancer_type,is_Tumor, is_KB)
