"""Deep neural nets for PSSM"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/11_DNN.ipynb.

# %% auto 0
__all__ = ['def_device', 'seed_everything', 'GeneralDataset', 'MLP', 'init_weights', 'lin_wn', 'conv_wn', 'CNN1D', 'PSSM_model',
           'CE', 'KLD', 'JSD', 'train_dl', 'predict_dl', 'train_dl_cv']

# %% ../nbs/11_DNN.ipynb 3
import torch
from torch.utils.data import Dataset
import torch.nn.functional as F
import fastcore.all as fc

import numpy as np, pandas as pd
import os, random
from .data import *
from .train import *
from .pssm import *
from fastai.vision.all import *
import pandas as pd

# %% ../nbs/11_DNN.ipynb 5
def seed_everything(seed=123):
    random.seed(seed)
    os.environ['PYTHONHASHSEED'] = str(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False

# %% ../nbs/11_DNN.ipynb 7
def_device = 'mps' if torch.backends.mps.is_available() else 'cuda' if torch.cuda.is_available() else 'cpu'

# %% ../nbs/11_DNN.ipynb 18
class GeneralDataset(Dataset):
    def __init__(self,
                 df,
                 feat_col,            # list/Index of feature columns (e.g., 100 cols)
                 target_col=None,     # list/Index of flattened PSSM cols; AA-first; A=23
                 A: int = 23,
                 dtype=np.float32):
        """
        If target_col is None -> test mode, returns only X.
        Otherwise returns (X, y) where y has shape (23, L), L inferred from target columns.
        """
        self.test = target_col is None
        self.aa = A

        # Features
        self.X = df[feat_col].to_numpy(dtype=dtype, copy=True)

        self.y = None
        if not self.test:
            y_flat = df[target_col].to_numpy(dtype=dtype, copy=True)

            total = y_flat.shape[1]
            if total % A != 0:
                raise ValueError(f"Target columns ({total}) not divisible by A={A}; cannot infer L.")
            self.position = total // self.aa

            # AA-first: reshape to (N, 23, L)
            self.y = y_flat.reshape(-1, A, self.position) # reshape from row-major flatten
            # if column-major as pandas.unstack is column major
            # self.y = y_flat.reshape(-1, self.position,self.aa).transpose(0, 2, 1) 

        self.len = len(df)

    def __len__(self):
        return self.len

    def __getitem__(self, index):
        X = torch.from_numpy(self.X[index])        # (feat_dim,)
        if self.test: return X
        y = torch.from_numpy(self.y[index])        # (23, L)
        return X, y

# %% ../nbs/11_DNN.ipynb 25
def MLP(num_features, 
          num_targets,
          hidden_units = [512, 218],
          dp = 0.2):
    
    # Start with the first layer from num_features to the first hidden layer
    layers = [
        nn.Linear(num_features, hidden_units[0]),
        nn.BatchNorm1d(hidden_units[0]),
        # nn.Dropout(dp),
        nn.PReLU()
    ]
    
    # Loop over hidden units to create intermediate layers
    for i in range(len(hidden_units) - 1):
        layers.extend([
            nn.Linear(hidden_units[i], hidden_units[i+1]),
            nn.BatchNorm1d(hidden_units[i+1]),
            # nn.Dropout(dp),
            nn.PReLU()
        ])
    
    # Add the output layer
    layers.append(nn.Linear(hidden_units[-1], num_targets))
    
    model = nn.Sequential(*layers)
    
    return model

# %% ../nbs/11_DNN.ipynb 30
def init_weights(m, leaky=0.):
    "Initiate any Conv layer with Kaiming norm."
    if isinstance(m, (nn.Conv1d,nn.Conv2d,nn.Conv3d)): nn.init.kaiming_normal_(m.weight, a=leaky)

# %% ../nbs/11_DNN.ipynb 31
def lin_wn(ni,nf,dp=0.1,act=nn.SiLU):
    "Weight norm of linear."
    layers =  nn.Sequential(
            nn.BatchNorm1d(ni),
            nn.Dropout(dp),
            nn.utils.parametrizations.weight_norm(nn.Linear(ni, nf)) )
    if act: layers.append(act())
    return layers

# %% ../nbs/11_DNN.ipynb 32
def conv_wn(ni, nf, ks=3, stride=1, padding=1, dp=0.1,act=nn.ReLU):
    "Weight norm of conv."
    layers =  nn.Sequential(
        nn.BatchNorm1d(ni),
        nn.Dropout(dp),
        nn.utils.parametrizations.weight_norm(nn.Conv1d(ni, nf, ks, stride, padding)) )
    if act: layers.append(act())
    return layers

# %% ../nbs/11_DNN.ipynb 33
class CNN1D(nn.Module):
    
    def __init__(self, ni, nf, amp_scale = 16):
        super().__init__()

        cha_1,cha_2,cha_3 = 256,512,512
        hidden_size = cha_1*amp_scale

        cha_po_1 = hidden_size//(cha_1*2)
        cha_po_2 = (hidden_size//(cha_1*4)) * cha_3
        
        self.lin = lin_wn(ni,hidden_size)
        
        # bs, 256, 16
        self.view = View(-1,cha_1,amp_scale)
        
        self.conv1 = nn.Sequential(
            conv_wn(cha_1, cha_2, ks=5, stride=1, padding=2, dp=0.1),
            nn.AdaptiveAvgPool1d(output_size = cha_po_1),
            conv_wn(cha_2, cha_2, ks=3, stride=1, padding=1, dp=0.1))
        
        self.conv2 = nn.Sequential(
            conv_wn(cha_2, cha_2, ks=3, stride=1, padding=1, dp=0.3),
            conv_wn(cha_2, cha_3, ks=5, stride=1, padding=2, dp=0.2))
        
        self.head = nn.Sequential(
            nn.MaxPool1d(kernel_size=4, stride=2, padding=1),
            nn.Flatten(),
            lin_wn(cha_po_2,nf,act=None) )


    def forward(self, x):
        # amplify features to 4096
        x = self.lin(x)
        
        # reshape to bs,256,16 for conv1d
        x = self.view(x) 

        x = self.conv1(x)
        
        x_s = x  # for skip connection (multiply)
        x = self.conv2(x)
        x = x * x_s

        # Final block
        x = self.head(x)

        return x

# %% ../nbs/11_DNN.ipynb 37
class PSSM_model(nn.Module):
    def __init__(self, 
                 n_features,
                 n_targets,
                 model='MLP'):
        super().__init__()
        self.n_features=n_features
        self.n_targets=n_targets
        self.n_aa = 23
        if self.n_targets % self.n_aa != 0: raise ValueError(f"n_targets ({n_targets}) must be divisible by n_aa ({n_aa}).")
        self.n_positions = self.n_targets//self.n_aa
        
        if model =='MLP': self.model=MLP(self.n_features, self.n_targets)
        elif model =='CNN': self.model=CNN1D(self.n_features, self.n_targets).apply(init_weights)
        else: raise ValueError('model must be MLP or CNN.')
    def forward(self,x):
        logits = self.model(x).reshape(-1, self.n_aa,self.n_positions)
        return logits

# %% ../nbs/11_DNN.ipynb 43
def CE(logits: torch.Tensor,
       target_probs: torch.Tensor,
      ):
    """
    Cross-entropy with soft labels.
    logits:       (B, 20, 10)
    target_probs: (B, 20, 10), each column (over AA) sums to 1
    """
    logp = F.log_softmax(logits, dim=1)              # (B, 20, 10)
    ce   = -(target_probs * logp).sum(dim=1)         # (B, 10)
    return ce.mean()

# %% ../nbs/11_DNN.ipynb 46
def KLD(logits: torch.Tensor,
          target_probs: torch.Tensor,
         ):
    """
    KL divergence between target_probs (p) and softmax(logits) (q).
    
    logits:       (B, 20, 10)
    target_probs: (B, 20, 10), each column (over AA) sums to 1
    """
    logq = F.log_softmax(logits, dim=1)    # log q(x)
    logp = torch.log(target_probs + 1e-12) # log p(x), safe for zeros
    kl   = (target_probs * (logp - logq)).sum(dim=1)   # (B, 10)
    return kl.mean()

# %% ../nbs/11_DNN.ipynb 48
def JSD(logits: torch.Tensor,
        target_probs: torch.Tensor,
       ):
    """
    Jensen-Shannon Divergence between target_probs (p) and softmax(logits) (q).

    logits:       (B, 20, 10)
    target_probs: (B, 20, 10), each column (over AA) sums to 1
    """
    # p, q distributions
    q = F.softmax(logits, dim=1)                # q(x)
    p = target_probs
    m = 0.5 * (p + q)                           # midpoint distribution

    # logs (with epsilon for stability)
    logp = torch.log(p + 1e-12)
    logq = torch.log(q + 1e-12)
    logm = torch.log(m + 1e-12)

    # KL(p||m) and KL(q||m)
    kld_pm = (p * (logp - logm)).sum(dim=1)
    kld_qm = (q * (logq - logm)).sum(dim=1)

    jsd = 0.5 * (kld_pm + kld_qm)               # (B, 10)
    return jsd.mean()

# %% ../nbs/11_DNN.ipynb 51
def train_dl(df, 
            feat_col, 
            target_col,
            split, # tuple of numpy array for split index
            model_func, # function to get pytorch model
             n_epoch = 4, # number of epochs
             bs = 32, # batch size
             lr = 1e-2, # will be useless if lr_find is True
            loss = CE, # loss function
            save = None, # models/{save}.pth
             sampler = None,
             lr_find=False, # if true, will use lr from lr_find
              ):
    "A DL trainer."
    
    train = df.loc[split[0]]
    valid = df.loc[split[1]]
    
    train_ds = GeneralDataset(train, feat_col, target_col)
    valid_ds = GeneralDataset(valid, feat_col, target_col)
    
    n_workers = fc.defaults.cpus

    dls = DataLoaders.from_dsets(train_ds, valid_ds, bs=bs, num_workers=n_workers)

    model = model_func()
    learn = Learner(dls.to(def_device), model.to(def_device), loss, 
                    metrics= [KLD,JSD]
                    # cbs = [GradientClip(1.0)] # prevent overfitting
                   )
    
    if lr_find:
        # get learning rate
        lr = learn.lr_find()
        plt.show()
        plt.close()
        print(lr)

        
    print('lr in training is', lr)
    learn.fit_one_cycle(n_epoch,lr) #cbs = [SaveModelCallback(fname = 'best')] # save best model
    
    if save is not None:
        learn.save(save)
        
    pred,target = learn.get_preds()

    # row first
    pred  = F.softmax(pred, dim=1).reshape(len(valid),-1)
    target = target.reshape(len(valid),-1)

    # column first
    # pred  = F.softmax(pred, dim=1).permute(0, 2, 1).reshape(len(valid),-1)
    # target = target.permute(0, 2, 1).reshape(len(valid),-1)
    
    pred = pd.DataFrame(pred.detach().cpu().numpy(),index=valid.index,columns=target_col)
    target = pd.DataFrame(target.detach().cpu().numpy(),index=valid.index,columns=target_col)
    
    return target, pred

# %% ../nbs/11_DNN.ipynb 56
def predict_dl(df, 
               feat_col, 
               target_col,
               model_func, # model architecture
               model_pth, # only name, not with .pth
              ):
    
    "Predict dataframe given a deep learning model"
    
    test_dset = GeneralDataset(df,feat_col)
    test_dl = DataLoader(test_dset,bs=512)
    
    model = model_func()
    
    learn = Learner(None, model.to(def_device), loss_func=1)
    learn.load(model_pth,weights_only=False)
    
    learn.model.eval()
    
    preds = []
    for data in test_dl:
        inputs = data.to(def_device)
        pred = learn.model(inputs)

        pred  = F.softmax(pred, dim=1).reshape(len(pred),-1)
        # pred  = F.softmax(pred, dim=1).permute(0, 2, 1).reshape(len(pred),-1)

        preds.append(pred.detach().cpu().numpy())

    preds = np.concatenate(preds)
    preds = pd.DataFrame(preds,index=df.index,columns=target_col)

    return preds

# %% ../nbs/11_DNN.ipynb 63
def train_dl_cv(df, 
                feat_col, 
                target_col, 
                splits, # list of tuples
                model_func, # functions like lambda x: return MLP_1(num_feat, num_target)
                save:str=None,
                **kwargs
                ):
    
    OOF = []
    
    for fold,split in enumerate(splits):

        print(f'------fold{fold}------')
        
        fname = f'{save}_fold{fold}' if save is not None else None
        
        
        # train model
        target, pred = train_dl(df,feat_col,target_col, split, model_func ,save=fname,**kwargs)

        pred['nfold'] = fold
        OOF.append(pred)
        

    # Concatenate OOF from each fold to a new dataframe
    oofs = pd.concat(OOF).sort_index()
    
    return oofs
