"""A collection of machine learning tools"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/10_ML.ipynb.

# %% auto 0
__all__ = ['get_score_jsd', 'get_score_kld', 'get_score_ce', 'get_splits', 'split_data', 'train_ml', 'train_ml_cv',
           'post_process', 'post_process_oof', 'get_score', 'calculate_ce', 'predict_ml']

# %% ../nbs/10_ML.ipynb 3
# katlas
from .data import Data
from .pssm import *
# from katlas.feature import *
from .clustering import *
from functools import partial

# essentials
import pandas as pd, numpy as np
from joblib import dump, load
import math,matplotlib.pyplot as plt
from pathlib import Path

# scipy
from scipy.stats import spearmanr, pearsonr
from scipy.stats import spearmanr,pearsonr

# sklearn
from sklearn.model_selection import *
from sklearn.metrics import mean_squared_error
from sklearn.linear_model import *
from sklearn.svm import *
from sklearn.ensemble import *

from sklearn import set_config
set_config(transform_output="pandas")

# %% ../nbs/10_ML.ipynb 5
def get_splits(df: pd.DataFrame, # df contains info for split
               stratified: str=None, # colname to make stratified kfold; sampling from different groups
               group: str=None, # colname to make group kfold; test and train are from different groups
               nfold: int=5,
               seed: int=123):
    
    "Split samples in a dataframe based on Stratified, Group, or StratifiedGroup Kfold method"
    def _log(colname):
        print(kf)
        split=splits[0]
        print(f'# {colname} in train set: {df.loc[split[0]][colname].unique().shape[0]}')
        print(f'# {colname} in test set: {df.loc[split[1]][colname].unique().shape[0]}')
        
    splits = []
    if stratified is not None and group is None:
        kf = StratifiedKFold(nfold, shuffle=True, random_state=seed)
        for split in kf.split(df.index, df[stratified]):
            splits.append(split)
            
        _log(stratified)
        
    elif group is not None and stratified is None:
        kf = GroupKFold(nfold)
        for split in kf.split(df.index, groups=df[group]):
            splits.append(split)
            
        _log(group)
        
    elif stratified is not None and group is not None:
        kf = StratifiedGroupKFold(nfold, shuffle=True, random_state=seed)
        for split in kf.split(df.index, groups=df[group], y=df[stratified]):
            splits.append(split)
            
        _log(stratified)

    else:
        raise ValueError("Either 'stratified' or 'group' argument must be provided.")
    
    return splits

# %% ../nbs/10_ML.ipynb 14
def split_data(df: pd.DataFrame, # dataframe of values
               feat_col: list, # feature columns
               target_col: list, # target columns
               split: tuple # one of the split in splits
               ):
    "Given split tuple, split dataframe into X_train, y_train, X_test, y_test"
    
    X_train = df.loc[split[0]][feat_col]
    y_train = df.loc[split[0]][target_col]
    
    X_test = df.loc[split[1]][feat_col]
    y_test = df.loc[split[1]][target_col]
    
    return X_train, y_train, X_test, y_test

# %% ../nbs/10_ML.ipynb 18
def train_ml(df, # dataframe of values
             feat_col, # feature columns
             target_col, # target columns
             split, # one split in splits
             model,  # a sklearn models
             save = None, # file (.joblib) to save, e.g. 'model.joblib'
             params={}, # parameters for model.fit from sklearn
            ):
    
    "Fit and predict using sklearn model format, return target and pred of valid dataset."
    
    # split data
    X_train, y_train, X_test, y_test = split_data(df, feat_col, target_col, split)
    
    # Fit the model
    model.fit(X_train, y_train, **params) # better convert y_train to numpy array and flatten
    
    if save is not None:
        # Save the model to a file
        # joblib.dump(model, save)
        dump(model, save)
        
    # Predict train
    y_train_pred = model.predict(X_train) # X_test is dataframe, y_pred is numpy array
    
    # Predict test
    y_pred = model.predict(X_test) # X_test is dataframe, y_pred is numpy array

    # Make dataframe
    y_pred = pd.DataFrame(y_pred,index=y_test.index, columns = y_test.columns)
    
    return y_test, y_pred

# %% ../nbs/10_ML.ipynb 21
def train_ml_cv( df, # dataframe of values
                 feat_col, # feature columns
                 target_col,  # target columns
                 splits, # splits
                 model, # sklearn model
                 save = None, # model name to be saved, e.g., 'LR'
                 params = {}, # act as kwargs, for model.fit
                ):
    
    "Cross-validation through the given splits"
    
    OOF = []
    
    for fold, split in enumerate(splits):
        # print(f'------ fold: {fold} --------')
        
        if save is not None: 
            save = f'models/{save}_{fold}.joblib'
            
        target, pred = train_ml(df, feat_col, target_col, split, model,save,params=params)
        
        pred['nfold'] = fold
        OOF.append(pred)
        
    # Concatenate OOF from each fold to a new dataframe
    oof = pd.concat(OOF).sort_index()
    
    
    return oof

# %% ../nbs/10_ML.ipynb 24
def post_process(pssm_df):
    "Convert neg value to 0, clean non-last three values in position zero, and normalize each position"
    pssm = pssm_df.copy()
    pssm = pssm.clip(lower=0)
    return clean_zero_normalize(pssm)

# %% ../nbs/10_ML.ipynb 27
def post_process_oof(oof_ml,target_col):
    oof = oof_ml.copy()
    oof[target_col] = oof.apply(lambda r: pd.Series(flatten_pssm(post_process(recover_pssm(r[target_col])),column_wise=False)), axis=1)
    return oof

# %% ../nbs/10_ML.ipynb 29
def get_score(target,pred,func):
    distance = [func(target.loc[i],pred.loc[i,target.columns]) for i in target.index]
    return pd.Series(distance,index=target.index)

# %% ../nbs/10_ML.ipynb 30
get_score_jsd = partial(get_score,func=js_divergence_flat)

# %% ../nbs/10_ML.ipynb 31
get_score_kld = partial(get_score,func=kl_divergence_flat)

# %% ../nbs/10_ML.ipynb 36
def calculate_ce(target_series,pred_series):
    return float((-(np.log(recover_pssm(pred_series+EPSILON))*(recover_pssm(target_series))).sum()).mean())

# %% ../nbs/10_ML.ipynb 37
get_score_ce = partial(get_score,func=calculate_ce)

# %% ../nbs/10_ML.ipynb 43
def predict_ml(df, # Dataframe that contains features
               feat_col, # feature columns
               target_col=None,
               model_pth = 'model.joblib'
              ):
    
    "Make predictions based on trained model."
    
    test = df[feat_col]
    
    model = load(model_pth)
    
    pred = model.predict(test)
    
    pred_df = pd.DataFrame(pred,index=df.index,columns=target_col)
    
    return pred_df
