/*
 * Decompiled with CFR 0.152.
 */
package com.unboundid.asn1;

import com.unboundid.asn1.ASN1Element;
import com.unboundid.asn1.ASN1Exception;
import com.unboundid.asn1.ASN1Messages;
import com.unboundid.asn1.ASN1StreamReaderSequence;
import com.unboundid.asn1.ASN1StreamReaderSet;
import com.unboundid.util.Debug;
import com.unboundid.util.Mutable;
import com.unboundid.util.StaticUtils;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.SocketTimeoutException;
import java.util.logging.Level;

@Mutable
@ThreadSafety(level=ThreadSafetyLevel.NOT_THREADSAFE)
public final class ASN1StreamReader {
    private boolean ignoreInitialSocketTimeout;
    private boolean ignoreSubsequentSocketTimeout;
    private final InputStream inputStream;
    private final int maxElementSize;
    private long totalBytesRead;

    public ASN1StreamReader(InputStream inputStream) {
        this(inputStream, Integer.MAX_VALUE);
    }

    public ASN1StreamReader(InputStream inputStream, int maxElementSize) {
        this.inputStream = inputStream.markSupported() ? inputStream : new BufferedInputStream(inputStream);
        this.maxElementSize = maxElementSize > 0 ? maxElementSize : Integer.MAX_VALUE;
        this.totalBytesRead = 0L;
        this.ignoreInitialSocketTimeout = false;
        this.ignoreSubsequentSocketTimeout = false;
    }

    public void close() throws IOException {
        this.inputStream.close();
    }

    long getTotalBytesRead() {
        return this.totalBytesRead;
    }

    @Deprecated
    public boolean ignoreSocketTimeoutException() {
        return this.ignoreInitialSocketTimeout;
    }

    public boolean ignoreInitialSocketTimeoutException() {
        return this.ignoreInitialSocketTimeout;
    }

    public boolean ignoreSubsequentSocketTimeoutException() {
        return this.ignoreSubsequentSocketTimeout;
    }

    @Deprecated
    public void setIgnoreSocketTimeout(boolean ignoreSocketTimeout) {
        this.ignoreInitialSocketTimeout = ignoreSocketTimeout;
        this.ignoreSubsequentSocketTimeout = ignoreSocketTimeout;
    }

    public void setIgnoreSocketTimeout(boolean ignoreInitialSocketTimeout, boolean ignoreSubsequentSocketTimeout) {
        this.ignoreInitialSocketTimeout = ignoreInitialSocketTimeout;
        this.ignoreSubsequentSocketTimeout = ignoreSubsequentSocketTimeout;
    }

    public int peek() throws IOException {
        this.inputStream.mark(1);
        int byteRead = this.read(true);
        this.inputStream.reset();
        return byteRead;
    }

    private int readType() throws IOException {
        int typeInt = this.read(true);
        if (typeInt < 0) {
            this.close();
        } else {
            ++this.totalBytesRead;
        }
        return typeInt;
    }

    private int readLength() throws IOException {
        int length = this.read(false);
        if (length < 0) {
            throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_FIRST_LENGTH.get());
        }
        ++this.totalBytesRead;
        if (length > 127) {
            int numLengthBytes = length & 0x7F;
            length = 0;
            if (numLengthBytes < 1 || numLengthBytes > 4) {
                throw new IOException(ASN1Messages.ERR_READ_LENGTH_TOO_LONG.get(numLengthBytes));
            }
            for (int i = 0; i < numLengthBytes; ++i) {
                int lengthInt = this.read(false);
                if (lengthInt < 0) {
                    throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_LENGTH_END.get());
                }
                length <<= 8;
                length |= lengthInt & 0xFF;
            }
            this.totalBytesRead += (long)numLengthBytes;
        }
        if (length < 0 || this.maxElementSize > 0 && length > this.maxElementSize) {
            throw new IOException(ASN1Messages.ERR_READ_LENGTH_EXCEEDS_MAX.get(length, this.maxElementSize));
        }
        return length;
    }

    private void skip(int numBytes) throws IOException {
        if (numBytes <= 0) {
            return;
        }
        long totalBytesSkipped = this.inputStream.skip(numBytes);
        while (totalBytesSkipped < (long)numBytes) {
            long bytesSkipped = this.inputStream.skip((long)numBytes - totalBytesSkipped);
            if (bytesSkipped <= 0L) {
                while (totalBytesSkipped < (long)numBytes) {
                    int byteRead = this.read(false);
                    if (byteRead < 0) {
                        throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
                    }
                    ++totalBytesSkipped;
                }
                continue;
            }
            totalBytesSkipped += bytesSkipped;
        }
        this.totalBytesRead += (long)numBytes;
    }

    public ASN1Element readElement() throws IOException {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        int valueBytesRead = 0;
        int bytesRemaining = length;
        byte[] value = new byte[length];
        while (valueBytesRead < length) {
            int bytesRead = this.read(false, value, valueBytesRead, bytesRemaining);
            if (bytesRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            valueBytesRead += bytesRead;
            bytesRemaining -= bytesRead;
        }
        this.totalBytesRead += (long)length;
        ASN1Element e = new ASN1Element((byte)type, value);
        Debug.debugASN1Read(e);
        return e;
    }

    public Boolean readBoolean() throws IOException, ASN1Exception {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        if (length == 1) {
            int value = this.read(false);
            if (value < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            ++this.totalBytesRead;
            return value != 0;
        }
        this.skip(length);
        throw new ASN1Exception(ASN1Messages.ERR_BOOLEAN_INVALID_LENGTH.get());
    }

    public Integer readEnumerated() throws IOException, ASN1Exception {
        return this.readInteger();
    }

    public Integer readInteger() throws IOException, ASN1Exception {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        if (length == 0 || length > 4) {
            this.skip(length);
            throw new ASN1Exception(ASN1Messages.ERR_INTEGER_INVALID_LENGTH.get(length));
        }
        boolean negative = false;
        int intValue = 0;
        for (int i = 0; i < length; ++i) {
            int byteRead = this.read(false);
            if (byteRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            if (i == 0) {
                negative = (byteRead & 0x80) != 0;
            }
            intValue <<= 8;
            intValue |= byteRead & 0xFF;
        }
        if (negative) {
            switch (length) {
                case 1: {
                    intValue |= 0xFFFFFF00;
                    break;
                }
                case 2: {
                    intValue |= 0xFFFF0000;
                    break;
                }
                case 3: {
                    intValue |= 0xFF000000;
                }
            }
        }
        this.totalBytesRead += (long)length;
        return intValue;
    }

    public Long readLong() throws IOException, ASN1Exception {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        if (length == 0 || length > 8) {
            this.skip(length);
            throw new ASN1Exception(ASN1Messages.ERR_LONG_INVALID_LENGTH.get(length));
        }
        boolean negative = false;
        long longValue = 0L;
        for (int i = 0; i < length; ++i) {
            int byteRead = this.read(false);
            if (byteRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            if (i == 0) {
                negative = (byteRead & 0x80) != 0;
            }
            longValue <<= 8;
            longValue |= (long)byteRead & 0xFFL;
        }
        if (negative) {
            switch (length) {
                case 1: {
                    longValue |= 0xFFFFFFFFFFFFFF00L;
                    break;
                }
                case 2: {
                    longValue |= 0xFFFFFFFFFFFF0000L;
                    break;
                }
                case 3: {
                    longValue |= 0xFFFFFFFFFF000000L;
                    break;
                }
                case 4: {
                    longValue |= 0xFFFFFFFF00000000L;
                    break;
                }
                case 5: {
                    longValue |= 0xFFFFFF0000000000L;
                    break;
                }
                case 6: {
                    longValue |= 0xFFFF000000000000L;
                    break;
                }
                case 7: {
                    longValue |= 0xFF00000000000000L;
                }
            }
        }
        this.totalBytesRead += (long)length;
        return longValue;
    }

    public void readNull() throws IOException, ASN1Exception {
        int type = this.readType();
        if (type < 0) {
            return;
        }
        int length = this.readLength();
        if (length != 0) {
            this.skip(length);
            throw new ASN1Exception(ASN1Messages.ERR_NULL_HAS_VALUE.get());
        }
    }

    public byte[] readBytes() throws IOException {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        int valueBytesRead = 0;
        int bytesRemaining = length;
        byte[] value = new byte[length];
        while (valueBytesRead < length) {
            int bytesRead = this.read(false, value, valueBytesRead, bytesRemaining);
            if (bytesRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            valueBytesRead += bytesRead;
            bytesRemaining -= bytesRead;
        }
        this.totalBytesRead += (long)length;
        return value;
    }

    public String readString() throws IOException {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        int valueBytesRead = 0;
        int bytesRemaining = length;
        byte[] value = new byte[length];
        while (valueBytesRead < length) {
            int bytesRead = this.read(false, value, valueBytesRead, bytesRemaining);
            if (bytesRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            valueBytesRead += bytesRead;
            bytesRemaining -= bytesRead;
        }
        this.totalBytesRead += (long)length;
        return StaticUtils.toUTF8String(value);
    }

    public ASN1StreamReaderSequence beginSequence() throws IOException {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        return new ASN1StreamReaderSequence(this, (byte)type, length);
    }

    public ASN1StreamReaderSet beginSet() throws IOException {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        return new ASN1StreamReaderSet(this, (byte)type, length);
    }

    private int read(boolean initial) throws IOException {
        try {
            return this.inputStream.read();
        }
        catch (SocketTimeoutException ste) {
            Debug.debugException(Level.FINEST, ste);
            if (initial && this.ignoreInitialSocketTimeout || !initial && this.ignoreSubsequentSocketTimeout) {
                while (true) {
                    try {
                        return this.inputStream.read();
                    }
                    catch (SocketTimeoutException ste2) {
                        Debug.debugException(Level.FINEST, ste2);
                        continue;
                    }
                    break;
                }
            }
            throw ste;
        }
    }

    private int read(boolean initial, byte[] buffer, int offset, int length) throws IOException {
        try {
            return this.inputStream.read(buffer, offset, length);
        }
        catch (SocketTimeoutException ste) {
            Debug.debugException(Level.FINEST, ste);
            if (initial && this.ignoreInitialSocketTimeout || !initial && this.ignoreSubsequentSocketTimeout) {
                while (true) {
                    try {
                        return this.inputStream.read(buffer, offset, length);
                    }
                    catch (SocketTimeoutException ste2) {
                        Debug.debugException(Level.FINEST, ste2);
                        continue;
                    }
                    break;
                }
            }
            throw ste;
        }
    }
}

