#!/usr/bin/env python3

from __future__ import absolute_import
import os
from fcntl import ioctl
from ctypes import Structure, CFUNCTYPE, string_at, POINTER, byref
from ctypes import c_int, c_char_p, c_void_p, c_uint, c_bool, c_double
from ctypes import c_uint32, c_uint64, c_float
from ctypes import sizeof
from .constant import TabletToolType


# ~ _IOCPARM_MASK = 0x1FFF
# ~ _IOC_IN = 0x40000000


# ~ def _IOW(magic, command, type_):

	# ~ return (
		# ~ (magic << 8)
		# ~ | command
		# ~ | ((sizeof(type_) & _IOCPARM_MASK) << 16)
		# ~ | _IOC_IN)


# ~ EVIOCGRAB = _IOW(ord('E'), 0x90, c_int)


class Interface(object):

	class Interface(Structure):

		_fields_ = (
			('open_restricted', CFUNCTYPE(c_int, c_char_p, c_int, c_void_p)),
			('close_restricted', CFUNCTYPE(None, c_int, c_void_p)))

	def __new__(self):

		return self.Interface(self.open_restricted(), self.close_restricted())

	@classmethod
	def open_restricted(cls):

		def open_restricted(path, flags, user_data):

			fd = os.open(path, flags)
			return fd

		CMPFUNC = CFUNCTYPE(c_int, c_char_p, c_int, c_void_p)
		return CMPFUNC(open_restricted)

	@classmethod
	def close_restricted(cls):

		def close_restricted(fd, user_data):

			os.close(fd)

		CMPFUNC = CFUNCTYPE(None, c_int, c_void_p)
		return CMPFUNC(close_restricted)


class TabletTool(object):
	"""An object representing a tool being used by a device with
	the :attr:`~libinput.constant.DeviceCapability.TABLET_TOOL` capability.

	Tablet events generated by such a device are bound to a specific tool
	rather than coming from the device directly. Depending on the hardware
	it is possible to track the same physical tool across multiple
	:class:`~libinput.device.Device` instances, see `Tracking unique tools`_.

	As long as a reference to a :class:`.TabletTool` is kept, multiple
	instances will compare equal if they refer to the same physical tool and
	the hardware supports it.
	"""

	def __init__(self, htablettool, libinput):

		self._handle = htablettool
		self._libinput = libinput

		self._libinput.libinput_tablet_tool_ref.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_ref.restype = c_void_p
		self._libinput.libinput_tablet_tool_unref.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_unref.restype = c_void_p
		self._libinput.libinput_tablet_tool_get_type.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_get_type.restype = TabletToolType
		self._libinput.libinput_tablet_tool_get_tool_id.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_get_tool_id.restype = c_uint64
		self._libinput.libinput_tablet_tool_has_pressure.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_has_pressure.restype = c_bool
		self._libinput.libinput_tablet_tool_has_distance.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_has_distance.restype = c_bool
		self._libinput.libinput_tablet_tool_has_tilt.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_has_tilt.restype = c_bool
		self._libinput.libinput_tablet_tool_has_rotation.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_has_rotation.restype = c_bool
		self._libinput.libinput_tablet_tool_has_slider.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_has_slider.restype = c_bool
		self._libinput.libinput_tablet_tool_has_wheel.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_has_wheel.restype = c_bool
		self._libinput.libinput_tablet_tool_has_button.argtypes = (
			c_void_p, c_uint32)
		self._libinput.libinput_tablet_tool_has_button.restype = c_bool
		self._libinput.libinput_tablet_tool_is_unique.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_is_unique.restype = c_bool
		self._libinput.libinput_tablet_tool_get_serial.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_tool_get_serial.restype = c_uint64

		self._libinput.libinput_tablet_tool_ref(self._handle)

	def __del__(self):

		self._libinput.libinput_tablet_tool_unref(self._handle)

	def __eq__(self, other):

		if isinstance(other, type(self)):
			return self._handle == other._handle
		else:
			return NotImplemented

	@property
	def type(self):
		"""The tool type of a tool object.

		See `Vendor-specific tablet tool types`_ for details.

		Returns:
			~libinput.constant.TabletToolType: The tool type for this tool
			object.
		"""

		return self._libinput.libinput_tablet_tool_get_type(self._handle)

	@property
	def tool_id(self):
		"""The tool ID of a tool object.

		If nonzero, this number identifies the specific type of the tool with
		more precision than the type returned in :attr:`type`,
		see `Vendor-specific tablet tool types`_. Not all tablets support
		a tool ID.

		Tablets known to support tool IDs include the Wacom Intuos 3, 4, 5,
		Wacom Cintiq and Wacom Intuos Pro series.

		Returns:
			int: The tool ID for this tool object or 0 if none is provided.
		"""

		return self._libinput.libinput_tablet_tool_get_tool_id(self._handle)

	def has_pressure(self):
		"""Return whether the tablet tool supports pressure.

		Returns:
			bool: :obj:`True` if the axis is available, :obj:`False` otherwise.
		"""

		return self._libinput.libinput_tablet_tool_has_pressure(self._handle)

	def has_distance(self):
		"""Return whether the tablet tool supports distance.

		Returns:
			bool: :obj:`True` if the axis is available, :obj:`False` otherwise.
		"""

		return self._libinput.libinput_tablet_tool_has_distance(self._handle)

	def has_tilt(self):
		"""Return whether the tablet tool supports tilt.

		Returns:
			bool: :obj:`True` if the axis is available, :obj:`False` otherwise.
		"""

		return self._libinput.libinput_tablet_tool_has_tilt(self._handle)

	def has_rotation(self):
		"""Return whether the tablet tool supports z-rotation.

		Returns:
			bool: :obj:`True` if the axis is available, :obj:`False` otherwise.
		"""

		return self._libinput.libinput_tablet_tool_has_rotation(self._handle)

	def has_slider(self):
		"""Return whether the tablet tool has a slider axis.

		Returns:
			bool: :obj:`True` if the axis is available, :obj:`False` otherwise.
		"""

		return self._libinput.libinput_tablet_tool_has_slider(self._handle)

	def has_wheel(self):
		"""Return whether the tablet tool has a relative wheel.

		Returns:
			bool: :obj:`True` if the axis is available, :obj:`False` otherwise.
		"""

		return self._libinput.libinput_tablet_tool_has_wheel(self._handle)

	def has_button(self, button):
		"""Check if a tablet tool has a specified button.

		Args:
			button (int): Button to check for. See ``input.h``.
		Returns:
			bool: :obj:`True` if the tool supports this button, :obj:`False`
			if it does not.
		"""

		return self._libinput.libinput_tablet_tool_has_button(
			self._handle, button)

	def is_unique(self):
		"""Return :obj:`True` if the physical tool can be uniquely identified
		by libinput, or :obj:`False` otherwise.

		If a tool can be uniquely identified, keeping a reference to the tool
		allows tracking the tool across proximity out sequences and across
		compatible tablets. See `Tracking unique tools`_ for more details.

		Returns:
			bool: :obj:`True` if the tool can be uniquely identified,
			:obj:`False` otherwise.
		"""

		return self._libinput.libinput_tablet_tool_is_unique(self._handle)

	@property
	def serial(self):
		"""The serial number of a tool.

		If the tool does not report a serial number, this method returns zero.
		See `Tracking unique tools`_ for details.

		Returns:
			int: The tool serial number.
		"""

		return self._libinput.libinput_tablet_tool_get_serial(self._handle)


class TabletPadModeGroup(object):
	"""A mode on a tablet pad is a virtual grouping of functionality, usually
	based on some visual feedback like LEDs on the pad.

	The set of buttons, rings and strips that share the same mode are
	a "mode group". Whenever the mode changes, all buttons, rings and strips
	within this mode group are affected. See `Tablet pad modes`_ for detail.

	Most tablets only have a single mode group, some tablets provide multiple
	mode groups through independent banks of LEDs (e.g. the Wacom Cintiq 24HD).
	libinput guarantees that at least one mode group is always available.
	"""

	def __init__(self, hmodegroup, libinput):

		self._handle = hmodegroup
		self._libinput = libinput

		self._libinput.libinput_tablet_pad_mode_group_ref.argtypes = (c_void_p,)
		self._libinput.libinput_tablet_pad_mode_group_ref.restype = c_void_p
		self._libinput.libinput_tablet_pad_mode_group_unref.argtypes = (
			c_void_p,)
		self._libinput.libinput_tablet_pad_mode_group_unref.restype = c_void_p
		self._libinput.libinput_tablet_pad_mode_group_get_index.argtypes = (
			c_void_p,)
		self._libinput.libinput_tablet_pad_mode_group_get_index.restype = (
			c_uint)
		self._libinput.libinput_tablet_pad_mode_group_get_num_modes.argtypes = (
			c_void_p,)
		self._libinput.libinput_tablet_pad_mode_group_get_num_modes.restype = (
			c_uint)
		self._libinput.libinput_tablet_pad_mode_group_get_mode.argtypes = (
			c_void_p,)
		self._libinput.libinput_tablet_pad_mode_group_get_mode.restype = c_uint
		self._libinput.libinput_tablet_pad_mode_group_has_button.argtypes = (
			c_void_p, c_uint)
		self._libinput.libinput_tablet_pad_mode_group_has_button.restype = (
			c_bool)
		self._libinput.libinput_tablet_pad_mode_group_has_ring.argtypes = (
			c_void_p, c_uint)
		self._libinput.libinput_tablet_pad_mode_group_has_ring.restype = c_bool
		self._libinput.libinput_tablet_pad_mode_group_has_strip.argtypes = (
			c_void_p, c_uint)
		self._libinput.libinput_tablet_pad_mode_group_has_strip.restype = (
			c_bool)
		self._libinput \
			.libinput_tablet_pad_mode_group_button_is_toggle.argtypes = (
				c_void_p, c_uint)
		self._libinput \
			.libinput_tablet_pad_mode_group_button_is_toggle.restype = c_bool

		self._libinput.libinput_tablet_pad_mode_group_ref(self._handle)

	def __del__(self):

		self._libinput.libinput_tablet_pad_mode_group_unref(self._handle)

	def __eq__(self, other):

		if isinstance(other, type(self)):
			return self._handle == other._handle
		else:
			return NotImplemented

	@property
	def index(self):
		"""The returned number is the same index as passed to
		:meth:`~libinput.device.Device.tablet_pad_get_mode_group`.

		For tablets with only one mode this number is always 0.

		Returns:
			int: The numeric index this mode group represents, starting at 0.
		"""

		return self._libinput.libinput_tablet_pad_mode_group_get_index(
			self._handle)

	@property
	def num_modes(self):
		"""Query the mode group for the number of available modes.

		The number of modes is usually decided by the number of physical LEDs
		available on the device. Different mode groups may have a different
		number of modes. Use :attr:`mode` to get the currently active mode.

		libinput guarantees that at least one mode is available. A device
		without mode switching capability has a single mode group and
		a single mode.

		Returns:
			int: The number of modes available in this mode group.
		"""

		return self._libinput.libinput_tablet_pad_mode_group_get_num_modes(
			self._handle)

	@property
	def mode(self):
		"""The current mode this mode group is in.

		Returns:
			int: The numeric index of the current mode in this group, starting
			at 0.
		"""

		return self._libinput.libinput_tablet_pad_mode_group_get_mode(
			self._handle)

	def has_button(self, button):
		"""Devices without mode switching capabilities return :obj:`True`
		for every button.

		Args:
			button (int): A button index, starting at 0.
		Returns:
			bool: :obj:`True` if the given button index is part of this
			mode group or :obj:`False` otherwise.
		"""

		return self._libinput.libinput_tablet_pad_mode_group_has_button(
			self._handle, button)

	def has_ring(self, ring):
		"""Devices without mode switching capabilities return :obj:`True`
		for every ring.

		Args:
			ring (int): A ring index, starting at 0.
		Returns:
			bool: :obj:`True` if the given ring index is part of this
			mode group or :obj:`False` otherwise.
		"""

		return self._libinput.libinput_tablet_pad_mode_group_has_ring(
			self._handle, ring)

	def has_strip(self, strip):
		"""Devices without mode switching capabilities return :obj:`True`
		for every strip.

		Args:
			strip (int): A strip index, starting at 0.
		Returns:
			bool: :obj:`True` if the given strip index is part of this
			mode group or :obj:`False` otherwise.
		"""

		return self._libinput.libinput_tablet_pad_mode_group_has_strip(
			self._handle, strip)

	def button_is_toggle(self, button):
		"""The toggle button in a mode group is the button assigned to cycle
		to or directly assign a new mode when pressed.

		Not all devices have a toggle button and some devices may have more
		than one toggle button. For example, the Wacom Cintiq 24HD has six
		toggle buttons in two groups, each directly selecting one of the three
		modes per group.

		Devices without mode switching capabilities return :obj:`False`
		for every button.

		Args:
			button (int): A button index, starting at 0.
		Returns:
			bool: :obj:`True` if the button is a mode toggle button for
			this group, or :obj:`False` otherwise.
		"""

		return self._libinput.libinput_tablet_pad_mode_group_button_is_toggle(
			self._handle, button)
