function [SBMLfieldnames, nNumberFields] = getStructureEnum(varargin)
% [SBMLfieldnames, nNumberFields] = getStructureEnum(typecode, level, version, pkgversion(optional))
%
% Takes
%
% 1. typecode; a string representing the type of object being queried
% 2. level, an integer representing an SBML level
% 3. version, an integer representing an SBML version
% 4. pkgversion, an integer representing the SBML package version 
%       (defaults to 1)
%
% Returns
%
% 1. an array of fieldnames for an SBML structure of the given typecode, level and version
% 2. the number of fieldnames
%

%<!---------------------------------------------------------------------------
% This file is part of libSBML.  Please visit http://sbml.org for more
% information about SBML, and the latest version of libSBML.
%
% Copyright (C) 2013-2018 jointly by the following organizations:
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. EMBL European Bioinformatics Institute (EMBL-EBI), Hinxton, UK
%     3. University of Heidelberg, Heidelberg, Germany
%
% Copyright (C) 2009-2013 jointly by the following organizations: 
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. EMBL European Bioinformatics Institute (EMBL-EBI), Hinxton, UK
%  
% Copyright (C) 2006-2008 by the California Institute of Technology,
%     Pasadena, CA, USA 
%  
% Copyright (C) 2002-2005 jointly by the following organizations: 
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. Japan Science and Technology Agency, Japan
% 
% This library is free software; you can redistribute it and/or modify it
% under the terms of the GNU Lesser General Public License as published by
% the Free Software Foundation.  A copy of the license agreement is provided
% in the file named "LICENSE.txt" included with this software distribution
% and also available online as http://sbml.org/software/libsbml/license.html
%----------------------------------------------------------------------- -->

typecode = varargin{1};
level = varargin{2};
version = varargin{3};

pkgCount = 0;

if (nargin > 3)
    if (nargin < 5)
        error('not enough arguments');
    end
    pkgCount = length(varargin{4});
 %   packages = cell(1, pkgCount);
  %  pkgVersion = zeros(1, pkgCount);
    packages = varargin{4};
    pkgVersion = varargin{5};
end


    SBMLfieldnames = [];
    nNumberFields = 0;
done = 1;


switch (typecode)
  case {'SBML_ALGEBRAIC_RULE', 'AlgebraicRule', 'algebraicRule', 'rule'}
    fhandle = str2func('getAlgebraicRuleFieldnames');
  case {'SBML_ASSIGNMENT_RULE', 'AssignmentRule', 'assignmentRule'}
    fhandle = str2func('getAssignmentRuleFieldnames');
  case {'SBML_COMPARTMENT', 'Compartment', 'compartment'}
    fhandle = str2func('getCompartmentFieldnames');
  case {'SBML_COMPARTMENT_TYPE', 'CompartmentType', 'compartmentType'}
    fhandle = str2func('getCompartmentTypeFieldnames');
  case {'SBML_COMPARTMENT_VOLUME_RULE', 'CompartmentVolumeRule', 'compartmentVolumeRule'}
    fhandle = str2func('getCompartmentVolumeRuleFieldnames');
  case {'SBML_CONSTRAINT', 'Constraint', 'constraint'}
    fhandle = str2func('getConstraintFieldnames');
  case {'SBML_DELAY', 'Delay', 'delay'}
    fhandle = str2func('getDelayFieldnames');
  case {'SBML_EVENT', 'Event', 'event'}
    fhandle = str2func('getEventFieldnames');
  case {'SBML_EVENT_ASSIGNMENT', 'EventAssignment', 'eventAssignment'}
    fhandle = str2func('getEventAssignmentFieldnames');
  case {'SBML_FUNCTION_DEFINITION', 'FunctionDefinition', 'functionDefinition'}
    fhandle = str2func('getFunctionDefinitionFieldnames');
  case {'SBML_INITIAL_ASSIGNMENT', 'InitialAssignment', 'initialAssignment'}
    fhandle = str2func('getInitialAssignmentFieldnames');
  case {'SBML_KINETIC_LAW', 'KineticLaw', 'kineticLaw'}
    fhandle = str2func('getKineticLawFieldnames');
  case {'SBML_LOCAL_PARAMETER', 'LocalParameter', 'localParameter'}
    fhandle = str2func('getLocalParameterFieldnames');
  case {'SBML_MODEL', 'Model', 'model'}
    fhandle = str2func('getModelFieldnames');
  case {'SBML_MODIFIER_SPECIES_REFERENCE', 'ModifierSpeciesReference', 'modifierSpeciesReference', 'modifier'}
    fhandle = str2func('getModifierSpeciesReferenceFieldnames');
  case {'SBML_PARAMETER', 'Parameter', 'parameter'}
    fhandle = str2func('getParameterFieldnames');
  case {'SBML_PARAMETER_RULE', 'ParameterRule', 'parameterRule'}
    fhandle = str2func('getParameterRuleFieldnames');
  case {'SBML_PRIORITY', 'Priority', 'priority'}
    fhandle = str2func('getPriorityFieldnames');
  case {'SBML_RATE_RULE', 'RateRule', 'ruleRule'}
    fhandle = str2func('getRateRuleFieldnames');
  case {'SBML_REACTION', 'Reaction', 'reaction'}
    fhandle = str2func('getReactionFieldnames');
  case {'SBML_SPECIES', 'Species', 'species'}
    fhandle = str2func('getSpeciesFieldnames');
  case {'SBML_SPECIES_CONCENTRATION_RULE', 'SpeciesConcentrationRule', 'speciesConcentrationRule'}
    fhandle = str2func('getSpeciesConcentrationRuleFieldnames');
  case {'SBML_SPECIES_REFERENCE', 'SpeciesReference', 'speciesReference', 'reactant', 'Reactant', 'product', 'Product'}
    fhandle = str2func('getSpeciesReferenceFieldnames');
  case {'SBML_SPECIES_TYPE', 'SpeciesType', 'speciesType'}
    fhandle = str2func('getSpeciesTypeFieldnames');
  case {'SBML_STOICHIOMETRY_MATH', 'StoichiometryMath', 'stoichiometryMath'}
    fhandle = str2func('getStoichiometryMathFieldnames');
  case {'SBML_TRIGGER', 'Trigger', 'trigger'}
    fhandle = str2func('getTriggerFieldnames');
  case {'SBML_UNIT', 'Unit', 'unit'}
    fhandle = str2func('getUnitFieldnames');
  case {'SBML_UNIT_DEFINITION', 'UnitDefinition', 'unitDefinition'}
    fhandle = str2func('getUnitDefinitionFieldnames');
  otherwise
    done = 0;  
end

if done == 1
  [SBMLfieldnames, nNumberFields] = feval(fhandle, level, version);
else
    i = 1;
    found = 0;
    while (i < pkgCount+1)
        [found, fhandle] = getFields(typecode, packages{i}, 0);
        if (found == 1)
            break;
        end
        i = i + 1;
    end
    if (found == 1)
        [SBMLfieldnames, nNumberFields] = feval(fhandle, level, version, pkgVersion(i));
        done = 1;
    end
end
 
if (done == 1 && pkgCount > 0)
    % we may need to extend the fields 
    i = 1;
    while (i < pkgCount+1)
        [found, fhandle] = getFields(typecode, packages{i}, 1);
        if (found == 1)
            copySBML = SBMLfieldnames;
            [newSBMLfieldnames, newNumberFields] = feval(fhandle, level, version, pkgVersion(i));
            SBMLfieldnames = [copySBML, newSBMLfieldnames];
            nNumberFields = nNumberFields + newNumberFields;
        end
%          check for SBase extension
        [found, fhandle] = getFields('SBase', packages{i}, 1);
        if (found == 1)
            copySBML = SBMLfieldnames;
            [newSBMLfieldnames, newNumberFields] = feval(fhandle, level, version, pkgVersion(i));
            SBMLfieldnames = [copySBML, newSBMLfieldnames];
            nNumberFields = nNumberFields + newNumberFields;
        end
    
        i = i + 1;
    end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [found, fhandle] = getFields(typecode, pkg, extension)
    fhandle = str2func('disp');
    found = 0;
    if (strcmp(pkg, 'fbc'))
        if (extension)
            if (isFBCExtension(typecode))
        [found, fhandle] = getFBCFieldnameFunction(typecode);
            end
        else
        [found, fhandle] = getFBCFieldnameFunction(typecode);
        end
    elseif strcmp(pkg, 'qual')
      if (extension)
        if (isQUALExtension(typecode))
          [found, fhandle] = getQUALFieldnameFunction(typecode);
        end
      else
        [found, fhandle] = getQUALFieldnameFunction(typecode);
      end
    elseif strcmp(pkg, 'groups')
      if (extension)
        if (isGROUPSExtension(typecode))
          [found, fhandle] = getGROUPSFieldnameFunction(typecode);
        end
      else
        [found, fhandle] = getGROUPSFieldnameFunction(typecode);
      end
    end
%%%%% REMOVE END
%%%%% ADD ADDITIONAL


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	
function extend = isFBCExtension(typecode)
   extend = 0;
   switch (typecode)
        case {'SBML_FBC_MODEL', 'FBCModel', 'SBML_MODEL', 'Model', 'model'}
             extend = 1;
       case {'SBML_FBC_SPECIES', 'FBCSpecies', 'SBML_SPECIES', 'Species', 'species'}
            extend = 1;
        case {'SBML_FBC_REACTION', 'FBCReaction','SBML_REACTION', 'Reaction', 'reaction'}
            extend = 1;
    case {'SBML_FBC_SBASE', 'FBCSBase', 'SBML_SBASE', 'SBase', 'sbase'}
      extend = 1;
  end
function extend = isQUALExtension(typecode)
  extend = 0;
  switch (typecode)
    case {'SBML_QUAL_MODEL', 'QUALModel', 'SBML_MODEL', 'Model', 'model'}
      extend = 1;
  end

function extend = isGROUPSExtension(typecode)
  extend = 0;
  switch (typecode)
    case {'SBML_GROUPS_MODEL', 'GROUPSModel', 'SBML_MODEL', 'Model', 'model'}
      extend = 1;
  end
%%%% ADD isExtension
%%%% ADD isExtension
%%%% END isExtension

function [found, fhandle] = getFBCFieldnameFunction(typecode)
  found = 1;
  switch (typecode)
    case {'SBML_FBC_FLUXBOUND', 'FluxBound', 'fluxBound', 'fbc_fluxBound'}
      fhandle = str2func('getFluxBoundFieldnames');
    case {'SBML_FBC_OBJECTIVE', 'Objective', 'objective', 'fbc_objective'}
      fhandle = str2func('getObjectiveFieldnames');
    case {'SBML_FBC_FLUXOBJECTIVE', 'FluxObjective', 'fluxObjective', 'fbc_fluxObjective'}
      fhandle = str2func('getFluxObjectiveFieldnames');
    case {'SBML_FBC_GENE_PRODUCT', 'GeneProduct', 'geneProduct', 'fbc_geneProduct'}
      fhandle = str2func('getGeneProductFieldnames');
    case {'SBML_FBC_GENE_PRODUCT_REF', 'GeneProductRef', 'geneProductRef', 'fbc_geneProductRef'}
      fhandle = str2func('getGeneProductRefFieldnames');
    case {'SBML_FBC_ASSOCIATION', 'Association', 'association', 'fbc_association'}
      fhandle = str2func('getAssociationFieldnames');
    case {'SBML_FBC_GENE_PRODUCT_ASSOCIATION', 'GeneProductAssociation', 'geneProductAssociation', 'fbc_geneProductAssociation'}
      fhandle = str2func('getGeneProductAssociationFieldnames');
    case {'SBML_FBC_USERDEFINEDCONSTRAINTCOMPONENT', 'UserDefinedConstraintComponent', 'userDefinedConstraintComponent', 'fbc_userDefinedConstraintComponent'}
      fhandle = str2func('getUserDefinedConstraintComponentFieldnames');
    case {'SBML_FBC_USERDEFINEDCONSTRAINT', 'UserDefinedConstraint', 'userDefinedConstraint', 'fbc_userDefinedConstraint'}
      fhandle = str2func('getUserDefinedConstraintFieldnames');
    case {'SBML_FBC_KEYVALUEPAIR', 'KeyValuePair', 'keyValuePair', 'fbc_keyValuePair'}
      fhandle = str2func('getKeyValuePairFieldnames');
    case {'SBML_FBC_MODEL', 'FBCModel', 'SBML_MODEL', 'Model', 'model'}
      fhandle = str2func('getFBCModelFieldnames');
    case {'SBML_FBC_SPECIES', 'FBCSpecies', 'SBML_SPECIES', 'Species', 'species'}
      fhandle = str2func('getFBCSpeciesFieldnames');
    case {'SBML_FBC_REACTION', 'FBCReaction', 'SBML_REACTION', 'Reaction', 'reaction'}
      fhandle = str2func('getFBCReactionFieldnames');
    case {'SBML_FBC_SBASE', 'FBCSBase', 'SBML_SBASE', 'SBase', 'sbase'}
      fhandle = str2func('getFBCSBaseFieldnames');
        otherwise
          fhandle = str2func('disp');
          found = 0;
  end

function [found, fhandle] = getQUALFieldnameFunction(typecode)
  found = 1;
  switch (typecode)
    case {'SBML_QUAL_QUALITATIVE_SPECIES', 'QualitativeSpecies', 'qualitativeSpecies', 'qual_qualitativeSpecies'}
      fhandle = str2func('getQualitativeSpeciesFieldnames');
    case {'SBML_QUAL_TRANSITION', 'Transition', 'transition', 'qual_transition'}
      fhandle = str2func('getTransitionFieldnames');
    case {'SBML_QUAL_INPUT', 'Input', 'input', 'qual_input'}
      fhandle = str2func('getInputFieldnames');
    case {'SBML_QUAL_OUTPUT', 'Output', 'output', 'qual_output'}
      fhandle = str2func('getOutputFieldnames');
    case {'SBML_QUAL_DEFAULT_TERM', 'DefaultTerm', 'defaultTerm', 'qual_defaultTerm'}
      fhandle = str2func('getDefaultTermFieldnames');
    case {'SBML_QUAL_FUNCTION_TERM', 'FunctionTerm', 'functionTerm', 'qual_functionTerm'}
      fhandle = str2func('getFunctionTermFieldnames');
    case {'SBML_QUAL_MODEL', 'QUALModel', 'SBML_MODEL', 'Model', 'model'}
      fhandle = str2func('getQUALModelFieldnames');
    otherwise
      fhandle = str2func('disp');
      found = 0;
  end

function [found, fhandle] = getGROUPSFieldnameFunction(typecode)
  found = 1;
  switch (typecode)
    case {'SBML_GROUPS_GROUP', 'Group', 'group', 'groups_group'}
      fhandle = str2func('getGroupFieldnames');
    case {'SBML_GROUPS_MEMBER', 'Member', 'member', 'groups_member'}
      fhandle = str2func('getMemberFieldnames');
    case {'SBML_GROUPS_MODEL', 'GROUPSModel', 'SBML_MODEL', 'Model', 'model'}
      fhandle = str2func('getGROUPSModelFieldnames');
    otherwise
      fhandle = str2func('disp');
      found = 0;
  end
%%%% ADD getFieldname
%%%% ADD getFieldname
%%%% END getFieldname

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getAlgebraicRuleFieldnames(level, ...
                                                             version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'type','','',0,6}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 10;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 10;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 11;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 11;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 12;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 12;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
                                   {'id','id','',1,20}, ...
		                 };
                  nNumberFields = 13;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getAssignmentRuleFieldnames(level, ...
                                                             version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 10;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 11;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 11;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 12;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 12;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
                                   {'id','id','',1,20}, ...
		                 };
                  nNumberFields = 13;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getCompartmentFieldnames(level, ...
                                                                    version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'volume','volume','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'outside','outside','',1,20}, ...
		                   {'isSetVolume','volume','',0,7}, ...
		                 };
		nNumberFields = 8;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'spatialDimensions','spatialDimensions','',1,20}, ...
		                   {'size','size','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'outside','outside','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetSize','size','',0,7}, ...
		                   {'isSetVolume','volume','',0,7}, ...
		                 };
		nNumberFields = 13;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'compartmentType','compartmentType','',1,20}, ...
		                   {'spatialDimensions','spatialDimensions','',1,20}, ...
		                   {'size','size','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'outside','outside','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetSize','size','',0,7}, ...
		                   {'isSetVolume','volume','',0,7}, ...
		                 };
		nNumberFields = 14;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'compartmentType','compartmentType','',1,20}, ...
		                   {'spatialDimensions','spatialDimensions','',1,20}, ...
		                   {'size','size','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'outside','outside','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetSize','size','',0,7}, ...
		                   {'isSetVolume','volume','',0,7}, ...
		                 };
		nNumberFields = 15;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'compartmentType','compartmentType','',1,20}, ...
		                   {'spatialDimensions','spatialDimensions','',1,20}, ...
		                   {'size','size','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'outside','outside','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetSize','size','',0,7}, ...
		                   {'isSetVolume','volume','',0,7}, ...
		                 };
		nNumberFields = 16;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'spatialDimensions','spatialDimensions','',1,20}, ...
		                   {'size','size','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetSize','size','',0,7}, ...
		                   {'isSetSpatialDimensions','spatialDimensions','',0,7}, ...
		                 };
		nNumberFields = 14;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'spatialDimensions','spatialDimensions','',1,20}, ...
		                   {'size','size','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetSize','size','',0,7}, ...
		                   {'isSetSpatialDimensions','spatialDimensions','',0,7}, ...
		                 };
		nNumberFields = 14;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getCompartmentTypeFieldnames(level, ...
                                                                        version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                 };
		nNumberFields = 6;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                 };
		nNumberFields = 7;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                 };
		nNumberFields = 8;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getCompartmentVolumeRuleFieldnames(level, ...
                                                             version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'type','','',0,6}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 10;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = [];
		nNumberFields = 0;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = [];
		nNumberFields = 0;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getConstraintFieldnames(level, ...
                                                                   version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                   {'message','message','',1,8}, ...
		                 };
		nNumberFields = 7;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                   {'message','message','',1,8}, ...
		                 };
		nNumberFields = 7;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                   {'message','message','',1,8}, ...
		                 };
		nNumberFields = 8;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                   {'message','message','',1,8}, ...
		                 };
		nNumberFields = 8;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                   {'message','message','',1,8}, ...
                                   {'id','id','',1,20}, ...
                                   {'name','name','',1,20}, ...
		                 };
                  nNumberFields = 10;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getDelayFieldnames(level, ...
                                                              version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 6;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 7;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 7;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
                                   {'id','id','',1,20}, ...
                                   {'name','name','',1,20}, ...
		                 };
                  nNumberFields =9;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getEventAssignmentFieldnames(level, ...
                                                                        version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 6;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 7;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 7;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 8;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 8;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'math','math','',1,5}, ...
                                   {'id','id','',1,20}, ...
                                   {'name','name','',1,20}, ...
		                 };
                  nNumberFields = 10;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getEventFieldnames(level, ...
                                                              version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'trigger','trigger','',1,13}, ...
		                   {'delay','delay','',1,13}, ...
		                   {'timeUnits','timeUnits','',1,20}, ...
		                   {'eventAssignment','eventAssignment','',1,20}, ...
		                 };
		nNumberFields = 10;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'trigger','trigger','',1,13}, ...
		                   {'delay','delay','',1,13}, ...
		                   {'timeUnits','timeUnits','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'eventAssignment','eventAssignment','',1,20}, ...
		                 };
		nNumberFields = 11;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'trigger','trigger','',1,20}, ...
		                   {'delay','delay','',1,20}, ...
		                   {'eventAssignment','eventAssignment','',1,20}, ...
		                 };
		nNumberFields = 10;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'useValuesFromTriggerTime','useValuesFromTriggerTime','',1,20}, ...
		                   {'trigger','trigger','',1,20}, ...
		                   {'delay','delay','',1,20}, ...
		                   {'eventAssignment','eventAssignment','',1,20}, ...
		                 };
		nNumberFields = 12;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'useValuesFromTriggerTime','useValuesFromTriggerTime','',1,20}, ...
		                   {'trigger','trigger','',1,20}, ...
		                   {'delay','delay','',1,20}, ...
		                   {'priority','priority','',1,20}, ...
		                   {'eventAssignment','eventAssignment','',1,20}, ...
		                 };
		nNumberFields = 13;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'useValuesFromTriggerTime','useValuesFromTriggerTime','',1,20}, ...
		                   {'trigger','trigger','',1,20}, ...
		                   {'delay','delay','',1,20}, ...
		                   {'priority','priority','',1,20}, ...
		                   {'eventAssignment','eventAssignment','',1,20}, ...
		                 };
		nNumberFields = 13;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getFunctionDefinitionFieldnames(level, ...
                                                                           version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 7;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 8;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 8;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 9;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 9;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 9;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getInitialAssignmentFieldnames(level, ...
                                                                          version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'symbol','symbol','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 7;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'symbol','symbol','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 7;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'symbol','symbol','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 8;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'symbol','symbol','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 8;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'symbol','symbol','',1,20}, ...
		                   {'math','math','',1,5}, ...
                                   {'id','id','',1,20}, ...
                                   {'name','name','',1,20}, ...
		                 };
                  nNumberFields = 10;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getKineticLawFieldnames(level, ...
                                                                   version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'parameter','parameter','',1,20}, ...
		                   {'timeUnits','timeUnits','',1,20}, ...
		                   {'substanceUnits','substanceUnits','',1,20}, ...
		                 };
		nNumberFields = 7;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'math','math','',1,5}, ...
		                   {'parameter','parameter','',1,20}, ...
		                   {'timeUnits','timeUnits','',1,20}, ...
		                   {'substanceUnits','substanceUnits','',1,20}, ...
		                 };
		nNumberFields = 9;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'math','math','',1,5}, ...
		                   {'parameter','parameter','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                 };
		nNumberFields = 8;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'math','math','',1,5}, ...
		                   {'parameter','parameter','',1,20}, ...
		                 };
		nNumberFields = 8;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'math','math','',1,5}, ...
		                   {'parameter','parameter','',1,20}, ...
		                 };
		nNumberFields = 9;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                   {'localParameter','localParameter','',1,20}, ...
		                 };
		nNumberFields = 8;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                   {'localParameter','localParameter','',1,20}, ...
                                   {'id','id','',1,20}, ...
                                   {'name','name','',1,20}, ...
		                 };
                  nNumberFields = 10;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getLocalParameterFieldnames(level, ...
                                                                       version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'value','value','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'isSetValue','value','',0,7}, ...
		                 };
		nNumberFields = 11;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'value','value','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'isSetValue','value','',0,7}, ...
		                 };
		nNumberFields = 11;
  end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getModelFieldnames(level, ...
                                                              version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'SBML_level','','',0,0}, ...
		                   {'SBML_version','','',0,0}, ...
		                   {'name','name','',1,20}, ...
		                   {'unitDefinition','unitDefinition','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'parameter','parameter','',1,20}, ...
		                   {'rule','rule','',1,20}, ...
		                   {'reaction','reaction','',1,20}, ...
		                   {'namespaces','','',0,3}, ...
		                 };
		nNumberFields = 13;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'SBML_level','','',0,0}, ...
		                   {'SBML_version','','',0,0}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'functionDefinition','functionDefinition','',1,20}, ...
		                   {'unitDefinition','unitDefinition','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'parameter','parameter','',1,20}, ...
		                   {'rule','rule','',1,20}, ...
		                   {'reaction','reaction','',1,20}, ...
		                   {'event','event','',1,20}, ...
		                   {'time_symbol','','',0,9}, ...
		                   {'delay_symbol','','',0,10}, ...
		                   {'namespaces','','',0,3}, ...
		                 };
		nNumberFields = 19;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'SBML_level','','',0,0}, ...
		                   {'SBML_version','','',0,0}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'functionDefinition','functionDefinition','',1,20}, ...
		                   {'unitDefinition','unitDefinition','',1,20}, ...
		                   {'compartmentType','compartmentType','',1,20}, ...
		                   {'speciesType','speciesType','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'parameter','parameter','',1,20}, ...
		                   {'initialAssignment','initialAssignment','',1,20}, ...
		                   {'rule','rule','',1,20}, ...
		                   {'constraint','constraint','',1,20}, ...
		                   {'reaction','reaction','',1,20}, ...
		                   {'event','event','',1,20}, ...
		                   {'time_symbol','','',0,9}, ...
		                   {'delay_symbol','','',0,10}, ...
		                   {'namespaces','','',0,3}, ...
		                 };
		nNumberFields = 24;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'SBML_level','','',0,0}, ...
		                   {'SBML_version','','',0,0}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'functionDefinition','functionDefinition','',1,20}, ...
		                   {'unitDefinition','unitDefinition','',1,20}, ...
		                   {'compartmentType','compartmentType','',1,20}, ...
		                   {'speciesType','speciesType','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'parameter','parameter','',1,20}, ...
		                   {'initialAssignment','initialAssignment','',1,20}, ...
		                   {'rule','rule','',1,20}, ...
		                   {'constraint','constraint','',1,20}, ...
		                   {'reaction','reaction','',1,20}, ...
		                   {'event','event','',1,20}, ...
		                   {'time_symbol','','',0,9}, ...
		                   {'delay_symbol','','',0,10}, ...
		                   {'namespaces','','',0,3}, ...
		                 };
		nNumberFields = 24;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'SBML_level','','',0,0}, ...
		                   {'SBML_version','','',0,0}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'functionDefinition','functionDefinition','',1,20}, ...
		                   {'unitDefinition','unitDefinition','',1,20}, ...
		                   {'compartmentType','compartmentType','',1,20}, ...
		                   {'speciesType','speciesType','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'parameter','parameter','',1,20}, ...
		                   {'initialAssignment','initialAssignment','',1,20}, ...
		                   {'rule','rule','',1,20}, ...
		                   {'constraint','constraint','',1,20}, ...
		                   {'reaction','reaction','',1,20}, ...
		                   {'event','event','',1,20}, ...
		                   {'time_symbol','','',0,9}, ...
		                   {'delay_symbol','','',0,10}, ...
		                   {'namespaces','','',0,3}, ...
		                 };
		nNumberFields = 25;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'SBML_level','','',0,0}, ...
		                   {'SBML_version','','',0,0}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'timeUnits','timeUnits','',1,20}, ...
		                   {'substanceUnits','substanceUnits','',1,20}, ...
		                   {'volumeUnits','volumeUnits','',1,20}, ...
		                   {'areaUnits','areaUnits','',1,20}, ...
		                   {'lengthUnits','lengthUnits','',1,20}, ...
		                   {'extentUnits','extentUnits','',1,20}, ...
		                   {'conversionFactor','conversionFactor','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'functionDefinition','functionDefinition','',1,20}, ...
		                   {'unitDefinition','unitDefinition','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'parameter','parameter','',1,20}, ...
		                   {'initialAssignment','initialAssignment','',1,20}, ...
		                   {'rule','rule','',1,20}, ...
		                   {'constraint','constraint','',1,20}, ...
		                   {'reaction','reaction','',1,20}, ...
		                   {'event','event','',1,20}, ...
		                   {'time_symbol','','',0,9}, ...
		                   {'delay_symbol','','',0,10}, ...
		                   {'avogadro_symbol','','',0,11}, ...
		                   {'namespaces','','',0,3}, ...
		                 };
		nNumberFields = 31;
        elseif (version == 2)
		SBMLfieldnames = { 
                           {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'SBML_level','','',0,0}, ...
		                   {'SBML_version','','',0,0}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'timeUnits','timeUnits','',1,20}, ...
		                   {'substanceUnits','substanceUnits','',1,20}, ...
		                   {'volumeUnits','volumeUnits','',1,20}, ...
		                   {'areaUnits','areaUnits','',1,20}, ...
		                   {'lengthUnits','lengthUnits','',1,20}, ...
		                   {'extentUnits','extentUnits','',1,20}, ...
		                   {'conversionFactor','conversionFactor','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'functionDefinition','functionDefinition','',1,20}, ...
		                   {'unitDefinition','unitDefinition','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'parameter','parameter','',1,20}, ...
		                   {'initialAssignment','initialAssignment','',1,20}, ...
		                   {'rule','rule','',1,20}, ...
		                   {'constraint','constraint','',1,20}, ...
		                   {'reaction','reaction','',1,20}, ...
		                   {'event','event','',1,20}, ...
		                   {'time_symbol','','',0,9}, ...
		                   {'delay_symbol','','',0,10}, ...
		                   {'avogadro_symbol','','',0,11}, ...
		                   {'rateOf_symbol','','',0,12}, ...
		                   {'namespaces','','',0,3}, ...
		                 };
		nNumberFields = 32;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getModifierSpeciesReferenceFieldnames(level, ...
                                                                                 version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'species','species','',1,20}, ...
		                 };
		nNumberFields = 5;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'species','species','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                 };
		nNumberFields = 8;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                 };
		nNumberFields = 8;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                 };
		nNumberFields = 9;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                 };
		nNumberFields = 9;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                 };
                  nNumberFields = 9;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getParameterFieldnames(level, ...
                                                                  version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'value','value','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'isSetValue','value','',0,7}, ...
		                 };
		nNumberFields = 7;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'value','value','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetValue','value','',0,7}, ...
		                 };
		nNumberFields = 10;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'value','value','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'isSetValue','value','',0,7}, ...
		                 };
		nNumberFields = 11;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'value','value','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetValue','value','',0,7}, ...
		                 };
		nNumberFields = 11;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'value','value','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetValue','value','',0,7}, ...
		                 };
		nNumberFields = 12;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'value','value','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetValue','value','',0,7}, ...
		                 };
		nNumberFields = 12;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'value','value','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetValue','value','',0,7}, ...
		                 };
		nNumberFields = 12;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getParameterRuleFieldnames(level, ...
                                                             version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'type','','',0,6}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 10;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = [];
		nNumberFields = 0;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = [];
		nNumberFields = 0;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getPriorityFieldnames(level, ...
                                                                 version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 7;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
                                   {'id','id','',1,20}, ...
                                   {'name','name','',1,20}, ...
		                 };
                  nNumberFields = 9;
  end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getRateRuleFieldnames(level, ...
                                                             version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 10;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 11;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 11;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 12;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 12;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
                                   {'id','id','',1,20}, ...
		                 };
                  nNumberFields = 13;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getReactionFieldnames(level, ...
                                                                 version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'reactant','reactant','',1,20}, ...
		                   {'product','product','',1,20}, ...
		                   {'kineticLaw','kineticLaw','',1,20}, ...
		                   {'reversible','reversible','',1,20}, ...
		                   {'fast','fast','',1,20}, ...
		                 };
		nNumberFields = 9;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'reactant','reactant','',1,20}, ...
		                   {'product','product','',1,20}, ...
		                   {'modifier','modifier','',1,20}, ...
		                   {'kineticLaw','kineticLaw','',1,20}, ...
		                   {'reversible','reversible','',1,20}, ...
		                   {'fast','fast','',1,20}, ...
		                   {'isSetFast','fast','',0,7}, ...
		                 };
		nNumberFields = 13;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'reactant','reactant','',1,20}, ...
		                   {'product','product','',1,20}, ...
		                   {'modifier','modifier','',1,20}, ...
		                   {'kineticLaw','kineticLaw','',1,20}, ...
		                   {'reversible','reversible','',1,20}, ...
		                   {'fast','fast','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'isSetFast','fast','',0,7}, ...
		                 };
		nNumberFields = 14;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'reactant','reactant','',1,20}, ...
		                   {'product','product','',1,20}, ...
		                   {'modifier','modifier','',1,20}, ...
		                   {'kineticLaw','kineticLaw','',1,20}, ...
		                   {'reversible','reversible','',1,20}, ...
		                   {'fast','fast','',1,20}, ...
		                   {'isSetFast','fast','',0,7}, ...
		                 };
		nNumberFields = 14;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'reactant','reactant','',1,20}, ...
		                   {'product','product','',1,20}, ...
		                   {'modifier','modifier','',1,20}, ...
		                   {'kineticLaw','kineticLaw','',1,20}, ...
		                   {'reversible','reversible','',1,20}, ...
		                   {'fast','fast','',1,20}, ...
		                   {'isSetFast','fast','',0,7}, ...
		                 };
		nNumberFields = 15;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'reactant','reactant','',1,20}, ...
		                   {'product','product','',1,20}, ...
		                   {'modifier','modifier','',1,20}, ...
		                   {'kineticLaw','kineticLaw','',1,20}, ...
		                   {'reversible','reversible','',1,20}, ...
		                   {'fast','fast','',1,20}, ...
		                   {'isSetFast','fast','',0,7}, ...
		                   {'compartment','compartment','',1,20}, ...
		                 };
		nNumberFields = 16;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'reactant','reactant','',1,20}, ...
		                   {'product','product','',1,20}, ...
		                   {'modifier','modifier','',1,20}, ...
		                   {'kineticLaw','kineticLaw','',1,20}, ...
		                   {'reversible','reversible','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                 };
		nNumberFields = 14;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getSpeciesConcentrationRuleFieldnames(level, ...
                                                             version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'type','','',0,6}, ...
		                   {'formula','formula','',1,5}, ...
		                   {'variable','variable','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                 };
		nNumberFields = 10;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = [];
		nNumberFields = 0;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = [];
		nNumberFields = 0;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getSpeciesFieldnames(level, ...
                                                                version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'initialAmount','initialAmount','',1,20}, ...
		                   {'units','units','',1,20}, ...
		                   {'boundaryCondition','boundaryCondition','',1,20}, ...
		                   {'charge','charge','',1,20}, ...
		                   {'isSetInitialAmount','initialAmount','',0,7}, ...
		                   {'isSetCharge','charge','',0,7}, ...
		                 };
		nNumberFields = 11;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'initialAmount','initialAmount','',1,20}, ...
		                   {'initialConcentration','initialConcentration','',1,20}, ...
		                   {'substanceUnits','substanceUnits','',1,20}, ...
		                   {'spatialSizeUnits','spatialSizeUnits','',1,20}, ...
		                   {'hasOnlySubstanceUnits','hasOnlySubstanceUnits','',1,20}, ...
		                   {'boundaryCondition','boundaryCondition','',1,20}, ...
		                   {'charge','charge','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetInitialAmount','initialAmount','',0,7}, ...
		                   {'isSetInitialConcentration','initialConcentration','',0,7}, ...
		                   {'isSetCharge','charge','',0,7}, ...
		                 };
		nNumberFields = 18;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'speciesType','speciesType','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'initialAmount','initialAmount','',1,20}, ...
		                   {'initialConcentration','initialConcentration','',1,20}, ...
		                   {'substanceUnits','substanceUnits','',1,20}, ...
		                   {'spatialSizeUnits','spatialSizeUnits','',1,20}, ...
		                   {'hasOnlySubstanceUnits','hasOnlySubstanceUnits','',1,20}, ...
		                   {'boundaryCondition','boundaryCondition','',1,20}, ...
		                   {'charge','charge','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetInitialAmount','initialAmount','',0,7}, ...
		                   {'isSetInitialConcentration','initialConcentration','',0,7}, ...
		                   {'isSetCharge','charge','',0,7}, ...
		                 };
		nNumberFields = 19;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'speciesType','speciesType','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'initialAmount','initialAmount','',1,20}, ...
		                   {'initialConcentration','initialConcentration','',1,20}, ...
		                   {'substanceUnits','substanceUnits','',1,20}, ...
		                   {'hasOnlySubstanceUnits','hasOnlySubstanceUnits','',1,20}, ...
		                   {'boundaryCondition','boundaryCondition','',1,20}, ...
		                   {'charge','charge','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetInitialAmount','initialAmount','',0,7}, ...
		                   {'isSetInitialConcentration','initialConcentration','',0,7}, ...
		                   {'isSetCharge','charge','',0,7}, ...
		                 };
		nNumberFields = 19;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'speciesType','speciesType','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'initialAmount','initialAmount','',1,20}, ...
		                   {'initialConcentration','initialConcentration','',1,20}, ...
		                   {'substanceUnits','substanceUnits','',1,20}, ...
		                   {'hasOnlySubstanceUnits','hasOnlySubstanceUnits','',1,20}, ...
		                   {'boundaryCondition','boundaryCondition','',1,20}, ...
		                   {'charge','charge','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetInitialAmount','initialAmount','',0,7}, ...
		                   {'isSetInitialConcentration','initialConcentration','',0,7}, ...
		                   {'isSetCharge','charge','',0,7}, ...
		                 };
		nNumberFields = 20;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'initialAmount','initialAmount','',1,20}, ...
		                   {'initialConcentration','initialConcentration','',1,20}, ...
		                   {'substanceUnits','substanceUnits','',1,20}, ...
		                   {'hasOnlySubstanceUnits','hasOnlySubstanceUnits','',1,20}, ...
		                   {'boundaryCondition','boundaryCondition','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetInitialAmount','initialAmount','',0,7}, ...
		                   {'isSetInitialConcentration','initialConcentration','',0,7}, ...
		                   {'conversionFactor','conversionFactor','',1,20}, ...
		                 };
		nNumberFields = 18;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'compartment','compartment','',1,20}, ...
		                   {'initialAmount','initialAmount','',1,20}, ...
		                   {'initialConcentration','initialConcentration','',1,20}, ...
		                   {'substanceUnits','substanceUnits','',1,20}, ...
		                   {'hasOnlySubstanceUnits','hasOnlySubstanceUnits','',1,20}, ...
		                   {'boundaryCondition','boundaryCondition','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetInitialAmount','initialAmount','',0,7}, ...
		                   {'isSetInitialConcentration','initialConcentration','',0,7}, ...
		                   {'conversionFactor','conversionFactor','',1,20}, ...
		                 };
		nNumberFields = 18;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getSpeciesReferenceFieldnames(level, ...
                                                                         version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'species','species','',1,20}, ...
		                   {'stoichiometry','stoichiometry','',1,20}, ...
		                   {'denominator','denominator','',1,20}, ...
		                 };
		nNumberFields = 6;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'species','species','',1,20}, ...
		                   {'stoichiometry','stoichiometry','',1,20}, ...
		                   {'denominator','denominator','',1,20}, ...
		                   {'stoichiometryMath','stoichiometryMath','',1,13}, ...
		                 };
		nNumberFields = 8;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'species','species','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'stoichiometry','stoichiometry','',1,20}, ...
		                   {'stoichiometryMath','stoichiometryMath','',1,13}, ...
		                 };
		nNumberFields = 10;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'stoichiometry','stoichiometry','',1,20}, ...
		                   {'stoichiometryMath','stoichiometryMath','',1,20}, ...
		                 };
		nNumberFields = 10;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'stoichiometry','stoichiometry','',1,20}, ...
		                   {'stoichiometryMath','stoichiometryMath','',1,20}, ...
		                 };
		nNumberFields = 11;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'stoichiometry','stoichiometry','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetStoichiometry','stoichiometry','',0,7}, ...
		                 };
		nNumberFields = 12;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
                          {'sboTerm','sboTerm','',1,20}, ...
		                   {'species','species','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'stoichiometry','stoichiometry','',1,20}, ...
		                   {'constant','constant','',1,20}, ...
		                   {'isSetStoichiometry','stoichiometry','',0,7}, ...
		                 };
		nNumberFields = 12;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getSpeciesTypeFieldnames(level, ...
                                                                    version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                 };
		nNumberFields = 6;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                 };
		nNumberFields = 7;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                 };
		nNumberFields = 8;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getStoichiometryMathFieldnames(level, ...
                                                                          version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 6;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 7;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getTriggerFieldnames(level, ...
                                                                version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 6;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 7;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'persistent','persistent','',1,20}, ...
		                   {'initialValue','initialValue','',1,20}, ...
		                   {'math','math','',1,5}, ...
		                 };
		nNumberFields = 9;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'persistent','persistent','',1,20}, ...
		                   {'initialValue','initialValue','',1,20}, ...
		                   {'math','math','',1,5}, ...
                                   {'id','id','',1,20}, ...
                                   {'name','name','',1,20}, ...
		                 };
                  nNumberFields = 11;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getUnitDefinitionFieldnames(level, ...
                                                                       version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'unit','unit','',1,20}, ...
		                 };
		nNumberFields = 5;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'unit','unit','',1,20}, ...
		                 };
		nNumberFields = 7;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'unit','unit','',1,20}, ...
		                 };
		nNumberFields = 7;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'unit','unit','',1,20}, ...
		                 };
		nNumberFields = 8;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'unit','unit','',1,20}, ...
		                 };
		nNumberFields = 9;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'unit','unit','',1,20}, ...
		                 };
		nNumberFields = 9;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'name','name','',1,20}, ...
		                   {'id','id','',1,20}, ...
		                   {'unit','unit','',1,20}, ...
		                 };
		nNumberFields = 9;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getUnitFieldnames(level, ...
                                                             version)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end
if (level == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'kind','kind','',1,20}, ...
		                   {'exponent','exponent','',1,20}, ...
		                   {'scale','scale','',1,20}, ...
		                 };
		nNumberFields = 6;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'kind','kind','',1,20}, ...
		                   {'exponent','exponent','',1,20}, ...
		                   {'scale','scale','',1,20}, ...
		                   {'multiplier','multiplier','',1,20}, ...
		                   {'offset','offset','',1,20}, ...
		                 };
		nNumberFields = 9;
	elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'kind','kind','',1,20}, ...
		                   {'exponent','exponent','',1,20}, ...
		                   {'scale','scale','',1,20}, ...
		                   {'multiplier','multiplier','',1,20}, ...
		                 };
		nNumberFields = 8;
	elseif (version == 3)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'kind','kind','',1,20}, ...
		                   {'exponent','exponent','',1,20}, ...
		                   {'scale','scale','',1,20}, ...
		                   {'multiplier','multiplier','',1,20}, ...
		                 };
		nNumberFields = 9;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                           {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'kind','kind','',1,20}, ...
		                   {'exponent','exponent','',1,20}, ...
		                   {'scale','scale','',1,20}, ...
		                   {'multiplier','multiplier','',1,20}, ...
		                 };
		nNumberFields = 10;
	end
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'kind','kind','',1,20}, ...
		                   {'exponent','exponent','',1,20}, ...
		                   {'scale','scale','',1,20}, ...
		                   {'multiplier','multiplier','',1,20}, ...
		                 };
		nNumberFields = 10;
        elseif (version == 2)
		SBMLfieldnames = { {'typecode','','',0,0}, ...
		                   {'metaid','metaid','',1,20}, ...
		                   {'notes','notes','',1,1}, ...
		                   {'annotation','annotation','',1,2}, ...
                          {'cvterms','','',0,4}, ...
		                   {'sboTerm','sboTerm','',1,20}, ...
		                   {'kind','kind','',1,20}, ...
		                   {'exponent','exponent','',1,20}, ...
		                   {'scale','scale','',1,20}, ...
		                   {'multiplier','multiplier','',1,20}, ...
                                   {'id','id','',1,20}, ...
                                   {'name','name','',1,20}, ...
		                 };
                  nNumberFields = 12;
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getFBCModelFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'fbc_objective', 'objective', 'fbc', 1, 20}, ...
        {'fbc_activeObjective', 'activeObjective', 'fbc', 1, 20}, ...
        {'fbc_fluxBound', 'fluxBound', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 6;
      elseif (pkgVersion == 2)
        SBMLfieldnames = {
        {'fbc_objective', 'objective', 'fbc', 1, 20}, ...
        {'fbc_activeObjective', 'activeObjective', 'fbc', 1, 20}, ...
        {'fbc_geneProduct', 'geneProduct', 'fbc', 1, 20}, ...
        {'fbc_strict', 'strict', 'fbc', 1, 20}, ...
        {'isSetfbc_strict', 'strict', 'fbc', 0, 7}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 8;
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'fbc_objective', 'objective', 'fbc', 1, 20}, ...
        {'fbc_activeObjective', 'activeObjective', 'fbc', 1, 20}, ...
        {'fbc_geneProduct', 'geneProduct', 'fbc', 1, 20}, ...
        {'fbc_userDefinedConstraint', 'userDefinedConstraint', 'fbc', 1, 20}, ...
        {'fbc_strict', 'strict', 'fbc', 1, 20}, ...
        {'isSetfbc_strict', 'strict', 'fbc', 0, 7}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 9;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getFBCSpeciesFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'fbc_charge', 'charge', 'fbc', 1, 20}, ...
        {'isSetfbc_charge', 'charge', 'fbc', 0, 7}, ...
        {'fbc_chemicalFormula', 'chemicalFormula', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 6;
      elseif (pkgVersion == 2)
        SBMLfieldnames = {
        {'fbc_charge', 'charge', 'fbc', 1, 20}, ...
        {'isSetfbc_charge', 'charge', 'fbc', 0, 7}, ...
        {'fbc_chemicalFormula', 'chemicalFormula', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 6;
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'fbc_charge', 'charge', 'fbc', 1, 20}, ...
        {'isSetfbc_charge', 'charge', 'fbc', 0, 7}, ...
        {'fbc_chemicalFormula', 'chemicalFormula', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 6;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getFBCReactionFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
      elseif (pkgVersion == 2)
        SBMLfieldnames = {
        {'fbc_geneProductAssociation', 'geneProductAssociation', 'fbc', 1, 20}, ...
        {'fbc_lowerFluxBound', 'lowerFluxBound', 'fbc', 1, 20}, ...
        {'fbc_upperFluxBound', 'upperFluxBound', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 6;
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'fbc_geneProductAssociation', 'geneProductAssociation', 'fbc', 1, 20}, ...
        {'fbc_lowerFluxBound', 'lowerFluxBound', 'fbc', 1, 20}, ...
        {'fbc_upperFluxBound', 'upperFluxBound', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 6;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getFBCSBaseFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
      elseif (pkgVersion == 2)
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'fbc_keyValuePair', 'keyValuePair', 'fbc', 1, 20}, ...
        {'fbc_kvp_xmlns', 'xmlns', 'fbc', 1, 20}, ...
        };
        nNumberFields = 2;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getFluxBoundFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_reaction', 'reaction', 'fbc', 1, 20}, ...
        {'fbc_operation', 'operation', 'fbc', 1, 20}, ...
        {'fbc_value', 'value', 'fbc', 1, 20}, ...
        {'isSetfbc_value', 'value', 'fbc', 0, 7}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 15;
      elseif (pkgVersion == 2)
      elseif (pkgVersion == 3)
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getObjectiveFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_type', 'type', 'fbc', 1, 20}, ...
        {'fbc_fluxObjective', 'fluxObjective', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 13;
      elseif (pkgVersion == 2)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_type', 'type', 'fbc', 1, 20}, ...
        {'fbc_fluxObjective', 'fluxObjective', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 13;
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_type', 'type', 'fbc', 1, 20}, ...
        {'fbc_fluxObjective', 'fluxObjective', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 13;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getFluxObjectiveFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_reaction', 'reaction', 'fbc', 1, 20}, ...
        {'fbc_coefficient', 'coefficient', 'fbc', 1, 20}, ...
        {'isSetfbc_coefficient', 'coefficient', 'fbc', 0, 7}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 14;
      elseif (pkgVersion == 2)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_reaction', 'reaction', 'fbc', 1, 20}, ...
        {'fbc_coefficient', 'coefficient', 'fbc', 1, 20}, ...
        {'isSetfbc_coefficient', 'coefficient', 'fbc', 0, 7}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 14;
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_reaction', 'reaction', 'fbc', 1, 20}, ...
        {'fbc_coefficient', 'coefficient', 'fbc', 1, 20}, ...
        {'isSetfbc_coefficient', 'coefficient', 'fbc', 0, 7}, ...
        {'fbc_variableType', 'variableType', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 15;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getGeneProductFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
      elseif (pkgVersion == 2)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_label', 'label', 'fbc', 1, 20}, ...
        {'fbc_associatedSpecies', 'associatedSpecies', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 13;
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_label', 'label', 'fbc', 1, 20}, ...
        {'fbc_associatedSpecies', 'associatedSpecies', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 13;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getGeneProductRefFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
      elseif (pkgVersion == 2)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_geneProduct', 'geneProduct', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 12;
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_geneProduct', 'geneProduct', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 12;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getAssociationFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
      elseif (pkgVersion == 2)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
		{'fbc_association', 'association', 'fbc', 1, 14}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 10;
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
		{'fbc_association', 'association', 'fbc', 1, 14}, ...	
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 10;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getGeneProductAssociationFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
      elseif (pkgVersion == 2)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_association', 'association', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 12;
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_association', 'association', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 12;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getUserDefinedConstraintComponentFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
      elseif (pkgVersion == 2)
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_coefficient', 'coefficient', 'fbc', 1, 20}, ...
        {'isSetfbc_coefficient', 'coefficient', 'fbc', 0, 7}, ...
        {'fbc_variable', 'variable', 'fbc', 1, 20}, ...
        {'fbc_variableType', 'variableType', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 15;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getUserDefinedConstraintFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
      elseif (pkgVersion == 2)
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'metaid', 'metaid', '', 1, 20}, ...
        {'notes', 'notes', '', 1, 1}, ...
        {'annotation', 'annotation', '', 1, 2}, ...
        {'cvterms', '', '', 0, 4}, ...
        {'sboTerm', 'sboTerm', '', 1, 20}, ...
        {'fbc_id', 'id', 'fbc', 1, 20}, ...
        {'fbc_name', 'name', 'fbc', 1, 20}, ...
        {'fbc_lowerBound', 'lowerBound', 'fbc', 1, 20}, ...
        {'fbc_upperBound', 'upperBound', 'fbc', 1, 20}, ...
        {'fbc_userDefinedConstraintComponent', 'userDefinedConstraintComponent', 'fbc', 1, 20}, ...
        {'level', '', '', 0, 0}, ...
        {'version', '', '', 0, 0}, ...
        {'fbc_version', 'version', 'fbc', 1, 20}, ...
        };
        nNumberFields = 14;
      end
    end
  end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getKeyValuePairFieldnames(level, ...
                                                       version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
      elseif (pkgVersion == 2)
      elseif (pkgVersion == 3)
        SBMLfieldnames = {
        {'typecode', '', '', 0, 0}, ...
        {'fbc_key', 'key', 'fbc', 1, 20}, ...
        {'fbc_value', 'value', 'fbc', 1, 20}, ...
        {'fbc_uri', 'uri', 'fbc', 1, 20}, ...
        };
        nNumberFields = 4;
      end
    end
  end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getQUALModelFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end
  SBMLfieldnames = [];
  nNumberFields = 0;
  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'qual_version','version','qual',1,20}, ...
        {'qual_qualitativeSpecies','qualitativeSpecies','qual',1,20}, ...
        {'qual_transition','transition','qual',1,20}, ...
        };
        nNumberFields = 3;
      end
    end
  end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getQualitativeSpeciesFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end
  SBMLfieldnames = [];
  nNumberFields = 0;
  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'typecode','','',0,0}, ...
        {'metaid','metaid','',1,20}, ...
        {'notes','notes','',1,1}, ...
        {'annotation','annotation','',1,2}, ...
        {'cvterms','','',0,4}, ...
        {'sboTerm','sboTerm','',1,20}, ...
        {'qual_id','id','qual',1,20}, ...
        {'qual_name','name','qual',1,20}, ...
        {'qual_compartment','compartment','qual',1,20}, ...
        {'qual_constant','constant','qual',1,20}, ...
        {'qual_initialLevel','initialLevel','qual',1,20}, ...
        {'qual_maxLevel','maxLevel','qual',1,20}, ...
        {'level','','', 0, 0}, ...
        {'version','','', 0, 0}, ...
        {'qual_version','version','qual',1,20}, ...
        };
        nNumberFields = 15;
      end
    end
  end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getTransitionFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end
  SBMLfieldnames = [];
  nNumberFields = 0;
  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'typecode','','',0,0}, ...
        {'metaid','metaid','',1,20}, ...
        {'notes','notes','',1,1}, ...
        {'annotation','annotation','',1,2}, ...
        {'cvterms','','',0,4}, ...
        {'sboTerm','sboTerm','',1,20}, ...
        {'qual_id','id','qual',1,20}, ...
        {'qual_name','name','qual',1,20}, ...
        {'qual_input','input','qual',1,20}, ...
        {'qual_output','output','qual',1,20}, ...
        {'qual_functionTerm','functionTerm','qual',1,20}, ...
        {'qual_defaultTerm','defaultTerm','qual',1,20}, ...
        {'level','','', 0, 0}, ...
        {'version','','', 0, 0}, ...
        {'qual_version','version','qual',1,20}, ...
        };
        nNumberFields = 14;
      end
    end
  end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getInputFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end
  SBMLfieldnames = [];
  nNumberFields = 0;
  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'typecode','','',0,0}, ...
        {'metaid','metaid','',1,20}, ...
        {'notes','notes','',1,1}, ...
        {'annotation','annotation','',1,2}, ...
        {'cvterms','','',0,4}, ...
        {'sboTerm','sboTerm','',1,20}, ...
        {'qual_id','id','qual',1,20}, ...
        {'qual_name','name','qual',1,20}, ...
        {'qual_sign','sign','qual',1,20}, ...
        {'qual_qualitativeSpecies','qualitativeSpecies','qual',1,20}, ...
        {'qual_transitionEffect','transitionEffect','qual',1,20}, ...
        {'qual_thresholdLevel','thresholdLevel','qual',1,20}, ...
        {'level','','', 0, 0}, ...
        {'version','','', 0, 0}, ...
        {'qual_version','version','qual',1,20}, ...
        };
        nNumberFields = 15;
      end
    end
  end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getOutputFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end
  SBMLfieldnames = [];
  nNumberFields = 0;
  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'typecode','','',0,0}, ...
        {'metaid','metaid','',1,20}, ...
        {'notes','notes','',1,1}, ...
        {'annotation','annotation','',1,2}, ...
        {'cvterms','','',0,4}, ...
        {'sboTerm','sboTerm','',1,20}, ...
        {'qual_id','id','qual',1,20}, ...
        {'qual_qualitativeSpecies','qualitativeSpecies','qual',1,20}, ...
        {'qual_transitionEffect','transitionEffect','qual',1,20}, ...
        {'qual_name','name','qual',1,20}, ...
        {'qual_outputLevel','outputLevel','qual',1,20}, ...
        {'level','','', 0, 0}, ...
        {'version','','', 0, 0}, ...
        {'qual_version','version','qual',1,20}, ...
        };
        nNumberFields = 14;
      end
    end
  end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getDefaultTermFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end
  SBMLfieldnames = [];
  nNumberFields = 0;
  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'typecode','','',0,0}, ...
        {'metaid','metaid','',1,20}, ...
        {'notes','notes','',1,1}, ...
        {'annotation','annotation','',1,2}, ...
        {'cvterms','','',0,4}, ...
        {'sboTerm','sboTerm','',1,20}, ...
        {'qual_resultLevel','resultLevel','qual',1,20}, ...
        {'level','','', 0, 0}, ...
        {'version','','', 0, 0}, ...
        {'qual_version','version','qual',1,20}, ...
        };
        nNumberFields = 10;
      end
    end
  end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getFunctionTermFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end
  SBMLfieldnames = [];
  nNumberFields = 0;
  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'typecode','','',0,0}, ...
        {'metaid','metaid','',1,20}, ...
        {'notes','notes','',1,1}, ...
        {'annotation','annotation','',1,2}, ...
        {'cvterms','','',0,4}, ...
        {'sboTerm','sboTerm','',1,20}, ...
        {'qual_resultLevel','resultLevel','qual',1,20}, ...
        {'math','math','',1,5}, ...
        {'level','','', 0, 0}, ...
        {'version','','', 0, 0}, ...
        {'qual_version','version','qual',1,20}, ...
        };
        nNumberFields = 11;
      end
    end
  end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getGROUPSModelFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end
  SBMLfieldnames = [];
  nNumberFields = 0;
  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'groups_version','version','groups',1,20}, ...
        {'groups_group','group','groups',1,20}, ...
        };
        nNumberFields = 2;
      end
    end
  end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getGroupFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end
  SBMLfieldnames = [];
  nNumberFields = 0;
  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'typecode','','',0,0}, ...
        {'metaid','metaid','',1,20}, ...
        {'notes','notes','',1,1}, ...
        {'annotation','annotation','',1,2}, ...
        {'cvterms','','',0,4}, ...
        {'sboTerm','sboTerm','',1,20}, ...
        {'groups_id','id','groups',1,20}, ...
        {'groups_name','name','groups',1,20}, ...
        {'groups_kind','kind','groups',1,20}, ...
        {'groups_member','member','groups',1,20}, ...
        {'level','','', 0, 0}, ...
        {'version','','', 0, 0}, ...
        {'groups_version','version','groups',1,20}, ...
        };
        nNumberFields = 13;
      end
    end
  end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getMemberFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end
  SBMLfieldnames = [];
  nNumberFields = 0;
  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        {'typecode','','',0,0}, ...
        {'metaid','metaid','',1,20}, ...
        {'notes','notes','',1,1}, ...
        {'annotation','annotation','',1,2}, ...
        {'cvterms','','',0,4}, ...
        {'sboTerm','sboTerm','',1,20}, ...
        {'groups_id','id','groups',1,20}, ...
        {'groups_name','name','groups',1,20}, ...
        {'groups_idRef','idRef','groups',1,20}, ...
        {'groups_metaIdRef','metaIdRef','groups',1,20}, ...
        {'level','','', 0, 0}, ...
        {'version','','', 0, 0}, ...
        {'groups_version','version','groups',1,20}, ...
        };
        nNumberFields = 13;
      end
    end
  end
%%%% ADD functions
%%%% ADD functions
%%%% END functions
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function valid = isValidLevelVersionCombination(level, version)
valid = 1;
if ~isIntegralNumber(level)
	error('level must be an integer');
elseif ~isIntegralNumber(version)
	error('version must be an integer');
end
if (level < 1 || level > 3)
	error('current SBML levels are 1, 2 or 3');
end
if (level == 1)
	if (version < 1 || version > 2)
		error('SBMLToolbox supports versions 1-2 of SBML Level 1');
	end
elseif (level == 2)
	if (version < 1 || version > 5)
		error('SBMLToolbox supports versions 1-5 of SBML Level 2');
	end
elseif (level == 3)
	if (version < 1 || version > 2)
		error('SBMLToolbox supports version 1-2 of SBML Level 3');
	end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function value = isIntegralNumber(number)
value = 0;
integerClasses = {'int8', 'uint8', 'int16', 'uint16', 'int32', 'uint32', 'int64', 'uint64'};
% since the function isinteger does not exist in MATLAB Rel 13
% this is not used
%if (isinteger(number))
if (ismember(class(number), integerClasses))
    value = 1;
elseif (isnumeric(number))
    % if it is an integer 
    if (number == fix(number))
        value = 1;
    end
end
