"""Utility functions for the CLI commands."""

import glob
import os
from pathlib import Path


def get_lily_directory() -> Path:
    """Get the prompts directory."""
    return Path.cwd() / "lily"


def generate_lily_init(files: list[str]) -> str:
    """Generate the __init__.py file for the lily directory.

    Args:
        files: The files to import in the __init__.py file.

    Returns:
        The contents of the __init__.py file.
    """
    imports = "\n".join([f"from .{file} import {file}" for file in files])
    return f"""# This file is auto-generated by Lilypad
from lilypad import configure

{imports}

configure()

__all__ = {files}
"""


def generate_llm_fn_stub(function_name: str) -> str:
    """The stub for the LLM function."""
    return f"""import lilypad

    
@lilypad.llm_fn(synced=True)
def {function_name}() -> str: ...


if __name__ == "__main__":
    output = {function_name}()
    print(output)
"""


def find_file_names(directory: Path, prefix: str = "") -> list[str]:
    """Finds all files in a directory.

    Args:
        directory: The directory to search for the prompt.
        prefix: The prefix of the prompt to search for.

    Returns:
        A list of file names found.
    """
    pattern = os.path.join(directory, f"[!_]{prefix}*.py")  # ignores private files
    matching_files_with_dir = glob.glob(pattern)

    # Removing the directory part from each path
    return [os.path.basename(file) for file in matching_files_with_dir]


def lily_directory_files(prefix: str = "") -> list[str]:
    """Finds all prompt files in the lily directory.

    Args:
        prefix: The prefix of the prompt to search for.

    Returns:
        A list of prompt file names found.
    """
    lily_directory = get_lily_directory()
    prompt_file_names = find_file_names(lily_directory, prefix)
    return [f"{name[:-3]}" for name in prompt_file_names]  # remove .py extension


def parse_prompt_file_name(prompt_file_name: str) -> str:
    """Returns the file name without the .py extension."""
    if prompt_file_name.endswith(".py"):
        return prompt_file_name[:-3]
    return prompt_file_name
