"""
lokalise.client
~~~~~~~~~~~~~~~
This module contains API client definition.
"""

import importlib
from collections.abc import Callable
from typing import TypeVar, cast

from lokalise.utils import snake_to_camel

from .base_client import BaseClient
from .client_methods.branches import BranchMethods
from .client_methods.comments import CommentMethods
from .client_methods.contributors import ContributorMethods
from .client_methods.files import FileMethods
from .client_methods.glossary_terms import GlossaryTermsMethods
from .client_methods.jwt import JwtMethods
from .client_methods.keys import KeyMethods
from .client_methods.languages import LanguageMethods
from .client_methods.orders import OrderMethods
from .client_methods.payment_cards import PaymentCardMethods
from .client_methods.permission_templates import PermissionTemplateMethods
from .client_methods.processes import ProcessMethods
from .client_methods.projects import ProjectMethods
from .client_methods.screenshots import ScreenshotMethods
from .client_methods.segments import SegmentMethods
from .client_methods.snapshots import SnapshotMethods
from .client_methods.tasks import TaskMethods
from .client_methods.team_user_billing_details import TeamUserBillingDetailsMethods
from .client_methods.team_user_groups import TeamUserGroupMethods
from .client_methods.team_users import TeamUserMethods
from .client_methods.teams import TeamMethods
from .client_methods.translation_providers import TranslationProviderMethods
from .client_methods.translation_statuses import TranslationStatusMethods
from .client_methods.translations import TranslationMethods
from .client_methods.webhooks import WebhookMethods
from .endpoints.base_endpoint import BaseEndpoint

T = TypeVar("T", bound=BaseEndpoint)


class Client(
    BaseClient,
    BranchMethods,
    CommentMethods,
    ContributorMethods,
    FileMethods,
    GlossaryTermsMethods,
    JwtMethods,
    KeyMethods,
    LanguageMethods,
    OrderMethods,
    PaymentCardMethods,
    PermissionTemplateMethods,
    ProjectMethods,
    ProcessMethods,
    SnapshotMethods,
    ScreenshotMethods,
    SegmentMethods,
    TaskMethods,
    TeamMethods,
    TeamUserMethods,
    TeamUserGroupMethods,
    TeamUserBillingDetailsMethods,
    TranslationMethods,
    TranslationProviderMethods,
    TranslationStatusMethods,
    WebhookMethods,
):
    """Client used to send API requests.

    Usage:

        import lokalise
        client = lokalise.Client('api_token')
        client.projects()
    """

    def reset_client(self) -> None:
        """Resets the API client by clearing all attributes.
        After reset, token is None and client is unusable until you set a new token.
        """
        self._token = None
        self._connect_timeout = None
        self._read_timeout = None
        self._enable_compression = False
        self._clear_endpoint_attrs()

    # === Endpoint helpers
    def get_endpoint(self, name: str) -> BaseEndpoint:
        """Lazily loads an endpoint with a given name and stores it
        under a specific instance attribute. For example, if the `name`
        is "projects", then it will load .endpoints.projects_endpoint module
        and then set attribute like this:
            self._projects_endpoint = ProjectsEndpoint(self)

        This is internal; prefer calling mixin methods like client.projects().

        :param str name: Endpoint name to load
        """
        endpoint_name = name + "_endpoint"
        camelized_name = snake_to_camel(endpoint_name)

        # Dynamically load the necessary endpoint module
        try:
            module = importlib.import_module(f".endpoints.{endpoint_name}", package="lokalise")
            endpoint_klass = getattr(module, camelized_name)
        except (ModuleNotFoundError, AttributeError) as e:
            raise ValueError(f"Unknown endpoint: {name}") from e

        # Find endpoint class in the module
        endpoint_klass: type[BaseEndpoint] = getattr(module, camelized_name)
        return self._fetch_attr(f"_{endpoint_name}", lambda: endpoint_klass(self))

    def _fetch_attr(self, attr_name: str, populator: Callable[[], T]) -> T:
        """Searches for the given attribute.
        Uses populator to set the attribute if it cannot be found.
        Used to lazy-load endpoints.
        """
        val = getattr(self, attr_name, None)
        if val is None:
            val = populator()
            setattr(self, attr_name, val)
        return cast(T, val)

    def _clear_endpoint_attrs(self) -> None:
        """Clears all lazily-loaded endpoint attributes"""
        for attr in [a for a in vars(self) if a.endswith("_endpoint")]:
            delattr(self, attr)
