# SPDX-License-Identifier: MIT
# Copyright (c) 2020, Mathias Laurin

from __future__ import annotations

import abc
import datetime as dt
import enum
import sys
from pathlib import Path
from typing import NamedTuple, Sequence, Tuple, Type, TypeVar, Union

from mbedtls.hashlib import Hash as _Hash
from mbedtls.pk import ECC, RSA

if sys.version_info < (3, 8):
    from typing_extensions import Literal
else:
    from typing import Literal

_Path = Union[str, Path]
_DER = bytes
_PEM = str
_KEY = Union[_DER, _PEM]
_PKEY = Union[ECC, RSA]

@enum.unique
class KeyUsage(enum.IntEnum):
    DIGITAL_SIGNATURE: int
    NON_REPUDIATION: int
    KEY_ENCIPHERMENT: int
    DATA_ENCIPHERMENT: int
    KEY_AGREEMENT: int
    KEY_CERT_SIGN: int
    CRL_SIGN: int
    ENCIPHER_ONLY: int
    DECIPHER_ONLY: int

def PEM_to_DER(pem: _PEM) -> _DER: ...
def DER_to_PEM(der: _DER, text: str) -> _PEM: ...

TCertificate = TypeVar("TCertificate", bound=Certificate)

class Certificate(abc.ABC):
    @classmethod
    def from_buffer(
        cls: Type[TCertificate], buffer: bytes
    ) -> TCertificate: ...
    @classmethod
    @abc.abstractmethod
    def from_file(cls: Type[TCertificate], path: _Path) -> TCertificate: ...
    @classmethod
    @abc.abstractmethod
    def from_DER(cls: Type[TCertificate], der: _DER) -> TCertificate: ...
    @classmethod
    @abc.abstractmethod
    def from_PEM(cls: Type[TCertificate], pem: _PEM) -> TCertificate: ...
    def __hash__(self) -> int: ...
    def __eq__(self, other: object) -> bool: ...
    @abc.abstractmethod
    def __str__(self) -> str: ...
    def __bytes__(self) -> bytes: ...
    def export(self, format: Literal["DER", "PEM"]) -> _KEY: ...
    def to_bytes(self) -> _DER: ...
    @abc.abstractmethod
    def to_DER(self) -> _DER: ...
    @abc.abstractmethod
    def to_PEM(self) -> _PEM: ...

class BasicConstraints(NamedTuple):
    ca: bool = ...
    max_path_length: int = ...

class CRT(Certificate):
    def __init__(self, buffer: bytes) -> None: ...
    def __next__(self) -> CRT: ...
    def __str__(self) -> str: ...
    @property
    def tbs_certificate(self) -> bytes: ...
    @property
    def signature_value(self) -> bytes: ...
    @property
    def version(self) -> int: ...
    @property
    def serial_number(self) -> int: ...
    @property
    def digestmod(self) -> _Hash: ...
    @property
    def issuer(self) -> str: ...
    @property
    def not_before(self) -> dt.datetime: ...
    @property
    def not_after(self) -> dt.datetime: ...
    @property
    def subject(self) -> str: ...
    @property
    def subject_public_key(self) -> _PKEY: ...
    @property
    def key_usage(self) -> object: ...
    @property
    def subject_alternative_names(self) -> Sequence[str]: ...
    @property
    def basic_constraints(self) -> BasicConstraints: ...
    def check_revocation(self, other: CRL) -> bool: ...
    @classmethod
    def from_file(cls, path: _Path) -> CRT: ...
    @classmethod
    def from_DER(cls, buffer: _DER) -> CRT: ...
    @classmethod
    def from_PEM(cls, pem: _PEM) -> CRT: ...
    def to_DER(self) -> _DER: ...
    def to_PEM(self) -> _PEM: ...
    def sign(
        self,
        csr: CSR,
        issuer_key: _PKEY,
        not_before: dt.datetime,
        not_after: dt.datetime,
        serial_number: int,
        basic_constraints: BasicConstraints = ...,
    ) -> CRT: ...
    @classmethod
    def selfsign(
        cls,
        csr: CSR,
        issuer_key: _PKEY,
        not_before: dt.datetime,
        not_after: dt.datetime,
        serial_number: int,
        basic_constraints: BasicConstraints = ...,
    ) -> CRT: ...
    def verify(self, crt: CRT) -> bool: ...
    @staticmethod
    def new(
        not_before: dt.datetime,
        not_after: dt.datetime,
        issuer: str,
        issuer_key: _PKEY,
        subject: str,
        subject_key: _PKEY,
        serial_number: int,
        digestmod: _Hash,
        basic_constraints: BasicConstraints = ...,
    ) -> CRT: ...

class CSR(Certificate):
    def __init__(self, buffer: bytes) -> None: ...
    def __str__(self) -> str: ...
    @property
    def digestmod(self) -> _Hash: ...
    @property
    def version(self) -> int: ...
    @property
    def subject(self) -> str: ...
    @property
    def subject_public_key(self) -> object: ...
    @classmethod
    def from_file(cls, path: _Path) -> CSR: ...
    @classmethod
    def from_DER(cls, buffer: _DER) -> CSR: ...
    @classmethod
    def from_PEM(cls, pem: _PEM) -> CSR: ...
    def to_DER(self) -> _DER: ...
    def to_PEM(self) -> _PEM: ...
    @staticmethod
    def new(subject_key: _PKEY, subject: str, digestmod: _Hash) -> CSR: ...

class CRLEntry(NamedTuple):
    serial: int
    revocation_date: dt.datetime

class CRL(Certificate):
    def __init__(self, buffer: bytes) -> None: ...
    def __next__(self) -> CRL: ...
    def __str__(self) -> str: ...
    @property
    def tbs_certificate(self) -> bytes: ...
    @property
    def signature_value(self) -> bytes: ...
    @property
    def version(self) -> int: ...
    @property
    def issuer_name(self) -> str: ...
    @property
    def this_update(self) -> dt.datetime: ...
    @property
    def next_update(self) -> dt.datetime: ...
    @property
    def revoked_certificates(self) -> Tuple[CRLEntry, ...]: ...
    @classmethod
    def from_file(cls, path: _Path) -> CRL: ...
    @classmethod
    def from_DER(cls, buffer: _DER) -> CRL: ...
    @classmethod
    def from_PEM(cls, pem: _PEM) -> CRL: ...
    def to_DER(self) -> _DER: ...
    def to_PEM(self) -> _PEM: ...
