#include <memory>
#include <string>
#include <vector>

#include <pybind11/pybind11.h>
#include <pybind11/stl.h>
#include <mecab.h>

namespace py = pybind11;

void initialize_tagger(py::module &m) {
  // Reference: https://taku910.github.io/mecab/doxygen/classMeCab_1_1Tagger.html
  py::class_<MeCab::Tagger>(m, "Tagger")
    .def(py::init([](const std::vector<std::string> &arguments) {
      std::vector<const char *> argv {""}; // argv[0] is excutable name
      for (const auto &argument : arguments) {
        argv.push_back(argument.c_str());
      }

      MeCab::Tagger *tagger = MeCab::Tagger::create(argv.size(), const_cast<char **>(argv.data()));
      if (tagger == nullptr) {
        throw pybind11::value_error(MeCab::getLastError());
      }

      return std::unique_ptr<MeCab::Tagger>(tagger);
    }))
    .def("parse", py::overload_cast<const char *>(&MeCab::Tagger::parse), py::return_value_policy::reference)
    .def("parse", py::overload_cast<MeCab::Lattice *>(&MeCab::Tagger::parse, py::const_))
    .def("set_theta", &MeCab::Tagger::set_theta)
    .def("theta", &MeCab::Tagger::theta)
    .def("dictionary_info", &MeCab::Tagger::dictionary_info, py::return_value_policy::reference)
    .def("what", &MeCab::Tagger::what, py::return_value_policy::copy)
    .def_static("version", &MeCab::Tagger::version)
  ;
}