import numpy as np

# Here we extend the Paul Bourke treatment (http://paulbourke.net/geometry/polygonise/marchingsource.cpp) with
# Eric Lengyel's (http://transvoxel.org/Lengyel-VoxelTerrain.pdf).
#
# We assume each cube is arranged in the Euclidean 3-space as follows, where v indicates the vertex number and e
# indicates the edge number. NOTE: The numbering has changed from marching_cubes.py to fit the Transvoxel(TM) lookup
# tables. It now matched the dual marching cubes numbering.
#
#           z
#           ^
#           |
#          v4 ----e4----v5
#          /|           /|
#         e7|          e5|
#        /  e8        /  |
#       v6----e6----v7   e9
#       |   |        |   |
#       |  v0----e0--|- v1---> x
#      e11 /        e10 /
#       | e3         | e1
#       |/           |/
#       v2----e2-----v3
#      /
#     y
#

INTERPOLATION_BITMASK = [1 << n for n in range(12)]
EDGE_BITMASK = INTERPOLATION_BITMASK[0:8]

# The following data originates from Eric Lengyel's Transvoxel Algorithm.
# http://transvoxel.org/
REGULAR_CELL_CLASS = np.array([
    0x00, 0x01, 0x01, 0x03, 0x01, 0x03, 0x02, 0x04, 0x01, 0x02, 0x03, 0x04, 0x03, 0x04, 0x04, 0x03,
    0x01, 0x03, 0x02, 0x04, 0x02, 0x04, 0x06, 0x0C, 0x02, 0x05, 0x05, 0x0B, 0x05, 0x0A, 0x07, 0x04,
    0x01, 0x02, 0x03, 0x04, 0x02, 0x05, 0x05, 0x0A, 0x02, 0x06, 0x04, 0x0C, 0x05, 0x07, 0x0B, 0x04,
    0x03, 0x04, 0x04, 0x03, 0x05, 0x0B, 0x07, 0x04, 0x05, 0x07, 0x0A, 0x04, 0x08, 0x0E, 0x0E, 0x03,
    0x01, 0x02, 0x02, 0x05, 0x03, 0x04, 0x05, 0x0B, 0x02, 0x06, 0x05, 0x07, 0x04, 0x0C, 0x0A, 0x04,
    0x03, 0x04, 0x05, 0x0A, 0x04, 0x03, 0x07, 0x04, 0x05, 0x07, 0x08, 0x0E, 0x0B, 0x04, 0x0E, 0x03,
    0x02, 0x06, 0x05, 0x07, 0x05, 0x07, 0x08, 0x0E, 0x06, 0x09, 0x07, 0x0F, 0x07, 0x0F, 0x0E, 0x0D,
    0x04, 0x0C, 0x0B, 0x04, 0x0A, 0x04, 0x0E, 0x03, 0x07, 0x0F, 0x0E, 0x0D, 0x0E, 0x0D, 0x02, 0x01,
    0x01, 0x02, 0x02, 0x05, 0x02, 0x05, 0x06, 0x07, 0x03, 0x05, 0x04, 0x0A, 0x04, 0x0B, 0x0C, 0x04,
    0x02, 0x05, 0x06, 0x07, 0x06, 0x07, 0x09, 0x0F, 0x05, 0x08, 0x07, 0x0E, 0x07, 0x0E, 0x0F, 0x0D,
    0x03, 0x05, 0x04, 0x0B, 0x05, 0x08, 0x07, 0x0E, 0x04, 0x07, 0x03, 0x04, 0x0A, 0x0E, 0x04, 0x03,
    0x04, 0x0A, 0x0C, 0x04, 0x07, 0x0E, 0x0F, 0x0D, 0x0B, 0x0E, 0x04, 0x03, 0x0E, 0x02, 0x0D, 0x01,
    0x03, 0x05, 0x05, 0x08, 0x04, 0x0A, 0x07, 0x0E, 0x04, 0x07, 0x0B, 0x0E, 0x03, 0x04, 0x04, 0x03,
    0x04, 0x0B, 0x07, 0x0E, 0x0C, 0x04, 0x0F, 0x0D, 0x0A, 0x0E, 0x0E, 0x02, 0x04, 0x03, 0x0D, 0x01,
    0x04, 0x07, 0x0A, 0x0E, 0x0B, 0x0E, 0x0E, 0x02, 0x0C, 0x0F, 0x04, 0x0D, 0x04, 0x0D, 0x03, 0x01,
    0x03, 0x04, 0x04, 0x03, 0x04, 0x03, 0x0D, 0x01, 0x04, 0x0D, 0x03, 0x01, 0x03, 0x01, 0x01, 0x00
])

REGULAR_CELL_DATA_PREFIX = np.array([0x00, 0x31, 0x62, 0x42, 0x53, 0x73, 0x93, 0x84, 0x84, 0xC4, 0x64, 0x64, 0x64, 0x64, 0x75, 0x95])

REGULAR_CELL_DATA = np.array([
    [-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1],
    [0, 1, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1],
    [0, 1, 2, 3, 4, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1],
    [0, 1, 2, 0, 2, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1],
    [0, 1, 4, 1, 3, 4, 1, 2, 3, -1, -1, -1, -1, -1, -1, -1],
    [0, 1, 2, 0, 2, 3, 4, 5, 6, -1, -1, -1, -1, -1, -1, -1],
    [0, 1, 2, 3, 4, 5, 6, 7, 8, -1, -1, -1, -1, -1, -1, -1],
    [0, 1, 4, 1, 3, 4, 1, 2, 3, 5, 6, 7, -1, -1, -1, -1],
    [0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, -1, -1, -1, -1],
    [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, -1, -1, -1, -1],
    [0, 4, 5, 0, 1, 4, 1, 3, 4, 1, 2, 3, -1, -1, -1, -1],
    [0, 5, 4, 0, 4, 1, 1, 4, 3, 1, 3, 2, -1, -1, -1, -1],
    [0, 4, 5, 0, 3, 4, 0, 1, 3, 1, 2, 3, -1, -1, -1, -1],
    [0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, -1, -1, -1, -1],
    [0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 5, 6, -1],
    [0, 4, 5, 0, 3, 4, 0, 1, 3, 1, 2, 3, 6, 7, 8, -1]
])

REGULAR_VERTEX_DATA = np.array([
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x3304, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x4113, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x3304, 0x2315, 0x4113, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4223, 0x1326, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x6201, 0x4223, 0x1326, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x4113, 0x5102, 0x4223, 0x1326, 0, 0, 0, 0, 0, 0],
    [0x4223, 0x1326, 0x3304, 0x2315, 0x4113, 0, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x8337, 0x4223, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x3304, 0x4223, 0x4113, 0x8337, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x8337, 0x4223, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x3304, 0x2315, 0x8337, 0x4223, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4113, 0x8337, 0x1326, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x8337, 0x1326, 0x3304, 0x6201, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x8337, 0x1326, 0x5102, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x2315, 0x8337, 0x1326, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x1146, 0x2245, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x1146, 0x2245, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x4113, 0x3304, 0x1146, 0x2245, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x4113, 0x5102, 0x1146, 0x2245, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4223, 0x1326, 0x3304, 0x1146, 0x2245, 0, 0, 0, 0, 0, 0],
    [0x1146, 0x2245, 0x6201, 0x4223, 0x1326, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x1146, 0x2245, 0x6201, 0x2315, 0x4113, 0x5102, 0x4223, 0x1326, 0, 0, 0],
    [0x4223, 0x1326, 0x1146, 0x2245, 0x2315, 0x4113, 0, 0, 0, 0, 0, 0],
    [0x4223, 0x4113, 0x8337, 0x3304, 0x1146, 0x2245, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x1146, 0x2245, 0x4223, 0x4113, 0x8337, 0, 0, 0, 0, 0],
    [0x4223, 0x6201, 0x2315, 0x8337, 0x3304, 0x1146, 0x2245, 0, 0, 0, 0, 0],
    [0x4223, 0x8337, 0x2315, 0x2245, 0x1146, 0x5102, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4113, 0x8337, 0x1326, 0x3304, 0x1146, 0x2245, 0, 0, 0, 0, 0],
    [0x4113, 0x8337, 0x1326, 0x1146, 0x2245, 0x6201, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x8337, 0x1326, 0x5102, 0x3304, 0x1146, 0x2245, 0, 0, 0, 0],
    [0x2245, 0x2315, 0x8337, 0x1326, 0x1146, 0, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x2245, 0x8157, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x3304, 0x2315, 0x2245, 0x8157, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x6201, 0x2245, 0x8157, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x2245, 0x8157, 0x4113, 0x5102, 0x3304, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4223, 0x1326, 0x2315, 0x2245, 0x8157, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x4223, 0x1326, 0x3304, 0x2315, 0x2245, 0x8157, 0, 0, 0, 0, 0],
    [0x6201, 0x2245, 0x8157, 0x4113, 0x5102, 0x4223, 0x1326, 0, 0, 0, 0, 0],
    [0x4223, 0x1326, 0x3304, 0x2245, 0x8157, 0x4113, 0, 0, 0, 0, 0, 0],
    [0x4223, 0x4113, 0x8337, 0x2315, 0x2245, 0x8157, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x3304, 0x4223, 0x4113, 0x8337, 0x2315, 0x2245, 0x8157, 0, 0, 0],
    [0x8337, 0x4223, 0x6201, 0x2245, 0x8157, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x3304, 0x2245, 0x8157, 0x8337, 0x4223, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4113, 0x8337, 0x1326, 0x2315, 0x2245, 0x8157, 0, 0, 0, 0, 0],
    [0x4113, 0x8337, 0x1326, 0x3304, 0x6201, 0x2315, 0x2245, 0x8157, 0, 0, 0, 0],
    [0x5102, 0x1326, 0x8337, 0x8157, 0x2245, 0x6201, 0, 0, 0, 0, 0, 0],
    [0x8157, 0x8337, 0x1326, 0x3304, 0x2245, 0, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x3304, 0x1146, 0x8157, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x1146, 0x8157, 0x2315, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x1146, 0x8157, 0x4113, 0x6201, 0, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x5102, 0x1146, 0x8157, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x3304, 0x1146, 0x8157, 0x5102, 0x4223, 0x1326, 0, 0, 0, 0, 0],
    [0x1326, 0x4223, 0x6201, 0x2315, 0x8157, 0x1146, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x1146, 0x8157, 0x4113, 0x6201, 0x5102, 0x4223, 0x1326, 0, 0, 0, 0],
    [0x1326, 0x1146, 0x8157, 0x4113, 0x4223, 0, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x3304, 0x1146, 0x8157, 0x4223, 0x4113, 0x8337, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x1146, 0x8157, 0x2315, 0x4223, 0x4113, 0x8337, 0, 0, 0, 0],
    [0x3304, 0x1146, 0x8157, 0x8337, 0x4223, 0x6201, 0, 0, 0, 0, 0, 0],
    [0x4223, 0x5102, 0x1146, 0x8157, 0x8337, 0, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x3304, 0x1146, 0x8157, 0x5102, 0x4113, 0x8337, 0x1326, 0, 0, 0, 0],
    [0x6201, 0x4113, 0x8337, 0x1326, 0x1146, 0x8157, 0x2315, 0, 0, 0, 0, 0],
    [0x6201, 0x3304, 0x1146, 0x8157, 0x8337, 0x1326, 0x5102, 0, 0, 0, 0, 0],
    [0x1326, 0x1146, 0x8157, 0x8337, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x1326, 0x8267, 0x1146, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x3304, 0x1326, 0x8267, 0x1146, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x4113, 0x1326, 0x8267, 0x1146, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x3304, 0x2315, 0x4113, 0x1326, 0x8267, 0x1146, 0, 0, 0, 0, 0],
    [0x5102, 0x4223, 0x8267, 0x1146, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x6201, 0x4223, 0x8267, 0x1146, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4223, 0x8267, 0x1146, 0x6201, 0x2315, 0x4113, 0, 0, 0, 0, 0],
    [0x1146, 0x8267, 0x4223, 0x4113, 0x2315, 0x3304, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x8337, 0x4223, 0x1326, 0x8267, 0x1146, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x3304, 0x4223, 0x4113, 0x8337, 0x1326, 0x8267, 0x1146, 0, 0, 0],
    [0x6201, 0x2315, 0x8337, 0x4223, 0x1326, 0x8267, 0x1146, 0, 0, 0, 0, 0],
    [0x5102, 0x3304, 0x2315, 0x8337, 0x4223, 0x1326, 0x8267, 0x1146, 0, 0, 0, 0],
    [0x8267, 0x1146, 0x5102, 0x4113, 0x8337, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x4113, 0x8337, 0x8267, 0x1146, 0x3304, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x8337, 0x8267, 0x1146, 0x5102, 0, 0, 0, 0, 0, 0],
    [0x1146, 0x3304, 0x2315, 0x8337, 0x8267, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x1326, 0x8267, 0x2245, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x1326, 0x8267, 0x2245, 0x6201, 0x5102, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x1326, 0x8267, 0x2245, 0x6201, 0x2315, 0x4113, 0, 0, 0, 0, 0],
    [0x1326, 0x8267, 0x2245, 0x2315, 0x4113, 0x5102, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4223, 0x8267, 0x2245, 0x3304, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x4223, 0x8267, 0x2245, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4223, 0x8267, 0x2245, 0x3304, 0x6201, 0x2315, 0x4113, 0, 0, 0, 0],
    [0x4113, 0x4223, 0x8267, 0x2245, 0x2315, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x1326, 0x8267, 0x2245, 0x4223, 0x4113, 0x8337, 0, 0, 0, 0, 0],
    [0x1326, 0x8267, 0x2245, 0x6201, 0x5102, 0x4223, 0x4113, 0x8337, 0, 0, 0, 0],
    [0x3304, 0x1326, 0x8267, 0x2245, 0x4223, 0x6201, 0x2315, 0x8337, 0, 0, 0, 0],
    [0x5102, 0x1326, 0x8267, 0x2245, 0x2315, 0x8337, 0x4223, 0, 0, 0, 0, 0],
    [0x3304, 0x2245, 0x8267, 0x8337, 0x4113, 0x5102, 0, 0, 0, 0, 0, 0],
    [0x8337, 0x8267, 0x2245, 0x6201, 0x4113, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x6201, 0x2315, 0x8337, 0x8267, 0x2245, 0x3304, 0, 0, 0, 0, 0],
    [0x2315, 0x8337, 0x8267, 0x2245, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x2245, 0x8157, 0x1326, 0x8267, 0x1146, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x3304, 0x2315, 0x2245, 0x8157, 0x1326, 0x8267, 0x1146, 0, 0, 0],
    [0x6201, 0x2245, 0x8157, 0x4113, 0x1326, 0x8267, 0x1146, 0, 0, 0, 0, 0],
    [0x2245, 0x8157, 0x4113, 0x5102, 0x3304, 0x1326, 0x8267, 0x1146, 0, 0, 0, 0],
    [0x4223, 0x8267, 0x1146, 0x5102, 0x2315, 0x2245, 0x8157, 0, 0, 0, 0, 0],
    [0x3304, 0x6201, 0x4223, 0x8267, 0x1146, 0x2315, 0x2245, 0x8157, 0, 0, 0, 0],
    [0x4223, 0x8267, 0x1146, 0x5102, 0x6201, 0x2245, 0x8157, 0x4113, 0, 0, 0, 0],
    [0x3304, 0x2245, 0x8157, 0x4113, 0x4223, 0x8267, 0x1146, 0, 0, 0, 0, 0],
    [0x4223, 0x4113, 0x8337, 0x2315, 0x2245, 0x8157, 0x1326, 0x8267, 0x1146, 0, 0, 0],
    [0x6201, 0x5102, 0x3304, 0x4223, 0x4113, 0x8337, 0x2315, 0x2245, 0x8157, 0x1326, 0x8267, 0x1146],
    [0x8337, 0x4223, 0x6201, 0x2245, 0x8157, 0x1326, 0x8267, 0x1146, 0, 0, 0, 0],
    [0x4223, 0x5102, 0x3304, 0x2245, 0x8157, 0x8337, 0x1326, 0x8267, 0x1146, 0, 0, 0],
    [0x8267, 0x1146, 0x5102, 0x4113, 0x8337, 0x2315, 0x2245, 0x8157, 0, 0, 0, 0],
    [0x6201, 0x4113, 0x8337, 0x8267, 0x1146, 0x3304, 0x2315, 0x2245, 0x8157, 0, 0, 0],
    [0x8337, 0x8267, 0x1146, 0x5102, 0x6201, 0x2245, 0x8157, 0, 0, 0, 0, 0],
    [0x3304, 0x2245, 0x8157, 0x8337, 0x8267, 0x1146, 0, 0, 0, 0, 0, 0],
    [0x8157, 0x2315, 0x3304, 0x1326, 0x8267, 0, 0, 0, 0, 0, 0, 0],
    [0x8267, 0x8157, 0x2315, 0x6201, 0x5102, 0x1326, 0, 0, 0, 0, 0, 0],
    [0x8267, 0x1326, 0x3304, 0x6201, 0x4113, 0x8157, 0, 0, 0, 0, 0, 0],
    [0x8267, 0x8157, 0x4113, 0x5102, 0x1326, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4223, 0x8267, 0x8157, 0x2315, 0x3304, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x6201, 0x4223, 0x8267, 0x8157, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x5102, 0x4223, 0x8267, 0x8157, 0x4113, 0x6201, 0, 0, 0, 0, 0],
    [0x4113, 0x4223, 0x8267, 0x8157, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x8157, 0x2315, 0x3304, 0x1326, 0x8267, 0x4223, 0x4113, 0x8337, 0, 0, 0, 0],
    [0x8157, 0x2315, 0x6201, 0x5102, 0x1326, 0x8267, 0x4223, 0x4113, 0x8337, 0, 0, 0],
    [0x8157, 0x8337, 0x4223, 0x6201, 0x3304, 0x1326, 0x8267, 0, 0, 0, 0, 0],
    [0x5102, 0x1326, 0x8267, 0x8157, 0x8337, 0x4223, 0, 0, 0, 0, 0, 0],
    [0x8267, 0x8157, 0x2315, 0x3304, 0x5102, 0x4113, 0x8337, 0, 0, 0, 0, 0],
    [0x6201, 0x4113, 0x8337, 0x8267, 0x8157, 0x2315, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x3304, 0x5102, 0x8337, 0x8267, 0x8157, 0, 0, 0, 0, 0, 0],
    [0x8337, 0x8267, 0x8157, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x8337, 0x8157, 0x8267, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x3304, 0x8337, 0x8157, 0x8267, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x4113, 0x8337, 0x8157, 0x8267, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x3304, 0x2315, 0x4113, 0x8337, 0x8157, 0x8267, 0, 0, 0, 0, 0],
    [0x5102, 0x4223, 0x1326, 0x8337, 0x8157, 0x8267, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x4223, 0x1326, 0x3304, 0x8337, 0x8157, 0x8267, 0, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x4113, 0x5102, 0x4223, 0x1326, 0x8337, 0x8157, 0x8267, 0, 0, 0],
    [0x4223, 0x1326, 0x3304, 0x2315, 0x4113, 0x8337, 0x8157, 0x8267, 0, 0, 0, 0],
    [0x4113, 0x8157, 0x8267, 0x4223, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x4223, 0x4113, 0x8157, 0x8267, 0x6201, 0x5102, 0x3304, 0, 0, 0, 0, 0],
    [0x8157, 0x8267, 0x4223, 0x6201, 0x2315, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x2315, 0x8157, 0x8267, 0x4223, 0x5102, 0, 0, 0, 0, 0, 0],
    [0x1326, 0x5102, 0x4113, 0x8157, 0x8267, 0, 0, 0, 0, 0, 0, 0],
    [0x8157, 0x4113, 0x6201, 0x3304, 0x1326, 0x8267, 0, 0, 0, 0, 0, 0],
    [0x1326, 0x5102, 0x6201, 0x2315, 0x8157, 0x8267, 0, 0, 0, 0, 0, 0],
    [0x8267, 0x1326, 0x3304, 0x2315, 0x8157, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x1146, 0x2245, 0x8337, 0x8157, 0x8267, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x1146, 0x2245, 0x8337, 0x8157, 0x8267, 0, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x4113, 0x3304, 0x1146, 0x2245, 0x8337, 0x8157, 0x8267, 0, 0, 0],
    [0x2315, 0x4113, 0x5102, 0x1146, 0x2245, 0x8337, 0x8157, 0x8267, 0, 0, 0, 0],
    [0x5102, 0x4223, 0x1326, 0x3304, 0x1146, 0x2245, 0x8337, 0x8157, 0x8267, 0, 0, 0],
    [0x1146, 0x2245, 0x6201, 0x4223, 0x1326, 0x8337, 0x8157, 0x8267, 0, 0, 0, 0],
    [0x6201, 0x2315, 0x4113, 0x5102, 0x4223, 0x1326, 0x3304, 0x1146, 0x2245, 0x8337, 0x8157, 0x8267],
    [0x4113, 0x4223, 0x1326, 0x1146, 0x2245, 0x2315, 0x8337, 0x8157, 0x8267, 0, 0, 0],
    [0x4223, 0x4113, 0x8157, 0x8267, 0x3304, 0x1146, 0x2245, 0, 0, 0, 0, 0],
    [0x6201, 0x5102, 0x1146, 0x2245, 0x4223, 0x4113, 0x8157, 0x8267, 0, 0, 0, 0],
    [0x8157, 0x8267, 0x4223, 0x6201, 0x2315, 0x3304, 0x1146, 0x2245, 0, 0, 0, 0],
    [0x2315, 0x8157, 0x8267, 0x4223, 0x5102, 0x1146, 0x2245, 0, 0, 0, 0, 0],
    [0x1326, 0x5102, 0x4113, 0x8157, 0x8267, 0x3304, 0x1146, 0x2245, 0, 0, 0, 0],
    [0x1326, 0x1146, 0x2245, 0x6201, 0x4113, 0x8157, 0x8267, 0, 0, 0, 0, 0],
    [0x5102, 0x6201, 0x2315, 0x8157, 0x8267, 0x1326, 0x3304, 0x1146, 0x2245, 0, 0, 0],
    [0x1326, 0x1146, 0x2245, 0x2315, 0x8157, 0x8267, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x2245, 0x8267, 0x8337, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x2245, 0x8267, 0x8337, 0x6201, 0x5102, 0x3304, 0, 0, 0, 0, 0],
    [0x4113, 0x6201, 0x2245, 0x8267, 0x8337, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4113, 0x8337, 0x8267, 0x2245, 0x3304, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x2245, 0x8267, 0x8337, 0x5102, 0x4223, 0x1326, 0, 0, 0, 0, 0],
    [0x6201, 0x4223, 0x1326, 0x3304, 0x8337, 0x2315, 0x2245, 0x8267, 0, 0, 0, 0],
    [0x4113, 0x6201, 0x2245, 0x8267, 0x8337, 0x5102, 0x4223, 0x1326, 0, 0, 0, 0],
    [0x4113, 0x4223, 0x1326, 0x3304, 0x2245, 0x8267, 0x8337, 0, 0, 0, 0, 0],
    [0x2315, 0x2245, 0x8267, 0x4223, 0x4113, 0, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x2245, 0x8267, 0x4223, 0x4113, 0x6201, 0x5102, 0x3304, 0, 0, 0, 0],
    [0x6201, 0x2245, 0x8267, 0x4223, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x2245, 0x8267, 0x4223, 0x5102, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4113, 0x2315, 0x2245, 0x8267, 0x1326, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x2315, 0x2245, 0x8267, 0x1326, 0x3304, 0x6201, 0, 0, 0, 0, 0],
    [0x5102, 0x6201, 0x2245, 0x8267, 0x1326, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x2245, 0x8267, 0x1326, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x8267, 0x8337, 0x2315, 0x3304, 0x1146, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x1146, 0x8267, 0x8337, 0x2315, 0x6201, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x1146, 0x8267, 0x8337, 0x4113, 0x6201, 0, 0, 0, 0, 0, 0],
    [0x8337, 0x4113, 0x5102, 0x1146, 0x8267, 0, 0, 0, 0, 0, 0, 0],
    [0x8267, 0x8337, 0x2315, 0x3304, 0x1146, 0x5102, 0x4223, 0x1326, 0, 0, 0, 0],
    [0x1146, 0x8267, 0x8337, 0x2315, 0x6201, 0x4223, 0x1326, 0, 0, 0, 0, 0],
    [0x8267, 0x8337, 0x4113, 0x6201, 0x3304, 0x1146, 0x5102, 0x4223, 0x1326, 0, 0, 0],
    [0x4113, 0x4223, 0x1326, 0x1146, 0x8267, 0x8337, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x2315, 0x4113, 0x4223, 0x8267, 0x1146, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x6201, 0x5102, 0x1146, 0x8267, 0x4223, 0x4113, 0, 0, 0, 0, 0],
    [0x1146, 0x8267, 0x4223, 0x6201, 0x3304, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x1146, 0x8267, 0x4223, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x8267, 0x1326, 0x5102, 0x4113, 0x2315, 0x3304, 0x1146, 0, 0, 0, 0, 0],
    [0x6201, 0x4113, 0x2315, 0x1326, 0x1146, 0x8267, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x3304, 0x1146, 0x8267, 0x1326, 0x5102, 0, 0, 0, 0, 0, 0],
    [0x1326, 0x1146, 0x8267, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x1326, 0x8337, 0x8157, 0x1146, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x8337, 0x8157, 0x1146, 0x1326, 0x6201, 0x5102, 0x3304, 0, 0, 0, 0, 0],
    [0x8337, 0x8157, 0x1146, 0x1326, 0x6201, 0x2315, 0x4113, 0, 0, 0, 0, 0],
    [0x4113, 0x5102, 0x3304, 0x2315, 0x1326, 0x8337, 0x8157, 0x1146, 0, 0, 0, 0],
    [0x8337, 0x8157, 0x1146, 0x5102, 0x4223, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x4223, 0x8337, 0x8157, 0x1146, 0x3304, 0, 0, 0, 0, 0, 0],
    [0x8337, 0x8157, 0x1146, 0x5102, 0x4223, 0x6201, 0x2315, 0x4113, 0, 0, 0, 0],
    [0x4223, 0x8337, 0x8157, 0x1146, 0x3304, 0x2315, 0x4113, 0, 0, 0, 0, 0],
    [0x4223, 0x4113, 0x8157, 0x1146, 0x1326, 0, 0, 0, 0, 0, 0, 0],
    [0x4223, 0x4113, 0x8157, 0x1146, 0x1326, 0x6201, 0x5102, 0x3304, 0, 0, 0, 0],
    [0x1146, 0x8157, 0x2315, 0x6201, 0x4223, 0x1326, 0, 0, 0, 0, 0, 0],
    [0x4223, 0x5102, 0x3304, 0x2315, 0x8157, 0x1146, 0x1326, 0, 0, 0, 0, 0],
    [0x4113, 0x8157, 0x1146, 0x5102, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x4113, 0x8157, 0x1146, 0x3304, 0, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x8157, 0x1146, 0x5102, 0x6201, 0, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x8157, 0x1146, 0x3304, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x2245, 0x3304, 0x1326, 0x8337, 0x8157, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x2245, 0x8157, 0x8337, 0x1326, 0x5102, 0, 0, 0, 0, 0, 0],
    [0x2245, 0x3304, 0x1326, 0x8337, 0x8157, 0x6201, 0x2315, 0x4113, 0, 0, 0, 0],
    [0x2245, 0x2315, 0x4113, 0x5102, 0x1326, 0x8337, 0x8157, 0, 0, 0, 0, 0],
    [0x4223, 0x8337, 0x8157, 0x2245, 0x3304, 0x5102, 0, 0, 0, 0, 0, 0],
    [0x8157, 0x2245, 0x6201, 0x4223, 0x8337, 0, 0, 0, 0, 0, 0, 0],
    [0x2245, 0x3304, 0x5102, 0x4223, 0x8337, 0x8157, 0x4113, 0x6201, 0x2315, 0, 0, 0],
    [0x4223, 0x8337, 0x8157, 0x2245, 0x2315, 0x4113, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x8157, 0x2245, 0x3304, 0x1326, 0x4223, 0, 0, 0, 0, 0, 0],
    [0x1326, 0x4223, 0x4113, 0x8157, 0x2245, 0x6201, 0x5102, 0, 0, 0, 0, 0],
    [0x8157, 0x2245, 0x3304, 0x1326, 0x4223, 0x6201, 0x2315, 0, 0, 0, 0, 0],
    [0x5102, 0x1326, 0x4223, 0x2315, 0x8157, 0x2245, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x5102, 0x4113, 0x8157, 0x2245, 0, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x8157, 0x2245, 0x6201, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x6201, 0x2315, 0x8157, 0x2245, 0x3304, 0, 0, 0, 0, 0, 0],
    [0x2315, 0x8157, 0x2245, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x1146, 0x1326, 0x8337, 0x2315, 0x2245, 0, 0, 0, 0, 0, 0, 0],
    [0x1146, 0x1326, 0x8337, 0x2315, 0x2245, 0x6201, 0x5102, 0x3304, 0, 0, 0, 0],
    [0x6201, 0x2245, 0x1146, 0x1326, 0x8337, 0x4113, 0, 0, 0, 0, 0, 0],
    [0x2245, 0x1146, 0x1326, 0x8337, 0x4113, 0x5102, 0x3304, 0, 0, 0, 0, 0],
    [0x5102, 0x1146, 0x2245, 0x2315, 0x8337, 0x4223, 0, 0, 0, 0, 0, 0],
    [0x1146, 0x3304, 0x6201, 0x4223, 0x8337, 0x2315, 0x2245, 0, 0, 0, 0, 0],
    [0x8337, 0x4113, 0x6201, 0x2245, 0x1146, 0x5102, 0x4223, 0, 0, 0, 0, 0],
    [0x4223, 0x8337, 0x4113, 0x3304, 0x2245, 0x1146, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x2315, 0x2245, 0x1146, 0x1326, 0x4223, 0, 0, 0, 0, 0, 0],
    [0x1146, 0x1326, 0x4223, 0x4113, 0x2315, 0x2245, 0x6201, 0x5102, 0x3304, 0, 0, 0],
    [0x1326, 0x4223, 0x6201, 0x2245, 0x1146, 0, 0, 0, 0, 0, 0, 0],
    [0x4223, 0x5102, 0x3304, 0x2245, 0x1146, 0x1326, 0, 0, 0, 0, 0, 0],
    [0x2245, 0x1146, 0x5102, 0x4113, 0x2315, 0, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x2315, 0x2245, 0x1146, 0x3304, 0x6201, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x2245, 0x1146, 0x5102, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x2245, 0x1146, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x1326, 0x8337, 0x2315, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x1326, 0x8337, 0x2315, 0x6201, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x3304, 0x1326, 0x8337, 0x4113, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x1326, 0x8337, 0x4113, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x4223, 0x8337, 0x2315, 0x3304, 0x5102, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x4223, 0x8337, 0x2315, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x5102, 0x4223, 0x8337, 0x4113, 0x6201, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x4223, 0x8337, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x4113, 0x2315, 0x3304, 0x1326, 0x4223, 0, 0, 0, 0, 0, 0, 0],
    [0x1326, 0x4223, 0x4113, 0x2315, 0x6201, 0x5102, 0, 0, 0, 0, 0, 0],
    [0x3304, 0x1326, 0x4223, 0x6201, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x1326, 0x4223, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x5102, 0x4113, 0x2315, 0x3304, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x4113, 0x2315, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0x6201, 0x3304, 0x5102, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]
])

FLIP_CELL_CLASS_1 = np.array([
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0]
])

# FLIP_CELL_CLASS_1 = np.array([
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0]
# ])

# FLIP_CELL_CLASS_2 = np.array([
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0]
# ])

# FLIP_CELL_CLASS_2 = np.array([
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0]
# ])

FLIP_CELL_CLASS_2 = np.array([
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0]
])

# FLIP_CELL_CLASS_2 = np.array([
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0]
# ])

# FLIP_CELL_CLASS_4 = np.array([
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0],
#     [0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0],
#     [1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 1, 1, 0],
#     [1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 1, 1, 0],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0]
# ])

# FLIP_CELL_CLASS_4 = np.array([
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1],
#     [0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1],
#     [0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1],
#     [0, 0, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]
# ])

FLIP_CELL_CLASS_4 = np.array([
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1],
    [0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1],
    [0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1],
    [0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1],
    [0, 0, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]
])

# FLIP_CELL_CLASS_4 = np.array([
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0]
# ])

# FLIP_CELL_CLASS_8 = np.array([
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]
# ])

# FLIP_CELL_CLASS_8 = np.array([
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0],
#     [1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0],
#     [1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0],
#     [1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0],
#     [1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0],
#     [1, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0],
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1]
# ])

FLIP_CELL_CLASS_8 = np.array([
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    [0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0],
    [1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0],
    [1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0],
    [1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0],
    [1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0],
    [1, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1]
])

# FLIP_CELL_CLASS_8 = np.array([
#     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
#     [0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1],
#     [0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1],
#     [0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1],
#     [0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
#     [0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]
# ])

FLIP_CELL_CLASS_16 = np.array([
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0]
])

# Nibbles (in this instance) are the upper and lower 4 bits of a byte
hi_nibble = lambda b : (((b) >> 4) & 0x0F)
lo_nibble = lambda b : ((b) & 0x0F)

# Pack and unpack functions to store vertex edges for uniqueness checks
# and to establish vertex priority.
# 3 integers (unpack) <--> (pack) single integer.
# def single_pack(a, b, c):
#     res = ((a) << 32) + ((b) << 16) + c
#     #res += ((b) << 16)
#     #res += c

#     return res

# def single_unpack(res):
#     a = (((res) >> 32) & 0xffff)
#     b = (((res) >> 16) & 0xffff)
#     c = ((res) & 0xffff)

#     return a, b, c

# For the multi_pack/unpack we need np.uint64 to get bitshift to work correctly 
# (not even np.uint32 works)

# def multi_pack(arr1, arr2, arr3):
#     arr = ((arr1.astype(np.uint64)) << 32)
#     arr += ((arr2.astype(np.uint64)) << 16)
#     arr += arr3.astype(np.uint64)
#
#     return arr
#
# def multi_unpack(arr):
#     a = (((arr.astype(np.uint64)) >> 32) & 0xffff)
#     b = (((arr.astype(np.uint64)) >> 16) & 0xffff)
#     c = ((arr.astype(np.uint64)) & 0xffff)
#
#     return a, b, c


# def multi_pack(arr1, arr2, arr3):
#     arr = ((arr1.astype(np.uint64)) << 42)
#     arr += ((arr2.astype(np.uint64)) << 21)
#     arr += arr3.astype(np.uint64)
    
#     return arr


# def multi_unpack(arr):
#     a = (((arr.astype(np.uint64)) >> 42) & 0x1FFFFF)
#     b = (((arr.astype(np.uint64)) >> 21) & 0x1FFFFF)
#     c = ((arr.astype(np.uint64)) & 0x1FFFFF)
    
#     return a, b, c

# Convert from marching cubes indexing to modified marching cubes
# MC_MAP = [0, 1, 3, 2, 4, 5, 7, 6]

class ModifiedMarchingCubes(object):
    """
    Base class for modified marching cubes. Creates a surface triangulation from a set of 
    voxels, represented as 8 vertices each.

    References
    ----------
        * https://www.volume-gfx.com/volume-rendering/marching-cubes/
        * http://paulbourke.net/geometry/polygonise/
        * http://transvoxel.org/Lengyel-VoxelTerrain.pdf
    """

    _MC_MAP_MODE_MODIFIED = True #allow us to switch between this and standard marching cubes easily

    # Datatype for storing triangles created by marching cubes. Mimics STL data structure.
    # TODO: Add ('normal', '3f4') for real
    dt = np.dtype([('normal', '3f4'), ('vertex0', '3f4'), ('vertex1', '3f4'),
                        ('vertex2', '3f4'), ('attrib', 'u2')])

    def __init__(self, isolevel=0):
        """
        Initialize the marching cubes algorithm

        Parameters
        ----------
        isolevel : int
            Threshold determining if vertex lies inside or outside the surface.
        """
        self.vertices = None  # A (-1,8,3) set of x,y,z coordinates corresponding to cubes
        self.values = None  # The corresponding vertex values (-1, 8)
        self.triangles = None  # The list of triangles
        self.isolevel = isolevel

        # self._edge_indices = None

    def cube_index(self, values):
        """
        Takes in a set of 8 vertex values (values of v0-v7) and determines if each is above or below the isolevel.

        Parameters
        ----------
        values : np.array
            Values at the 8 vertices v0-v7 corresponding to if the box is inside/outside the volume.

        Returns
        -------
        indices: int
            Value to use for lookup in MC_EDGES, MC_TRIANGLES.
        """

        return ((values < self.isolevel) * EDGE_BITMASK).sum(1) #[EDGE_BITMASK[x] for x in MC_MAP]).sum(1)

    def interpolate_vertex(self, v0, v1, v0_value, v1_value, i=None, j0=None, j1=None):
        """
        Interpolate triangle vertex along edge formed by v0->v1.

        Parameters
        ----------
        v0, v1 :
            Vertices of edge v0->v1.
        v0_value, v1_value:
            Scalar values at vertices v0, v1. Same values as for vertex_values in edge_index().

        Returns
        -------
        Interpolated vertex of a triangle.
        """
        # Interpolate along the edge v0 -> v1
        mu = 1.*(self.isolevel - v0_value) / (v1_value - v0_value)
        p = v0 + mu[:, None]*(v1-v0)
        
        #if np.any(mu < 0):
        #    print('mu < 0', mu[mu < 0])
            
        #if np.any(mu > 1):
        #    print('mu > 1', mu[(mu > 1)*(mu < 1e6)])

        # Are v0 and v1 the same vertex? (common in dual marching cubes)
        # If so, choose v0 as the triangle vertex position.
        idxs = (np.abs(v1_value - v0_value) < 1e-12)#self.isolevel)
        p[idxs, :] = v0[idxs, :]

        return p


    def create_intersection_list(self, cube_index, vertices, values):
        """
        Creates a vertex for each of the 12 edges based on edge_index values.

        Parameters
        ----------
        cube_index : int
            MC_EDGES[cube_index]
        vertices : np.array
            Cube vertices.
        values : np.array
            Cube vertex values.

        Returns
        -------
        vertices of intersection list
        """

        # Find the voxel edges on which our triangle vertices lie
        interpolation_vertices = REGULAR_VERTEX_DATA[cube_index,:]
        
        # Corners
        c0 = hi_nibble(interpolation_vertices)
        c1 = lo_nibble(interpolation_vertices)

        # Construct vertex indices storing an edge in a single number
        #v_idxs = []
        
        # for i in range(c0.shape[0]):
        #     for j in range(len(c0[i,:])):
        #         e0 = c0[i,j]  # MC_MAP[c0[i,j]]
        #         e1 = c1[i,j]  # MC_MAP[c1[i,j]]
        #
        #         # DB: below sorting doesn't appear necessary
        #         # v0 = vertices[i,e0,:]
        #         # v1 = vertices[i,e1,:]
        #         #
        #         # # Ordered
        #         # if (v0[0] > v1[0]) or (v0[1] > v1[1]) or (v0[2] > v1[2]):
        #         #     tmp = e0
        #         #     e0 = e1
        #         #     e1 = tmp
        #
        #         v_idx = single_pack(i, e0, e1)
        #         v_idxs.append(v_idx)
        #
        # v_idxs = np.vstack(v_idxs).squeeze()
        
        
        #sort
        c0_ = np.minimum(c0, c1)
        c1_ = np.maximum(c0, c1)

        # v_idxs = multi_pack((np.arange(c0.shape[0])[:, None]*np.ones(c0.shape[1])[None,:]).astype('uint64'), c0_, c1_).ravel()
        v_idxs = np.vstack([(np.arange(c0.shape[0])[:, None]*np.ones(c0.shape[1])[None,:]).ravel(), c0_.ravel(), c1_.ravel()]).T

        # Grab the unique vertices and calculate edges
        # edges_unique, edges_inverse = np.unique(v_idxs, return_inverse=True)
        edges_unique, edges_inverse = np.unique(v_idxs, return_inverse=True, axis=0)

        # Store these for manifold check later
        # self._edge_indices = v_idxs.reshape(-1, 12, order='F')

        
        # i, j0, j1 = multi_unpack(edges_unique)
        i, j0, j1 = edges_unique.T.astype('uint32')

        v0 = vertices[i, j0, :]
        v1 = vertices[i, j1, :]
        v0_value = values[i, j0]
        v1_value = values[i, j1]

        # Calculate each edge once
        p = self.interpolate_vertex(v0, v1, v0_value, v1_value, i, j0, j1)

        # Reshape the calculated vertices to match the shape we'll index into later
        pr = p[edges_inverse].reshape(-1, 12, 3, order='C')


        return pr

    def create_triangles(self, intersections, triangles):
        """
        Connect the intersection list into triangles.

        Parameters
        ----------
        intersections : np.array
            Array of edge intersection vertices.
        triangles : np.array
            Array of edge indices to form triangles.

        Returns
        -------
        None
        """

        # Get rid of -1 values in the lookup table
        idxs = np.array(np.where(triangles[:, 0] != -1)).reshape(-1)
        triangles_pruned = triangles[idxs].flatten()
        idxs = np.repeat(idxs, triangles.shape[1])[triangles_pruned != -1]
        triangles_pruned = triangles_pruned[triangles_pruned != -1]

        triangles_returned = intersections[idxs, triangles_pruned].reshape(int(idxs.shape[0] / 3), 3, 3)
        normals = np.cross((triangles_returned[:, 2] - triangles_returned[:, 1]),
                           (triangles_returned[:, 0] - triangles_returned[:, 1]))

        # Prune the 0-area triangles
        nn = np.linalg.norm(normals, axis=1)
        triangles_returned = triangles_returned[(nn > 0), :]
        normals = normals[(nn > 0), :]

        # Store triangles in STL format
        triangles_stl = np.zeros(triangles_returned.shape[0], dtype=self.dt)
        triangles_stl['vertex0'] = triangles_returned[:, 0, :]
        triangles_stl['vertex1'] = triangles_returned[:, 1, :]
        triangles_stl['vertex2'] = triangles_returned[:, 2, :]
        triangles_stl['normal'] = normals


        self.triangles = triangles_stl

        return self.triangles

    def set_vertices(self, vertices, values):
        """
        Set input vertices (self.vertices) & set values (self.values) based on input vertices to use for isosurface
        extraction.

        * self.vertices is a (-1,8,3) set of x,y,z coordinates corresponding to cubes
        * self.values are the corresponding vertex values (-1, 8)

        overridden in derived classes
        """

        self.vertices = vertices
        self.values = values

    def export_triangles(self):
        """
        Return the list of triangles as a dtyped np array for use in triangular rendering.

        Returns
        -------
        triangles : np.array
            Array of triangles of type self.dt
        """

        return self.triangles

    # def manifold_check(self, find_class, invert_class, cell_class, cube_index):
    #         # Manifold check: If two adjacent edges share adjacent cell classes that result in 4 
    #         # triangles sharing a single edge, we need to swap one of the cell classes so the mesh 
    #         # becomes manifold. This follows the procedure from R. Wenger, Isosurfaces: Geometry, 
    #         # Topology, and Algorithms, CRC Press, 2013.

    #         # Find invert_class cells sharing an edge with a find_class cell
    #         class_find = (cell_class == find_class)
    #         class_invert = (cell_class == invert_class)
    #         class_find_edges = self._edge_indices[class_find].ravel()
    #         class_invert_edges = self._edge_indices[class_invert].ravel()
    #         class_common_edges = np.array(list(set(list(class_find_edges)).intersection(list(class_invert_edges))))
            
    #         # If we found any...
    #         if len(class_common_edges) > 0:
    #             common_edge_bool = (self._edge_indices == class_common_edges[:, None, None])
    #             inds_adj = np.where(common_edge_bool)[1]
    #             cells_adj = inds_adj #[cell_class[inds_adj] == invert_class]
                
    #             # Invert the cube code of the cell of replace_class
    #             cube_index[cells_adj] = ((cube_index[cells_adj]) ^ (0xFF))
    #             cell_class[cells_adj] = REGULAR_CELL_CLASS[cube_index[cells_adj]]

    #         return cell_class, cube_index


    def num_shared_vertices(self):
        a = (((self.vertices[None,:,None,:] == self.vertices[:,None,:,None]).sum(4) == 3).sum(2) > 0).sum(2)
        np.fill_diagonal(a,0)  # We don't consider the same cell to be next to itself
        #print(a)
        return a

    def march(self, return_triangles=True, dual_march=False):
        """
        March over the input vertices.

        Parameters
        ----------
        return_triangles : bool
            Return the list of triangles post-march or nah?

        Returns
        -------
        Optionally, an array of triangles after marching.
        """

        # Classify this voxel
        cube_index = self.cube_index(self.values)
        mask = ~((cube_index == 0) | (cube_index == 0xFF))
        
        verts, vals = self.vertices, self.values
        
        if False:#not dual_march:
            # FIXME - disabled for now - this fails when we extract from an octree and do piecewise-linear interpolation
            # between vertices as the piecewise linear interpolation function accesses a non-masked depth array.
            
            # only crop to index mask if we are not doing a dual march
            cube_index= cube_index[mask]
            verts, vals = verts[mask,:, :], vals[mask,:]

        # Find the edges we wish the triangles to intersect
        intersections = self.create_intersection_list(cube_index, verts, vals)

        # Get this voxel's equivalence class
        cell_class = REGULAR_CELL_CLASS[cube_index]
        # print(cell_class)
        # print(self.values >= self.isolevel)

        if dual_march:
            # Invert instance where cell_class 1 is next to cell class 13
            # cell_class, cube_index = self.manifold_check(0x01, 0x0D, cell_class, cube_index)

            # cell_class, cube_index = self.manifold_check(0x00, 0x04, cell_class, cube_index)

            shared_vertices = self.num_shared_vertices()
            shared_indices_1 = np.where(shared_vertices == 1)
            shared_indices_2 = np.where(shared_vertices == 2)
            shared_indices_4 = np.where(shared_vertices == 4)
            shared_indices_8 = np.where(shared_vertices == 8)
            shared_indices_16 = np.where(shared_vertices == 16)

            to_flip = []

            #to_flip.extend(shared_indices_1[0]*(FLIP_CELL_CLASS_1[cell_class[shared_indices_1[0]],cell_class[shared_indices_1[1]]] == 1))
            #to_flip.extend(shared_indices_2[0]*(FLIP_CELL_CLASS_2[cell_class[shared_indices_2[0]],cell_class[shared_indices_2[1]]] == 1))
            to_flip.extend(shared_indices_4[0]*(FLIP_CELL_CLASS_4[cell_class[shared_indices_4[0]],cell_class[shared_indices_4[1]]] == 1))
            #to_flip.extend(shared_indices_8[0]*(FLIP_CELL_CLASS_8[cell_class[shared_indices_8[0]],cell_class[shared_indices_8[1]]] == 1))
            #to_flip.extend(shared_indices_16[0]*(FLIP_CELL_CLASS_16[cell_class[shared_indices_16[0]],cell_class[shared_indices_16[1]]] == 1))

            to_flip = np.unique(to_flip)

            # Invert the cube code of the cell of replace_class
            cube_index[to_flip] = ((cube_index[to_flip]) ^ (0xFF))
            cell_class[to_flip] = REGULAR_CELL_CLASS[cube_index[to_flip]]

            # Redo the intersections
            intersections = self.create_intersection_list(cube_index, self.vertices, self.values)
             
        # Grab triangle draw order
        cell_data = REGULAR_CELL_DATA[cell_class]
        # Create the triangles
        self.create_triangles(intersections, cell_data)

        # Do we want this function to kick the triangles back to us?
        if return_triangles:
            return self.export_triangles()


class RasterMarchingCubes(ModifiedMarchingCubes):
    """
    Marching cubes with some optimisations data on a regular grid
    """
    
    v_offsets = np.array([[0, 0, 0],
                          [1, 0, 0],
                          [0, 1, 0],
                          [1, 1, 0],
                          [0, 0, 1],
                          [1, 0, 1],
                          [0, 1, 1],
                          [1, 1, 1]])
    
    def __init__(self, image, isolevel=0, voxelsize=(1., 1., 1.)):
        """
        Initialize the marching cubes algorithm

        Parameters
        ----------
        isolevel : int
            Threshold determining if vertex lies inside or outside the surface.
        """
        self.image = image.squeeze().astype('f')
        self.voxelsize = voxelsize
        self.triangles = None  # The list of triangles
        self.isolevel = isolevel
    
    def cube_index(self):
        """
        Takes in a set of 8 vertex values (values of v0-v7) and determines if each is above or below the isolevel.

        Parameters
        ----------
        values : np.array
            Values at the 8 vertices v0-v7 corresponding to if the box is inside/outside the volume.

        Returns
        -------
        indices: int
            Value to use for lookup in MC_EDGES, MC_TRIANGLES.
        """
        
        imm = self.image < self.isolevel
        
        res = (imm[:-1, :-1, :-1] << 0) + \
              (imm[1:, :-1, :-1] << 1) + \
              (imm[:-1, 1:, :-1] << 2) + \
              (imm[1:, 1:, :-1] << 3) + \
              (imm[:-1, :-1, 1:] << 4) + \
              (imm[1:, :-1, 1:] << 5) + \
              (imm[:-1, 1:, 1:] << 6) + \
              (imm[1:, 1:, 1:] << 7)
        
        return res
    
    def gen_vertices_and_vals(self, cube_mask):
        coords = np.argwhere(cube_mask)[:, None, :] + self.v_offsets[None, :, :]
        
        values = self.image[coords[:, :, 0], coords[:, :, 1], coords[:, :, 2]]
        
        #print(coords.shape, values.shape)
        
        return coords.astype('f') * np.array(self.voxelsize)[None, None, :], values

    def march(self, return_triangles=True, dual_march=False):
        """
        March over the input vertices.

        Parameters
        ----------
        return_triangles : bool
            Return the list of triangles post-march or nah?

        Returns
        -------
        Optionally, an array of triangles after marching.
        """
    
        # Classify this voxel
        cube_index = self.cube_index()

        #we don't need to calculate anything if all vertices of a cube are above or below the threshold
        mask = ~((cube_index == 0) | (cube_index == 0xFF))
        cube_index = cube_index[mask].ravel()
    
        # Find the edges we wish the triangles to intersect
        intersections = self.create_intersection_list(cube_index, *self.gen_vertices_and_vals(mask))
    
        # Get this voxel's equivalence class
        cell_class = REGULAR_CELL_CLASS[cube_index]
        
    
        # Grab triangle draw order
        cell_data = REGULAR_CELL_DATA[cell_class]
        # Create the triangles
        self.create_triangles(intersections, cell_data)
    
        # Do we want this function to kick the triangles back to us?
        if return_triangles:
            return self.export_triangles()
    
    
