class ProgressLog:
    """Class to store the improvement of lower
    and upper bounds over time during the search.
    Results stored here are useful to analyze the
    performance of a given formulation/parameter setting
    for solving a instance. To be able to automatically
    generate summarized experimental results, fill the
    :attr:`~mip.ProgressLog.instance` and
    :attr:`~mip.ProgressLog.settings` of this object with the instance
    name and formulation/parameter setting details, respectively.

    Attributes:

        log(List[Tuple[float, Tuple[float, float]]]): List of tuples in the format
            :math:`(time, (lb, ub))`, where :math:`time` is the processing time
            in seconds and :math:`lb` and :math:`ub` are the lower and upper bounds,
            respectively

        instance(str): instance name

        settings(str): identification of the formulation/parameter
            settings used in the optimization (whatever is relevant to
            identify a given computational experiment)
    """

    def __init__(self):
        self.log = []

        self.instance = ""

        self.settings = ""

    def write(self, file_name: str = ""):
        """Saves the progress log. If no extension is informed,
        the :code:`.plog` extension will be used. If only a directory is
        informed then the name will be built considering the
        :attr:`~mip.ProgressLog.instance` and
        :attr:`~mip.ProgressLog.settings` attributes"""
        if not self.instance:
            raise ValueError(
                "Enter model name (instance name) to save experimental data."
            )
        if not file_name:
            file_name = f"{self.instance}-{self.settings}.plog"
        else:
            if file_name.endswith("/") or file_name.endswith("\\"):
                file_name += f"{self.instance}-{self.settings}.plog"

        if not file_name.endswith(".plog"):
            file_name += ".plog"

        with open(file_name, "w") as f:
            f.write(f"instance: {self.instance}")
            f.write(f"settings: {self.settings}")
            for s, (log_s, log_b) in self.log:
                f.write(f"{s},{log_s},{log_b}")

    def read(self, file_name: str):
        """Reads a progress log stored in a file"""
        with open(file_name) as f:
            line = next(f)
            self.instance = line.split(":")[1].lstrip()
            self.settings = line.split(":")[1].lstrip()
            for line in f:
                cols = line.split(",")
                (s, (log_s, log_b)) = (
                    float(cols[0]),
                    (float(cols[1]), float(cols[2])),
                )
                self.log.append((s, (log_s, log_b)))
