from pydantic import BaseModel, ValidationError
from typing import Union, Optional, List, Any

from moloni.base.client import MoloniBaseClient
from moloni.base.helpers import endpoint, fill_query_params, validate_data
from moloni.base import ApiResponse


class ApiRequestModel(BaseModel):
    _api_client: Any = None

    def connect(self, *args, **kwargs):
        self._api_client = MaturitydatesClient(*args, **kwargs)
        return self

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        pass


class MaturitydatesCountModifiedSinceModel(ApiRequestModel):
    company_id: Union[str, int]
    lastmodified: Optional[str] = None

    def request(self) -> ApiResponse:
        """
        request(self) -> ApiResponse

        Make an API request using the initialized client.

        This method checks if the `_api_client` attribute is set (i.e., if the client has been initialized via the `connect` method).
        If the client is initialized, it will make an API request using the provided method name and the model's data,
        excluding the `_api_client` attribute itself from the request payload. If the client is not initialized, it will raise a `ValueError`.

        Returns:
            The response from the API.

        Raises:
            ValueError: If the client is not initialized via the `connect` method.

        Example:
            # Assuming you have a model instance `request_model` and an API client `api_client`
            with request_model.connect(auth_config=auth_config) as api:
                response = api.request()

            # The above example assumes that the `connect` method has been used to initialize the client.
            # The request method then sends the model's data to the API and returns the API's response.
        """
        if hasattr(self, "_api_client"):
            response = self._api_client.count_modified_since(
                self.model_dump(exclude={"_api_client"}, exclude_unset=True)
            )
            return response
        else:
            raise ValueError("Client not initialized. Use the 'connect' method.")


class MaturitydatesDeleteModel(ApiRequestModel):
    company_id: Union[str, int]
    maturity_date_id: Optional[Union[str, int]] = None

    def request(self) -> ApiResponse:
        """
        request(self) -> ApiResponse

        Make an API request using the initialized client.

        This method checks if the `_api_client` attribute is set (i.e., if the client has been initialized via the `connect` method).
        If the client is initialized, it will make an API request using the provided method name and the model's data,
        excluding the `_api_client` attribute itself from the request payload. If the client is not initialized, it will raise a `ValueError`.

        Returns:
            The response from the API.

        Raises:
            ValueError: If the client is not initialized via the `connect` method.

        Example:
            # Assuming you have a model instance `request_model` and an API client `api_client`
            with request_model.connect(auth_config=auth_config) as api:
                response = api.request()

            # The above example assumes that the `connect` method has been used to initialize the client.
            # The request method then sends the model's data to the API and returns the API's response.
        """
        if hasattr(self, "_api_client"):
            response = self._api_client.delete(
                self.model_dump(exclude={"_api_client"}, exclude_unset=True)
            )
            return response
        else:
            raise ValueError("Client not initialized. Use the 'connect' method.")


class MaturitydatesGetAllModel(ApiRequestModel):
    company_id: Union[str, int]

    def request(self) -> ApiResponse:
        """
        request(self) -> ApiResponse

        Make an API request using the initialized client.

        This method checks if the `_api_client` attribute is set (i.e., if the client has been initialized via the `connect` method).
        If the client is initialized, it will make an API request using the provided method name and the model's data,
        excluding the `_api_client` attribute itself from the request payload. If the client is not initialized, it will raise a `ValueError`.

        Returns:
            The response from the API.

        Raises:
            ValueError: If the client is not initialized via the `connect` method.

        Example:
            # Assuming you have a model instance `request_model` and an API client `api_client`
            with request_model.connect(auth_config=auth_config) as api:
                response = api.request()

            # The above example assumes that the `connect` method has been used to initialize the client.
            # The request method then sends the model's data to the API and returns the API's response.
        """
        if hasattr(self, "_api_client"):
            response = self._api_client.get_all(
                self.model_dump(exclude={"_api_client"}, exclude_unset=True)
            )
            return response
        else:
            raise ValueError("Client not initialized. Use the 'connect' method.")


class MaturitydatesGetModifiedSinceModel(ApiRequestModel):
    company_id: Union[str, int]
    lastmodified: Optional[str] = None

    def request(self) -> ApiResponse:
        """
        request(self) -> ApiResponse

        Make an API request using the initialized client.

        This method checks if the `_api_client` attribute is set (i.e., if the client has been initialized via the `connect` method).
        If the client is initialized, it will make an API request using the provided method name and the model's data,
        excluding the `_api_client` attribute itself from the request payload. If the client is not initialized, it will raise a `ValueError`.

        Returns:
            The response from the API.

        Raises:
            ValueError: If the client is not initialized via the `connect` method.

        Example:
            # Assuming you have a model instance `request_model` and an API client `api_client`
            with request_model.connect(auth_config=auth_config) as api:
                response = api.request()

            # The above example assumes that the `connect` method has been used to initialize the client.
            # The request method then sends the model's data to the API and returns the API's response.
        """
        if hasattr(self, "_api_client"):
            response = self._api_client.get_modified_since(
                self.model_dump(exclude={"_api_client"}, exclude_unset=True)
            )
            return response
        else:
            raise ValueError("Client not initialized. Use the 'connect' method.")


class MaturitydatesInsertModel(ApiRequestModel):
    company_id: Union[str, int]
    associated_discount: Optional[str] = None
    days: Optional[str] = None
    name: Optional[str] = None

    def request(self) -> ApiResponse:
        """
        request(self) -> ApiResponse

        Make an API request using the initialized client.

        This method checks if the `_api_client` attribute is set (i.e., if the client has been initialized via the `connect` method).
        If the client is initialized, it will make an API request using the provided method name and the model's data,
        excluding the `_api_client` attribute itself from the request payload. If the client is not initialized, it will raise a `ValueError`.

        Returns:
            The response from the API.

        Raises:
            ValueError: If the client is not initialized via the `connect` method.

        Example:
            # Assuming you have a model instance `request_model` and an API client `api_client`
            with request_model.connect(auth_config=auth_config) as api:
                response = api.request()

            # The above example assumes that the `connect` method has been used to initialize the client.
            # The request method then sends the model's data to the API and returns the API's response.
        """
        if hasattr(self, "_api_client"):
            response = self._api_client.insert(
                self.model_dump(exclude={"_api_client"}, exclude_unset=True)
            )
            return response
        else:
            raise ValueError("Client not initialized. Use the 'connect' method.")


class MaturitydatesUpdateModel(ApiRequestModel):
    company_id: Union[str, int]
    associated_discount: Optional[str] = None
    days: Optional[str] = None
    maturity_date_id: Optional[Union[str, int]] = None
    name: Optional[str] = None

    def request(self) -> ApiResponse:
        """
        request(self) -> ApiResponse

        Make an API request using the initialized client.

        This method checks if the `_api_client` attribute is set (i.e., if the client has been initialized via the `connect` method).
        If the client is initialized, it will make an API request using the provided method name and the model's data,
        excluding the `_api_client` attribute itself from the request payload. If the client is not initialized, it will raise a `ValueError`.

        Returns:
            The response from the API.

        Raises:
            ValueError: If the client is not initialized via the `connect` method.

        Example:
            # Assuming you have a model instance `request_model` and an API client `api_client`
            with request_model.connect(auth_config=auth_config) as api:
                response = api.request()

            # The above example assumes that the `connect` method has been used to initialize the client.
            # The request method then sends the model's data to the API and returns the API's response.
        """
        if hasattr(self, "_api_client"):
            response = self._api_client.update(
                self.model_dump(exclude={"_api_client"}, exclude_unset=True)
            )
            return response
        else:
            raise ValueError("Client not initialized. Use the 'connect' method.")


class MaturitydatesClient(MoloniBaseClient):

    @endpoint("/<version>/maturityDates/countModifiedSince/", method="post")
    def count_modified_since(
        self, data: Union[MaturitydatesCountModifiedSinceModel, dict], **kwargs
    ):
        """
        count_modified_since(self, data: Union[MaturitydatesCountModifiedSinceModel, dict], **kwargs)

        Args:

            data (Union[MaturitydatesCountModifiedSinceModel, dict]): A model instance or dictionary containing the following fields:

                - company_id (Union[str, int]): company_id of the MaturitydatesCountModifiedSinceModel.

                - lastmodified (str): lastmodified of the MaturitydatesCountModifiedSinceModel.



        Returns:
            ApiResponse: The response from the API.
        """

        data = validate_data(data, self.validate, MaturitydatesCountModifiedSinceModel)

        return self._request(
            fill_query_params(kwargs.pop("path"), self.version), data={**data, **kwargs}
        )

    @endpoint("/<version>/maturityDates/delete/", method="post")
    def delete(self, data: Union[MaturitydatesDeleteModel, dict], **kwargs):
        """
        delete(self, data: Union[MaturitydatesDeleteModel, dict], **kwargs)

        Args:

            data (Union[MaturitydatesDeleteModel, dict]): A model instance or dictionary containing the following fields:

                - company_id (Union[str, int]): company_id of the MaturitydatesDeleteModel.

                - maturity_date_id (Union[str, int]): maturity_date_id of the MaturitydatesDeleteModel.



        Returns:
            ApiResponse: The response from the API.
        """

        data = validate_data(data, self.validate, MaturitydatesDeleteModel)

        return self._request(
            fill_query_params(kwargs.pop("path"), self.version), data={**data, **kwargs}
        )

    @endpoint("/<version>/maturityDates/getAll/", method="post")
    def get_all(self, data: Union[MaturitydatesGetAllModel, dict], **kwargs):
        """
        get_all(self, data: Union[MaturitydatesGetAllModel, dict], **kwargs)

        Args:

            data (Union[MaturitydatesGetAllModel, dict]): A model instance or dictionary containing the following fields:

                - company_id (Union[str, int]): company_id of the MaturitydatesGetAllModel.



        Returns:
            ApiResponse: The response from the API.
        """

        data = validate_data(data, self.validate, MaturitydatesGetAllModel)

        return self._request(
            fill_query_params(kwargs.pop("path"), self.version), data={**data, **kwargs}
        )

    @endpoint("/<version>/maturityDates/getModifiedSince/", method="post")
    def get_modified_since(
        self, data: Union[MaturitydatesGetModifiedSinceModel, dict], **kwargs
    ):
        """
        get_modified_since(self, data: Union[MaturitydatesGetModifiedSinceModel, dict], **kwargs)

        Args:

            data (Union[MaturitydatesGetModifiedSinceModel, dict]): A model instance or dictionary containing the following fields:

                - company_id (Union[str, int]): company_id of the MaturitydatesGetModifiedSinceModel.

                - lastmodified (str): lastmodified of the MaturitydatesGetModifiedSinceModel.



        Returns:
            ApiResponse: The response from the API.
        """

        data = validate_data(data, self.validate, MaturitydatesGetModifiedSinceModel)

        return self._request(
            fill_query_params(kwargs.pop("path"), self.version), data={**data, **kwargs}
        )

    @endpoint("/<version>/maturityDates/insert/", method="post")
    def insert(self, data: Union[MaturitydatesInsertModel, dict], **kwargs):
        """
        insert(self, data: Union[MaturitydatesInsertModel, dict], **kwargs)

        Args:

            data (Union[MaturitydatesInsertModel, dict]): A model instance or dictionary containing the following fields:

                - associated_discount (str): associated_discount of the MaturitydatesInsertModel.

                - company_id (Union[str, int]): company_id of the MaturitydatesInsertModel.

                - days (str): days of the MaturitydatesInsertModel.

                - name (str): name of the MaturitydatesInsertModel.



        Returns:
            ApiResponse: The response from the API.
        """

        data = validate_data(data, self.validate, MaturitydatesInsertModel)

        return self._request(
            fill_query_params(kwargs.pop("path"), self.version), data={**data, **kwargs}
        )

    @endpoint("/<version>/maturityDates/update/", method="post")
    def update(self, data: Union[MaturitydatesUpdateModel, dict], **kwargs):
        """
        update(self, data: Union[MaturitydatesUpdateModel, dict], **kwargs)

        Args:

            data (Union[MaturitydatesUpdateModel, dict]): A model instance or dictionary containing the following fields:

                - associated_discount (str): associated_discount of the MaturitydatesUpdateModel.

                - company_id (Union[str, int]): company_id of the MaturitydatesUpdateModel.

                - days (str): days of the MaturitydatesUpdateModel.

                - maturity_date_id (Union[str, int]): maturity_date_id of the MaturitydatesUpdateModel.

                - name (str): name of the MaturitydatesUpdateModel.



        Returns:
            ApiResponse: The response from the API.
        """

        data = validate_data(data, self.validate, MaturitydatesUpdateModel)

        return self._request(
            fill_query_params(kwargs.pop("path"), self.version), data={**data, **kwargs}
        )
