import pandas as pd

from nakb import NAKBClient


# Helper: fake JSON Solr response
def fake_solr_response(docs=None):
    return {"response": {"docs": docs or []}}


def test_query_raw_basic(requests_mock):
    url = "https://nakb.org/node/solr/nakb/select"

    docs = [{"pdbid": "1A1H", "entity_id": 1}]
    requests_mock.get(url, json=fake_solr_response(docs))

    client = NAKBClient()
    df = client.query_raw(rows=5)

    assert isinstance(df, pd.DataFrame)
    assert df.loc[0, "pdbid"] == "1A1H"


def test_query_raw_with_fq_and_fl(requests_mock):
    url = "https://nakb.org/node/solr/nakb/select"

    docs = [{"pdbid": "2XYZ", "chain_id": "A"}]
    requests_mock.get(url, json=fake_solr_response(docs))

    client = NAKBClient()
    df = client.query_raw(
        q="*:*",
        fl="pdbid,chain_id",
        fq=["pdbid:2XYZ", "chain_id:A"],
        rows=1,
    )

    # verify result
    assert df.loc[0, "pdbid"] == "2XYZ"

    # verify URL params sent
    req = requests_mock.request_history[0]
    assert req.qs["fl"][0] == "pdbid,chain_id"
    assert [v.lower() for v in req.qs["fq"]] == ["pdbid:2xyz", "chain_id:a"]


def test_query_exact_match(requests_mock):
    url = "https://nakb.org/node/solr/nakb/select"

    docs = [{"pdbid": "3AAA"}]
    requests_mock.get(url, json=fake_solr_response(docs))

    client = NAKBClient()
    df = client.query(pdbid="3AAA")

    # verify fq generated
    req = requests_mock.request_history[0]
    assert req.qs["fq"][0].lower() == 'pdbid:"3aaa"'

    assert df.loc[0, "pdbid"].lower() == "3aaa"


def test_query_contains(requests_mock):
    url = "https://nakb.org/node/solr/nakb/select"

    docs = [{"description": "DNA binding protein"}]
    requests_mock.get(url, json=fake_solr_response(docs))

    client = NAKBClient()
    df = client.query(description__contains="DNA")

    req = requests_mock.request_history[0]
    assert req.qs["fq"][0].lower() == "description:*dna*"

    assert "DNA binding" in df.loc[0, "description"]


def test_query_list_or(requests_mock):
    url = "https://nakb.org/node/solr/nakb/select"

    docs = [{"pdbid": "4BBB"}]
    requests_mock.get(url, json=fake_solr_response(docs))

    client = NAKBClient()
    df = client.query(pdbid=["4BBB", "5CCC"])

    req = requests_mock.request_history[0]
    assert req.qs["fq"][0].lower() == 'pdbid:("4bbb" or "5ccc")'

    assert df.loc[0, "pdbid"] == "4BBB"


def test_query_operators_lt_gt_range(requests_mock):
    url = "https://nakb.org/node/solr/nakb/select"

    docs = [{"score": 5}]
    requests_mock.get(url, json=fake_solr_response(docs))

    client = NAKBClient()
    df = client.query(score__lt=10, score__gt=1, score__range=(1, 10))

    req = requests_mock.request_history[0]
    fq = req.qs["fq"]

    assert "score:[* to 10]" in fq
    assert "score:[1 to *]" in fq
    assert "score:[1 to 10]" in fq

    assert df.loc[0, "score"] == 5


def test_query_fields_fl(requests_mock):
    url = "https://nakb.org/node/solr/nakb/select"

    docs = [{"pdbid": "1A1H", "entity_id": 1}]
    requests_mock.get(url, json=fake_solr_response(docs))

    client = NAKBClient()
    df = client.query(fields=["pdbid", "entity_id"])

    req = requests_mock.request_history[0]
    assert req.qs["fl"][0] == "pdbid,entity_id"

    assert list(df.columns) == ["pdbid", "entity_id"]


def test_get_fields(requests_mock):
    url = "https://nakb.org/node/solr/nakb/select"

    docs = [{"pdbid": "1A1H", "entity_id": 1, "chain_id": "A"}]
    requests_mock.get(url, json=fake_solr_response(docs))

    client = NAKBClient()
    fields = client.get_fields()

    assert set(fields) == {"pdbid", "entity_id", "chain_id"}
