from pathlib import Path

import pytest

from neva.arch import Application
from neva.config import ConfigRepository


class TestBasicUsage:
    @pytest.mark.asyncio
    async def test_simple_config_access(
        self,
        application: Application,
    ) -> None:
        config = application.make(ConfigRepository).unwrap()

        app_name = config.get("app.name").unwrap()
        assert app_name == "TestApp"

        is_debug = config.get("app.debug").unwrap()
        assert is_debug is True

    @pytest.mark.asyncio
    async def test_config_with_default(
        self,
        application: Application,
    ) -> None:
        config = application.make(ConfigRepository).unwrap()

        missing = config.get("app.missing_key", "default_value").unwrap()
        assert missing == "default_value"

    @pytest.mark.asyncio
    async def test_multiple_config_keys(
        self,
        application: Application,
    ) -> None:
        config = application.make(ConfigRepository).unwrap()

        assert config.get("app.name").unwrap() == "TestApp"
        assert config.get("app.debug").unwrap() is True
        assert config.get("app.environment").unwrap() == "testing"


class TestCustomConfig:
    @pytest.fixture
    def test_config(
        self,
        tmp_path: Path,
    ) -> Path:
        config_dir = tmp_path / "config"
        config_dir.mkdir()

        _ = (config_dir / "app.py").write_text(
            """
config = {
    "name": "CustomApp",
    "debug": False,
    "environment": "custom",
    "custom_feature": True,
}
"""
        )

        _ = (config_dir / "providers.py").write_text("""config = {"providers": []}""")

        return config_dir

    @pytest.mark.asyncio
    async def test_uses_custom_config(
        self,
        application: Application,
    ) -> None:
        config = application.make(ConfigRepository).unwrap()

        assert config.get("app.name").unwrap() == "CustomApp"
        assert config.get("app.debug").unwrap() is False
        assert config.get("app.environment").unwrap() == "custom"
        assert config.get("app.custom_feature").unwrap() is True


class TestConfigManipulation:
    @pytest.mark.asyncio
    async def test_add_database_config(
        self,
        test_config: Path,
    ) -> None:
        _ = (test_config / "database.py").write_text(
            """
config = {
    "default": "sqlite",
    "connections": {
        "sqlite": {
            "driver": "sqlite",
            "database": ":memory:",
        }
    }
}
"""
        )

        app = Application(config_path=test_config)

        async with app.lifespan():
            config = app.make(ConfigRepository).unwrap()

            # Verify database config is loaded
            assert config.get("database.default").unwrap() == "sqlite"
            assert config.get("database.connections.sqlite.driver").unwrap() == "sqlite"
            assert (
                config.get("database.connections.sqlite.database").unwrap()
                == ":memory:"
            )

    @pytest.mark.asyncio
    async def test_add_multiple_config_files(
        self,
        test_config: Path,
    ) -> None:
        _ = (test_config / "cache.py").write_text(
            """config = {"driver": "memory", "ttl": 3600}"""
        )

        _ = (test_config / "logging.py").write_text(
            """config = {"level": "DEBUG", "format": "json"}"""
        )

        app = Application(config_path=test_config)

        async with app.lifespan():
            config = app.make(ConfigRepository).unwrap()

            # Verify both configs are loaded
            assert config.get("cache.driver").unwrap() == "memory"
            assert config.get("cache.ttl").unwrap() == 3600
            assert config.get("logging.level").unwrap() == "DEBUG"
            assert config.get("logging.format").unwrap() == "json"


class TestAppLifecycle:
    @pytest.mark.asyncio
    async def test_manual_app_creation(
        self,
        test_config: Path,
    ) -> None:
        app = Application(config_path=test_config)

        async with app.lifespan():
            config = app.make(ConfigRepository).unwrap()
            assert config.get("app.name").unwrap() == "TestApp"

    @pytest.mark.asyncio
    async def test_multiple_apps_in_one_test(
        self,
        test_config: Path,
    ) -> None:
        app1 = Application(config_path=test_config)
        async with app1.lifespan():
            config1 = app1.make(ConfigRepository).unwrap()
            assert config1.get("app.name").unwrap() == "TestApp"

        app2 = Application(config_path=test_config)
        async with app2.lifespan():
            config2 = app2.make(ConfigRepository).unwrap()
            assert config2.get("app.name").unwrap() == "TestApp"

        assert app1 is not app2


class TestIsolation:
    @pytest.mark.asyncio
    async def test_isolation_test_one(
        self,
        application: Application,
    ) -> None:
        config = application.make(ConfigRepository).unwrap()

        original_name = config.get("app.name").unwrap()
        assert original_name == "TestApp"

    @pytest.mark.asyncio
    async def test_isolation_test_two(
        self,
        application: Application,
    ) -> None:
        config = application.make(ConfigRepository).unwrap()

        assert config.get("app.name").unwrap() == "TestApp"
