"""SHA256 hasher."""

import hashlib
from typing import override
from neva.security.hashing.hashers.protocol import Hasher


def ensure_bytes(value: str | bytes) -> bytes:
    """Ensure the value is in bytes.

    Returns:
        the value in bytes
    """
    return value.encode() if isinstance(value, str) else value


class Sha256Hasher(Hasher):
    """SHA256 hasher."""

    @override
    def make(
        self,
        plaintext: str | bytes,
        *,
        salt: bytes | None = None,
    ) -> str:
        return hashlib.sha256(ensure_bytes(plaintext)).hexdigest()

    @override
    def check(
        self,
        plaintext: str | bytes,
        hashed: str | bytes,
    ) -> bool:
        return hashlib.sha256(ensure_bytes(plaintext)).digest() == ensure_bytes(hashed)

    @override
    def needs_rehash(
        self,
        hashed: str | bytes,
    ) -> bool:
        return False
