"""Splitter registry for NLQL.

This module manages registration and lookup of text splitters.
"""

from collections.abc import Callable

from nlql.errors import NLQLRegistryError
from nlql.text.splitting import SplitterFunc, default_sentence_splitter


class SplitterRegistry:
    """Registry for text splitters."""

    def __init__(self) -> None:
        self._splitters: dict[str, SplitterFunc] = {}
        self._register_builtin_splitters()

    def _register_builtin_splitters(self) -> None:
        """Register built-in splitters."""
        self._splitters["SENTENCE"] = default_sentence_splitter

    def register(self, name: str, func: SplitterFunc) -> None:
        """Register a splitter.

        Args:
            name: Splitter name (e.g., "SENTENCE")
            func: Splitter function that takes text and returns list of strings

        Raises:
            NLQLRegistryError: If splitter name is invalid
        """
        if not name or not name.isupper():
            raise NLQLRegistryError(f"Splitter name must be uppercase, got: {name}")

        self._splitters[name] = func

    def get(self, name: str) -> SplitterFunc | None:
        """Get a splitter by name.

        Args:
            name: Splitter name

        Returns:
            Splitter function if registered, None otherwise
        """
        return self._splitters.get(name)

    def has(self, name: str) -> bool:
        """Check if a splitter is registered."""
        return name in self._splitters


# Global registry instance
_global_splitter_registry = SplitterRegistry()


def register_splitter(name: str) -> Callable[[SplitterFunc], SplitterFunc]:
    """Decorator to register a custom text splitter.

    Args:
        name: Splitter name (e.g., "SENTENCE")

    Returns:
        Decorator function

    Example:
        >>> @register_splitter("SENTENCE")
        ... def german_sentence_splitter(text: str) -> list[str]:
        ...     import nltk
        ...     return nltk.sent_tokenize(text, language='german')
    """

    def decorator(func: SplitterFunc) -> SplitterFunc:
        _global_splitter_registry.register(name, func)
        return func

    return decorator


def get_splitter(name: str) -> SplitterFunc | None:
    """Get a registered splitter by name."""
    return _global_splitter_registry.get(name)

