# NLQL Release Checklist

This checklist ensures all steps are completed before releasing a new version of NLQL.

## Pre-Release Checklist

### 1. Code Quality

- [ ] All tests pass: `pytest`
- [ ] Code linting passes: `ruff check src tests`
- [ ] Type checking passes: `mypy src`
- [ ] No critical security issues
- [ ] Code coverage is acceptable (>80%)

### 2. Documentation

- [ ] README.md is up to date
- [ ] CHANGELOG.md is updated with new version
- [ ] All new features are documented
- [ ] API documentation is complete
- [ ] Examples are working and up to date
- [ ] Documentation builds successfully: `mkdocs build --strict`

### 3. Version Management

- [ ] Version number updated in `src/nlql/version.py`
- [ ] Version follows semantic versioning (MAJOR.MINOR.PATCH)
- [ ] Breaking changes are clearly documented

### 4. Git

- [ ] All changes are committed
- [ ] Branch is up to date with main
- [ ] No uncommitted changes: `git status`

## Release Process

### 1. Create Release Branch (Optional)

```bash
git checkout -b release/v0.2.0
```

### 2. Update Version

Edit `src/nlql/version.py`:
```python
__version__ = "0.2.0"
```

### 3. Update CHANGELOG

Add release notes to `CHANGELOG.md` (if exists).

### 4. Commit and Tag

```bash
git add .
git commit -m "Release v0.2.0"
git tag v0.2.0
git push origin main --tags
```

### 5. Build Package

```bash
# Clean previous builds
rm -rf dist/ build/ *.egg-info

# Build
python -m build

# Verify
twine check dist/*
```

### 6. Upload to PyPI

```bash
# Upload to production PyPI
twine upload dist/*
```

### 7. Verify Installation

```bash
# Wait 1-2 minutes, then test
pip install --upgrade python-nlql
python -c "from nlql import __version__; print(__version__)"
```

### 8. Create GitHub Release

1. Go to https://github.com/natural-language-query-language/python-nlql/releases
2. Click "Draft a new release"
3. Tag: `v0.2.0`
4. Title: `Release v0.2.0`
5. Description: Copy from CHANGELOG.md
6. Publish release

### 9. Deploy Documentation

Documentation is automatically deployed via GitHub Actions when pushing to `main`.

Verify at: https://natural-language-query-language.github.io/python-nlql

## Post-Release

### 1. Verify Release

- [ ] Package is available on PyPI: https://pypi.org/project/python-nlql/
- [ ] Documentation is updated
- [ ] GitHub release is created
- [ ] Installation works: `pip install python-nlql`

### 2. Announce Release

- [ ] Update project homepage (if applicable)
- [ ] Post on social media / forums (if applicable)
- [ ] Notify users of breaking changes

### 3. Prepare Next Development Cycle

```bash
# Bump version to next development version
# Edit src/nlql/version.py: __version__ = "0.3.0.dev0"
git add src/nlql/version.py
git commit -m "Bump version to 0.3.0.dev0"
git push origin main
```

## Rollback Procedure

If a critical issue is discovered after release:

### Option 1: Yank Release (PyPI)

```bash
# Mark version as yanked (users can still install explicitly)
twine upload --skip-existing --repository pypi dist/*
# Then manually yank on PyPI website
```

### Option 2: Release Hotfix

1. Create hotfix branch from tag
2. Fix the issue
3. Release new patch version (e.g., 0.2.1)
4. Follow normal release process

## Emergency Contacts

- PyPI Support: https://pypi.org/help/
- GitHub Support: https://support.github.com/

## Notes

- **Never delete a release from PyPI** - use yank instead
- **Always test in a clean environment** before releasing
- **Keep CHANGELOG.md updated** for user transparency
- **Follow semantic versioning** to avoid breaking user code

