# Publishing NLQL to PyPI

This guide explains how to build and publish the NLQL package to PyPI using **hatchling** as the build backend.

## Prerequisites

### 1. PyPI Account Setup

1. **Register accounts** (if not already done):
   - Production PyPI: https://pypi.org/account/register/
   - Test PyPI: https://test.pypi.org/account/register/

2. **Generate API Tokens**:
   
   **For Production PyPI**:
   - Go to https://pypi.org/manage/account/token/
   - Click "Add API token"
   - Token name: `python-nlql-upload`
   - Scope: "Entire account" (or specific to `python-nlql` project after first upload)
   - Copy the token (starts with `pypi-`)

   **For Test PyPI**:
   - Go to https://test.pypi.org/manage/account/token/
   - Follow the same steps
   - Token name: `python-nlql-test-upload`

### 2. Configure Credentials

**Option A: Using `.pypirc` file** (Recommended)

Create or edit `~/.pypirc` (Linux/Mac: `~/.pypirc`, Windows: `%USERPROFILE%\.pypirc`):

```ini
[distutils]
index-servers =
    pypi
    testpypi

[pypi]
username = __token__
password = pypi-YOUR_PRODUCTION_TOKEN_HERE

[testpypi]
repository = https://test.pypi.org/legacy/
username = __token__
password = pypi-YOUR_TEST_TOKEN_HERE
```

**Option B: Using Environment Variables**

```bash
# For production PyPI
export TWINE_USERNAME=__token__
export TWINE_PASSWORD=pypi-YOUR_PRODUCTION_TOKEN_HERE

# For test PyPI
export TWINE_REPOSITORY=testpypi
export TWINE_USERNAME=__token__
export TWINE_PASSWORD=pypi-YOUR_TEST_TOKEN_HERE
```

### 3. Install Build Tools

```bash
pip install --upgrade build twine
```

## Version Management

### Update Version Number

Edit `src/nlql/version.py`:

```python
"""Version information for NLQL."""

__version__ = "0.2.0"  # Update this
```

**Version Numbering Convention** (Semantic Versioning):
- `MAJOR.MINOR.PATCH` (e.g., `1.2.3`)
- **MAJOR**: Breaking changes
- **MINOR**: New features (backward compatible)
- **PATCH**: Bug fixes

### Update Changelog

Update `CHANGELOG.md` (if exists) or create release notes documenting:
- New features
- Bug fixes
- Breaking changes
- Deprecations

## Pre-Release Checklist

Before publishing, ensure:

- [ ] All tests pass: `pytest`
- [ ] Code quality checks pass: `ruff check src tests`
- [ ] Type checking passes: `mypy src`
- [ ] Documentation builds: `mkdocs build`
- [ ] Version number updated in `src/nlql/version.py`
- [ ] `CHANGELOG.md` updated (if applicable)
- [ ] Git committed and tagged:
  ```bash
  git add .
  git commit -m "Release v0.2.0"
  git tag v0.2.0
  git push origin main --tags
  ```

## Building the Package

### Clean Previous Builds

```bash
# Remove old build artifacts
rm -rf dist/ build/ *.egg-info
# Or on Windows PowerShell:
Remove-Item -Recurse -Force dist, build, *.egg-info -ErrorAction SilentlyContinue
```

### Build with Hatchling

```bash
python -m build
```

This creates two files in `dist/`:
- `python_nlql-0.2.0-py3-none-any.whl` (wheel distribution)
- `python-nlql-0.2.0.tar.gz` (source distribution)

### Verify Build Artifacts

```bash
# List contents
tar -tzf dist/python-nlql-0.2.0.tar.gz | head -20
unzip -l dist/python_nlql-0.2.0-py3-none-any.whl | head -20

# Check package metadata
twine check dist/*
```

Expected output:
```
Checking dist/python_nlql-0.2.0-py3-none-any.whl: PASSED
Checking dist/python-nlql-0.2.0.tar.gz: PASSED
```

## Publishing to PyPI

### Step 1: Upload to Production PyPI

**⚠️ Warning**: Once uploaded to PyPI, you **cannot** delete or re-upload the same version. Make sure everything is correct!

```bash
twine upload dist/*
```

Or specify repository explicitly:

```bash
twine upload --repository pypi dist/*
```

**Expected Output**:
```
Uploading distributions to https://upload.pypi.org/legacy/
Uploading python_nlql-0.2.0-py3-none-any.whl
100% ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━ 50.0/50.0 kB • 00:01 • ?
Uploading python-nlql-0.2.0.tar.gz
100% ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━ 45.0/45.0 kB • 00:01 • ?

View at:
https://pypi.org/project/python-nlql/0.2.0/
```

### Step 2: Verify Installation

Wait 1-2 minutes for PyPI to process the upload, then test installation:

```bash
# Create a fresh virtual environment
python -m venv test_env
source test_env/bin/activate  # On Windows: test_env\Scripts\activate

# Install from PyPI
pip install python-nlql

# Test import
python -c "from nlql import NLQL; print(NLQL.__module__)"

# Check version
python -c "from nlql import __version__; print(__version__)"
```

### Step 3: Verify on PyPI Website

Visit https://pypi.org/project/python-nlql/ and check:
- [ ] Version number is correct
- [ ] README renders correctly
- [ ] Project links work (Homepage, Documentation, Repository, Issues)
- [ ] Dependencies are listed correctly
- [ ] Classifiers are appropriate

## Post-Release Tasks

1. **Create GitHub Release**:
   - Go to https://github.com/natural-language-query-language/python-nlql/releases
   - Click "Draft a new release"
   - Tag: `v0.2.0`
   - Title: `Release v0.2.0`
   - Description: Copy from CHANGELOG.md
   - Attach build artifacts (optional)

2. **Announce Release**:
   - Update documentation website
   - Post on social media / community forums (if applicable)
   - Notify users of breaking changes

3. **Prepare for Next Development Cycle**:
   ```bash
   # Bump version to next development version
   # Edit src/nlql/version.py: __version__ = "0.3.0.dev0"
   git add src/nlql/version.py
   git commit -m "Bump version to 0.3.0.dev0"
   git push origin main
   ```

## Troubleshooting

### Error: "File already exists"

**Problem**: Trying to upload a version that already exists on PyPI.

**Solution**:
- You cannot replace an existing version on PyPI
- Increment the version number in `src/nlql/version.py`
- Rebuild and upload again

### Error: "Invalid distribution filename"

**Problem**: Build artifacts have incorrect naming.

**Solution**:
```bash
# Clean and rebuild
rm -rf dist/ build/ *.egg-info
python -m build
```

### Error: "403 Forbidden" or "Invalid credentials"

**Problem**: Authentication failed.

**Solution**:
- Verify API token is correct in `.pypirc`
- Ensure token has correct permissions
- For project-scoped tokens, upload first version with account-scoped token

### Error: "Package name already taken"

**Problem**: Package name `python-nlql` is already registered by someone else.

**Solution**:
- Choose a different package name (e.g., `nlql-lang`, `nlql-query`)
- Update `name` in `pyproject.toml`
- Update all documentation references

### Error: "README rendering failed"

**Problem**: PyPI cannot render the README.md.

**Solution**:
- Ensure `readme = "README.md"` in `pyproject.toml`
- Check README.md for unsupported Markdown syntax
- Test locally: `twine check dist/*`

### Build includes unwanted files

**Problem**: `dist/` contains test files, cache, etc.

**Solution**:
- Check `.gitignore` is properly configured
- Hatchling respects `.gitignore` by default
- Explicitly exclude in `pyproject.toml`:
  ```toml
  [tool.hatch.build.targets.wheel]
  exclude = [
      "tests/",
      "docs/",
      ".github/",
  ]
  ```

## Quick Reference

### Complete Release Workflow

```bash
# 1. Update version
vim src/nlql/version.py

# 2. Run tests
pytest

# 3. Clean and build
rm -rf dist/ build/ *.egg-info
python -m build

# 4. Check build
twine check dist/*

# 5. Upload to PyPI
twine upload dist/*

# 6. Tag and push
git tag v0.2.0
git push origin main --tags

# 7. Test installation
pip install --upgrade python-nlql
```

## Additional Resources

- **PyPI Help**: https://pypi.org/help/
- **Packaging Guide**: https://packaging.python.org/
- **Hatchling Docs**: https://hatch.pypa.io/latest/
- **Twine Docs**: https://twine.readthedocs.io/
- **Semantic Versioning**: https://semver.org/


