"""Function registry for NLQL.

This module manages registration and lookup of functions like LENGTH, NOW, COUNT, etc.
"""

from collections.abc import Callable
from datetime import datetime
from typing import Any

from nlql.errors import NLQLRegistryError

# Type alias for function implementations
FunctionImpl = Callable[..., Any]


class FunctionRegistry:
    """Registry for NLQL functions."""

    def __init__(self) -> None:
        self._functions: dict[str, FunctionImpl] = {}
        self._register_builtin_functions()

    def _register_builtin_functions(self) -> None:
        """Register built-in functions."""
        self._functions["LENGTH"] = self._builtin_length
        self._functions["NOW"] = self._builtin_now
        self._functions["COUNT"] = self._builtin_count

    def _builtin_length(self, text: str) -> int:
        """Built-in LENGTH function."""
        return len(text)

    def _builtin_now(self) -> datetime:
        """Built-in NOW function."""
        return datetime.now()

    def _builtin_count(self, text: str, substring: str) -> int:
        """Built-in COUNT function."""
        return text.count(substring)

    def register(self, name: str, func: FunctionImpl) -> None:
        """Register a function.

        Args:
            name: Function name (e.g., "days_ago")
            func: Function implementation

        Raises:
            NLQLRegistryError: If function name is invalid
        """
        if not name:
            raise NLQLRegistryError("Function name cannot be empty")

        self._functions[name] = func

    def get(self, name: str) -> FunctionImpl | None:
        """Get a function by name.

        Args:
            name: Function name

        Returns:
            Function implementation if registered, None otherwise
        """
        return self._functions.get(name)

    def has(self, name: str) -> bool:
        """Check if a function is registered."""
        return name in self._functions


# Global registry instance
_global_function_registry = FunctionRegistry()


def register_function(name: str) -> Callable[[FunctionImpl], FunctionImpl]:
    """Decorator to register a custom function.

    Args:
        name: Function name (e.g., "days_ago")

    Returns:
        Decorator function

    Example:
        >>> from datetime import datetime, timedelta
        >>> @register_function("days_ago")
        ... def days_ago(days: int) -> datetime:
        ...     return datetime.now() - timedelta(days=days)
    """

    def decorator(func: FunctionImpl) -> FunctionImpl:
        _global_function_registry.register(name, func)
        return func

    return decorator


def get_function(name: str) -> FunctionImpl | None:
    """Get a registered function by name."""
    return _global_function_registry.get(name)

