"""Query result objects for NLQL."""

from dataclasses import dataclass, field
from typing import Any


@dataclass
class Result:
    """Represents a single query result.

    Attributes:
        content: The text content of this result
        score: Relevance score (e.g., similarity score), if applicable
        metadata: Additional metadata from the data source
        unit: The text unit type (DOCUMENT, CHUNK, SENTENCE, SPAN)
        source_id: Optional identifier for the source document/chunk
    """

    content: str
    score: float | None = None
    metadata: dict[str, Any] = field(default_factory=dict)
    unit: str = "CHUNK"
    source_id: str | None = None

    def __repr__(self) -> str:
        score_str = f"{self.score:.3f}" if self.score is not None else "N/A"
        return f"Result(unit={self.unit}, score={score_str}, content={self.content[:50]}...)"

