"""Default embedding provider with lazy loading.

This module provides a default embedding implementation using sentence-transformers.
The dependency is lazy-loaded to avoid making it a hard requirement.
"""

from typing import Any

from nlql.errors import NLQLConfigError


class DefaultEmbeddingProvider:
    """Default embedding provider using all-MiniLM-L6-v2.

    This provider lazy-loads sentence-transformers only when actually used.
    """

    def __init__(self, model_name: str = "all-MiniLM-L6-v2") -> None:
        self.model_name = model_name
        self._model: Any = None

    def _load_model(self) -> Any:
        """Lazy load the sentence-transformers model."""
        if self._model is None:
            try:
                from sentence_transformers import SentenceTransformer
            except ImportError as e:
                raise NLQLConfigError(
                    "sentence-transformers is required for default embedding provider. "
                    "Install it with: pip install sentence-transformers\n"
                    "Or register a custom embedding provider using "
                    "@nlql.register_embedding_provider"
                ) from e

            self._model = SentenceTransformer(self.model_name)

        return self._model

    def embed(self, texts: list[str]) -> list[list[float]]:
        """Generate embeddings for a list of texts.

        Args:
            texts: List of text strings to embed

        Returns:
            List of embedding vectors
        """
        model = self._load_model()
        embeddings = model.encode(texts, convert_to_numpy=True)
        return embeddings.tolist()

    def __call__(self, texts: list[str]) -> list[list[float]]:
        """Allow the provider to be called directly."""
        return self.embed(texts)

