"""Tests for ORDER BY functionality."""

import pytest

from nlql import NLQL
from nlql.adapters import MemoryAdapter


@pytest.fixture
def adapter_with_data() -> MemoryAdapter:
    """Create a MemoryAdapter with test data."""
    adapter = MemoryAdapter()
    
    # Add chunks with various metadata for sorting
    adapter.add_text("First chunk", {"score": 10, "author": "Alice", "date": "2024-01-01"})
    adapter.add_text("Second chunk", {"score": 5, "author": "Bob", "date": "2024-01-02"})
    adapter.add_text("Third chunk", {"score": 15, "author": "Charlie", "date": "2024-01-03"})
    adapter.add_text("Fourth chunk", {"score": 8, "author": "Alice", "date": "2024-01-04"})
    adapter.add_text("Fifth chunk", {"score": 12, "author": "Bob", "date": "2024-01-05"})
    
    return adapter


def test_order_by_numeric_asc(adapter_with_data: MemoryAdapter) -> None:
    """Test ORDER BY with numeric field in ascending order."""
    nlql = NLQL(adapter=adapter_with_data)
    results = nlql.execute("SELECT CHUNK ORDER BY score ASC")
    
    assert len(results) == 5
    scores = [r.metadata["score"] for r in results]
    assert scores == [5, 8, 10, 12, 15]


def test_order_by_numeric_desc(adapter_with_data: MemoryAdapter) -> None:
    """Test ORDER BY with numeric field in descending order."""
    nlql = NLQL(adapter=adapter_with_data)
    results = nlql.execute("SELECT CHUNK ORDER BY score DESC")
    
    assert len(results) == 5
    scores = [r.metadata["score"] for r in results]
    assert scores == [15, 12, 10, 8, 5]


def test_order_by_string_asc(adapter_with_data: MemoryAdapter) -> None:
    """Test ORDER BY with string field in ascending order."""
    nlql = NLQL(adapter=adapter_with_data)
    results = nlql.execute("SELECT CHUNK ORDER BY author ASC")
    
    assert len(results) == 5
    authors = [r.metadata["author"] for r in results]
    # Alice (2), Bob (2), Charlie (1)
    assert authors == ["Alice", "Alice", "Bob", "Bob", "Charlie"]


def test_order_by_string_desc(adapter_with_data: MemoryAdapter) -> None:
    """Test ORDER BY with string field in descending order."""
    nlql = NLQL(adapter=adapter_with_data)
    results = nlql.execute("SELECT CHUNK ORDER BY author DESC")
    
    assert len(results) == 5
    authors = [r.metadata["author"] for r in results]
    assert authors == ["Charlie", "Bob", "Bob", "Alice", "Alice"]


def test_order_by_with_limit(adapter_with_data: MemoryAdapter) -> None:
    """Test ORDER BY combined with LIMIT."""
    nlql = NLQL(adapter=adapter_with_data)
    results = nlql.execute("SELECT CHUNK ORDER BY score DESC LIMIT 3")
    
    assert len(results) == 3
    scores = [r.metadata["score"] for r in results]
    assert scores == [15, 12, 10]


def test_order_by_with_where(adapter_with_data: MemoryAdapter) -> None:
    """Test ORDER BY combined with WHERE clause."""
    nlql = NLQL(adapter=adapter_with_data)
    results = nlql.execute('SELECT CHUNK WHERE META("author") == "Alice" ORDER BY score DESC')
    
    assert len(results) == 2
    scores = [r.metadata["score"] for r in results]
    assert scores == [10, 8]
    
    # Verify all results are from Alice
    authors = [r.metadata["author"] for r in results]
    assert all(author == "Alice" for author in authors)


def test_order_by_with_where_and_limit(adapter_with_data: MemoryAdapter) -> None:
    """Test ORDER BY combined with WHERE and LIMIT."""
    nlql = NLQL(adapter=adapter_with_data)
    results = nlql.execute('SELECT CHUNK WHERE META("author") == "Bob" ORDER BY score ASC LIMIT 1')
    
    assert len(results) == 1
    assert results[0].metadata["score"] == 5
    assert results[0].metadata["author"] == "Bob"


def test_order_by_missing_field() -> None:
    """Test ORDER BY with a field that doesn't exist in some chunks."""
    adapter = MemoryAdapter()
    adapter.add_text("Has rating", {"rating": 5})
    adapter.add_text("No rating", {})
    adapter.add_text("Also has rating", {"rating": 3})
    
    nlql = NLQL(adapter=adapter)
    results = nlql.execute("SELECT CHUNK ORDER BY rating ASC")
    
    # Chunks with None values should be at the end
    assert len(results) == 3
    # First two should have ratings
    assert results[0].metadata.get("rating") == 3
    assert results[1].metadata.get("rating") == 5
    # Last one should have no rating
    assert results[2].metadata.get("rating") is None


def test_order_by_similarity() -> None:
    """Test ORDER BY SIMILARITY (using similarity scores in metadata)."""
    adapter = MemoryAdapter()
    adapter.add_text("First", {"similarity": 0.8})
    adapter.add_text("Second", {"similarity": 0.95})
    adapter.add_text("Third", {"similarity": 0.6})
    
    nlql = NLQL(adapter=adapter)
    
    # Test ascending order
    results = nlql.execute("SELECT CHUNK ORDER BY SIMILARITY ASC")
    similarities = [r.metadata["similarity"] for r in results]
    assert similarities == [0.6, 0.8, 0.95]
    
    # Test descending order
    results = nlql.execute("SELECT CHUNK ORDER BY SIMILARITY DESC")
    similarities = [r.metadata["similarity"] for r in results]
    assert similarities == [0.95, 0.8, 0.6]


def test_order_by_default_direction() -> None:
    """Test that ORDER BY defaults to ASC when direction is not specified."""
    adapter = MemoryAdapter()
    adapter.add_text("A", {"value": 3})
    adapter.add_text("B", {"value": 1})
    adapter.add_text("C", {"value": 2})
    
    nlql = NLQL(adapter=adapter)
    results = nlql.execute("SELECT CHUNK ORDER BY value")
    
    values = [r.metadata["value"] for r in results]
    assert values == [1, 2, 3]  # Should be ascending by default

