"""Operator registry for NLQL.

This module manages registration and lookup of operators like MATCH, SIMILAR_TO, etc.
"""

from collections.abc import Callable
from typing import Any

from nlql.errors import NLQLRegistryError

# Type alias for operator functions
OperatorFunc = Callable[..., Any]


class OperatorRegistry:
    """Registry for NLQL operators."""

    def __init__(self) -> None:
        self._operators: dict[str, OperatorFunc] = {}
        self._register_builtin_operators()

    def _register_builtin_operators(self) -> None:
        """Register built-in operators with placeholder implementations."""
        # These are placeholders - actual implementations will be in engine
        self._operators["MATCH"] = self._placeholder_match
        self._operators["SIMILAR_TO"] = self._placeholder_similar_to
        self._operators["CONTAINS"] = self._placeholder_contains
        self._operators["IS"] = self._placeholder_is
        self._operators["META"] = self._placeholder_meta

    def _placeholder_match(self, text: str, pattern: str) -> bool:
        """Placeholder for MATCH operator."""
        return pattern.lower() in text.lower()

    def _placeholder_similar_to(self, text: str, query: str) -> float:
        """Placeholder for SIMILAR_TO operator."""
        # Simple placeholder - actual implementation uses embeddings
        return 0.5

    def _placeholder_contains(self, text: str, substring: str) -> bool:
        """Placeholder for CONTAINS operator."""
        return substring in text

    def _placeholder_is(self, text: str, label: str) -> bool:
        """Placeholder for IS operator."""
        # Placeholder - actual implementation uses classification
        return False

    def _placeholder_meta(self, metadata: dict[str, Any], field: str) -> Any:
        """Placeholder for META operator."""
        return metadata.get(field)

    def register(self, name: str, func: OperatorFunc) -> None:
        """Register an operator.

        Args:
            name: Operator name (e.g., "SIMILAR_TO")
            func: Operator implementation function

        Raises:
            NLQLRegistryError: If operator name is invalid
        """
        if not name or not name.isupper():
            raise NLQLRegistryError(f"Operator name must be uppercase, got: {name}")

        self._operators[name] = func

    def get(self, name: str) -> OperatorFunc | None:
        """Get an operator by name.

        Args:
            name: Operator name

        Returns:
            Operator function if registered, None otherwise
        """
        return self._operators.get(name)

    def has(self, name: str) -> bool:
        """Check if an operator is registered."""
        return name in self._operators


# Global registry instance
_global_operator_registry = OperatorRegistry()


def register_operator(name: str) -> Callable[[OperatorFunc], OperatorFunc]:
    """Decorator to register a custom operator.

    Args:
        name: Operator name (e.g., "SIMILAR_TO")

    Returns:
        Decorator function

    Example:
        >>> @register_operator("CUSTOM_OP")
        ... def my_operator(text: str, param: str) -> bool:
        ...     return param in text
    """

    def decorator(func: OperatorFunc) -> OperatorFunc:
        _global_operator_registry.register(name, func)
        return func

    return decorator


def get_operator(name: str) -> OperatorFunc | None:
    """Get a registered operator by name."""
    return _global_operator_registry.get(name)

