"""Basic smoke tests for NLQL."""

import pytest

from nlql import NLQL, NLQLConfig, Result
from nlql.adapters import MemoryAdapter


def test_import() -> None:
    """Test that NLQL can be imported."""
    from nlql import NLQL

    assert NLQL is not None


def test_version() -> None:
    """Test that version is available."""
    from nlql import __version__

    assert __version__ is not None
    assert isinstance(__version__, str)


def test_nlql_initialization_with_adapter() -> None:
    """Test NLQL initialization with explicit adapter."""
    adapter = MemoryAdapter()
    nlql = NLQL(adapter=adapter)

    assert nlql is not None
    assert nlql.adapter is adapter


def test_nlql_initialization_requires_adapter() -> None:
    """Test that NLQL requires an adapter."""
    with pytest.raises(TypeError, match="adapter must be an instance of BaseAdapter"):
        NLQL(adapter=None)  # type: ignore

    with pytest.raises(TypeError, match="adapter must be an instance of BaseAdapter"):
        NLQL(adapter="not an adapter")  # type: ignore


def test_simple_query_execution() -> None:
    """Test basic query execution."""
    adapter = MemoryAdapter()
    adapter.add_text("AI agents are autonomous systems", {"topic": "AI"})
    adapter.add_text("Machine learning powers modern AI", {"topic": "ML"})
    adapter.add_text("Natural language processing", {"topic": "NLP"})

    nlql = NLQL(adapter=adapter)

    # Simple query without WHERE clause
    results = nlql.execute("SELECT CHUNK LIMIT 2")

    assert len(results) == 2
    assert all(isinstance(r, Result) for r in results)
    assert all(r.content for r in results)


def test_memory_adapter_basic() -> None:
    """Test MemoryAdapter basic functionality."""
    adapter = MemoryAdapter()

    # Test add_chunk
    chunk_id = adapter.add_chunk("First chunk", {"id": 1})
    assert chunk_id is not None
    assert len(adapter) == 1

    # Test add_text
    adapter.add_text("Second chunk", {"id": 2})
    assert len(adapter) == 2

    # Test query - MemoryAdapter returns ALL chunks (no filtering/limiting)
    from nlql.adapters.base import QueryPlan

    # MemoryAdapter ignores QueryPlan parameters and returns all data
    results = adapter.query(QueryPlan(limit=1))
    assert len(results) == 2  # Returns all chunks, ignores limit
    assert results[0].content == "First chunk"
    assert results[1].content == "Second chunk"

    # Empty QueryPlan should also return all chunks
    results = adapter.query(QueryPlan())
    assert len(results) == 2


def test_memory_adapter_add_texts() -> None:
    """Test MemoryAdapter batch add functionality."""
    adapter = MemoryAdapter()

    texts = ["First text", "Second text", "Third text"]
    metadatas = [{"id": 1}, {"id": 2}, {"id": 3}]

    chunk_ids = adapter.add_texts(texts, metadatas)

    assert len(chunk_ids) == 3
    assert len(adapter) == 3


def test_memory_adapter_add_document() -> None:
    """Test MemoryAdapter document chunking."""
    adapter = MemoryAdapter()

    long_text = """
    This is the first paragraph. It contains some information about AI.

    This is the second paragraph. It talks about machine learning.

    This is the third paragraph. It discusses natural language processing.
    """

    chunk_ids = adapter.add_document(
        long_text, metadata={"source": "test.txt"}, chunk_size=100, chunk_overlap=20
    )

    assert len(chunk_ids) > 0
    assert len(adapter) == len(chunk_ids)

    # Check that metadata is inherited
    from nlql.adapters.base import QueryPlan

    results = adapter.query(QueryPlan())
    for result in results:
        assert result.metadata.get("source") == "test.txt"
        assert "chunk_index" in result.metadata


def test_memory_adapter_clear() -> None:
    """Test MemoryAdapter clear functionality."""
    adapter = MemoryAdapter()
    adapter.add_text("Some text")
    assert len(adapter) == 1

    adapter.clear()
    assert len(adapter) == 0


def test_config() -> None:
    """Test configuration creation."""
    config = NLQLConfig(default_limit=10, debug_mode=True)

    assert config.default_limit == 10
    assert config.debug_mode is True


def test_default_limit_config() -> None:
    """Test that default_limit configuration is actually used."""
    adapter = MemoryAdapter()
    adapter.add_text("Text 1")
    adapter.add_text("Text 2")
    adapter.add_text("Text 3")
    adapter.add_text("Text 4")
    adapter.add_text("Text 5")

    # Test with default_limit=2
    config = NLQLConfig(default_limit=2)
    nlql = NLQL(adapter=adapter, config=config)

    # Query without explicit LIMIT should use default_limit
    results = nlql.execute("SELECT CHUNK")
    assert len(results) == 2

    # Query with explicit LIMIT should override default_limit
    results = nlql.execute("SELECT CHUNK LIMIT 4")
    assert len(results) == 4

    # Test without default_limit (should return all)
    config_no_limit = NLQLConfig(default_limit=None)
    nlql_no_limit = NLQL(adapter=adapter, config=config_no_limit)
    results = nlql_no_limit.execute("SELECT CHUNK")
    assert len(results) == 5


def test_debug_mode_config() -> None:
    """Test that debug_mode configuration enables logging."""
    import logging

    adapter = MemoryAdapter()
    adapter.add_text("Test text")

    # Create a logger handler to capture log messages
    log_capture = []

    class ListHandler(logging.Handler):
        def emit(self, record):
            log_capture.append(record.getMessage())

    handler = ListHandler()
    handler.setLevel(logging.DEBUG)

    # Get the executor logger and add our handler
    executor_logger = logging.getLogger("nlql.engine.executor")
    executor_logger.addHandler(handler)
    executor_logger.setLevel(logging.DEBUG)

    try:
        # Test with debug_mode=True
        config = NLQLConfig(debug_mode=True)
        nlql = NLQL(adapter=adapter, config=config)
        nlql.execute("SELECT CHUNK")

        # Should have debug log messages
        assert len(log_capture) > 0
        assert any("Executing query" in msg for msg in log_capture)

        # Test with debug_mode=False
        log_capture.clear()
        config_no_debug = NLQLConfig(debug_mode=False)
        nlql_no_debug = NLQL(adapter=adapter, config=config_no_debug)
        nlql_no_debug.execute("SELECT CHUNK")

        # Should have no debug log messages
        assert len(log_capture) == 0

    finally:
        # Clean up
        executor_logger.removeHandler(handler)


def test_type_system() -> None:
    """Test basic type system."""
    from nlql.types import DateType, NumberType, TextType

    # Number type
    num1 = NumberType(10)
    num2 = NumberType(20)
    assert num1 < num2
    assert num2 > num1
    assert num1 == NumberType(10)

    # Text type
    text1 = TextType("apple")
    text2 = TextType("banana")
    assert text1 < text2
    assert text2 > text1

    # Date type
    from datetime import datetime

    date1 = DateType(datetime(2024, 1, 1))
    date2 = DateType(datetime(2024, 12, 31))
    assert date1 < date2


def test_registry() -> None:
    """Test registry system."""
    from nlql.registry import register_function, register_operator

    # Register a custom function
    @register_function("test_func")
    def my_test_function(x: int) -> int:
        return x * 2

    # Register a custom operator
    @register_operator("TEST_OP")
    def my_test_operator(text: str) -> bool:
        return "test" in text.lower()

    # Verify registration
    from nlql.registry import get_function, get_operator

    func = get_function("test_func")
    assert func is not None
    assert func(5) == 10

    op = get_operator("TEST_OP")
    assert op is not None
    assert op("This is a test") is True

