"""Tests for nested logical expressions in WHERE clauses."""

import pytest

from nlql import NLQL
from nlql.adapters import MemoryAdapter


@pytest.fixture
def adapter():
    """Create a memory adapter with test data."""
    adapter = MemoryAdapter()
    
    # Add test documents with various metadata
    adapter.add_text(
        "Document about AI and machine learning",
        {"category": "AI", "author": "Alice", "year": 2024}
    )
    
    adapter.add_text(
        "Document about natural language processing",
        {"category": "NLP", "author": "Alice", "year": 2024}
    )
    
    adapter.add_text(
        "Document about computer vision",
        {"category": "CV", "author": "Bob", "year": 2024}
    )
    
    adapter.add_text(
        "Document about robotics",
        {"category": "Robotics", "author": "Charlie", "year": 2023}
    )
    
    adapter.add_text(
        "Document about databases",
        {"category": "Database", "author": "Alice", "year": 2023}
    )
    
    return adapter


def test_and_with_nested_or(adapter):
    """Test AND with nested OR: A AND (B OR C)"""
    nlql = NLQL(adapter=adapter)
    
    query = """
    SELECT CHUNK
    WHERE 
        META("author") == "Alice"
        AND (META("category") == "AI" OR META("category") == "NLP")
    """
    
    results = nlql.execute(query)
    
    # Should return 2 results: Alice's AI and NLP documents
    assert len(results) == 2
    categories = {r.metadata["category"] for r in results}
    assert categories == {"AI", "NLP"}
    assert all(r.metadata["author"] == "Alice" for r in results)


def test_or_with_nested_and(adapter):
    """Test OR with nested AND: A OR (B AND C)"""
    nlql = NLQL(adapter=adapter)
    
    query = """
    SELECT CHUNK
    WHERE 
        META("category") == "Robotics"
        OR (META("author") == "Alice" AND META("year") == 2024)
    """
    
    results = nlql.execute(query)
    
    # Should return 3 results: Robotics + Alice's 2024 documents (AI, NLP)
    assert len(results) == 3
    
    # Check that we have the robotics document
    robotics = [r for r in results if r.metadata["category"] == "Robotics"]
    assert len(robotics) == 1
    
    # Check that we have Alice's 2024 documents
    alice_2024 = [r for r in results if r.metadata["author"] == "Alice" and r.metadata["year"] == 2024]
    assert len(alice_2024) == 2


def test_complex_nested_logic(adapter):
    """Test complex nested logic: (A OR B) AND (C OR D)"""
    nlql = NLQL(adapter=adapter)
    
    query = """
    SELECT CHUNK
    WHERE 
        (META("category") == "AI" OR META("category") == "NLP")
        AND (META("author") == "Alice" OR META("author") == "Bob")
    """
    
    results = nlql.execute(query)
    
    # Should return 2 results: Alice's AI and NLP (Bob has CV, not AI/NLP)
    assert len(results) == 2
    assert all(r.metadata["author"] == "Alice" for r in results)
    categories = {r.metadata["category"] for r in results}
    assert categories == {"AI", "NLP"}


def test_triple_and_with_or(adapter):
    """Test multiple AND with nested OR: A AND B AND (C OR D)"""
    nlql = NLQL(adapter=adapter)
    
    query = """
    SELECT CHUNK
    WHERE 
        META("author") == "Alice"
        AND META("year") == 2024
        AND (META("category") == "AI" OR META("category") == "Database")
    """
    
    results = nlql.execute(query)
    
    # Should return 1 result: Alice's AI document from 2024
    # (Database is from 2023, so excluded by year filter)
    assert len(results) == 1
    assert results[0].metadata["category"] == "AI"
    assert results[0].metadata["author"] == "Alice"
    assert results[0].metadata["year"] == 2024


def test_not_with_nested_or(adapter):
    """Test NOT with nested OR: NOT (A OR B)"""
    nlql = NLQL(adapter=adapter)
    
    query = """
    SELECT CHUNK
    WHERE 
        NOT (META("category") == "AI" OR META("category") == "NLP")
    """
    
    results = nlql.execute(query)
    
    # Should return 3 results: CV, Robotics, Database
    assert len(results) == 3
    categories = {r.metadata["category"] for r in results}
    assert categories == {"CV", "Robotics", "Database"}


def test_deeply_nested_logic(adapter):
    """Test deeply nested logic: (A AND (B OR C)) OR (D AND E)"""
    nlql = NLQL(adapter=adapter)
    
    query = """
    SELECT CHUNK
    WHERE 
        (META("author") == "Alice" AND (META("category") == "AI" OR META("category") == "NLP"))
        OR (META("author") == "Bob" AND META("year") == 2024)
    """
    
    results = nlql.execute(query)
    
    # Should return 3 results: Alice's AI and NLP + Bob's CV
    assert len(results) == 3
    
    alice_docs = [r for r in results if r.metadata["author"] == "Alice"]
    assert len(alice_docs) == 2
    assert {r.metadata["category"] for r in alice_docs} == {"AI", "NLP"}
    
    bob_docs = [r for r in results if r.metadata["author"] == "Bob"]
    assert len(bob_docs) == 1
    assert bob_docs[0].metadata["category"] == "CV"

