"""Tests for NLQL parser."""

import pytest

from nlql.ast.nodes import Literal, SelectStatement, WhereClause
from nlql.errors import NLQLParseError
from nlql.parser import NLQLParser


def test_parser_initialization() -> None:
    """Test parser can be initialized."""
    parser = NLQLParser()
    assert parser is not None


def test_parse_simple_select() -> None:
    """Test parsing simple SELECT statement."""
    parser = NLQLParser()

    ast = parser.parse("SELECT CHUNK")

    assert isinstance(ast, SelectStatement)
    assert ast.select_unit == "CHUNK"
    assert ast.where is None
    assert ast.limit is None


def test_parse_select_with_limit() -> None:
    """Test parsing SELECT with LIMIT."""
    parser = NLQLParser()

    ast = parser.parse("SELECT CHUNK LIMIT 10")

    assert isinstance(ast, SelectStatement)
    assert ast.select_unit == "CHUNK"
    assert ast.limit == 10


def test_parse_select_document() -> None:
    """Test parsing SELECT DOCUMENT."""
    parser = NLQLParser()

    ast = parser.parse("SELECT DOCUMENT")

    assert ast.select_unit == "DOCUMENT"


def test_parse_select_sentence() -> None:
    """Test parsing SELECT SENTENCE."""
    parser = NLQLParser()

    ast = parser.parse("SELECT SENTENCE")

    assert ast.select_unit == "SENTENCE"


def test_parse_select_span() -> None:
    """Test parsing SELECT SPAN."""
    parser = NLQLParser()

    ast = parser.parse("SELECT SPAN(SENTENCE, window=3)")

    assert ast.select_unit == "SPAN"
    assert ast.span_config is not None
    assert ast.span_config["unit"] == "SENTENCE"
    assert ast.span_config["window"] == 3


def test_parse_with_where_operator() -> None:
    """Test parsing with WHERE clause using operator."""
    parser = NLQLParser()

    ast = parser.parse('SELECT CHUNK WHERE CONTAINS("AI")')

    assert isinstance(ast, SelectStatement)
    assert ast.where is not None
    assert isinstance(ast.where, WhereClause)


def test_parse_with_where_function() -> None:
    """Test parsing with WHERE clause using function."""
    parser = NLQLParser()

    ast = parser.parse("SELECT CHUNK WHERE LENGTH(content) > 100")

    assert ast.where is not None


def test_parse_invalid_syntax() -> None:
    """Test that invalid syntax raises NLQLParseError."""
    parser = NLQLParser()

    with pytest.raises(NLQLParseError):
        parser.parse("INVALID QUERY")


def test_parse_error_includes_context() -> None:
    """Test that parse errors include context information."""
    parser = NLQLParser()

    try:
        parser.parse("SELECT CHUNK WHERE")
    except NLQLParseError as e:
        # Error should have some context
        assert e.message is not None
        # Line/column may or may not be present depending on error type
    else:
        pytest.fail("Expected NLQLParseError")


def test_parse_order_by() -> None:
    """Test parsing ORDER BY clause."""
    parser = NLQLParser()

    ast = parser.parse("SELECT CHUNK ORDER BY SIMILARITY DESC")

    assert len(ast.order_by) > 0
    assert ast.order_by[0].direction == "DESC"


def test_parse_complex_query() -> None:
    """Test parsing a more complex query."""
    parser = NLQLParser()

    query = """
        SELECT SENTENCE
        WHERE SIMILAR_TO("AI agents") > 0.8
        ORDER BY SIMILARITY DESC
        LIMIT 5
    """

    ast = parser.parse(query)

    assert ast.select_unit == "SENTENCE"
    assert ast.where is not None
    assert len(ast.order_by) > 0
    assert ast.limit == 5

