# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['odisi']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.26.2,<2.0.0', 'polars>=0.19.17,<0.20.0']

setup_kwargs = {
    'name': 'python-odisi',
    'version': '0.3.0',
    'description': 'Import and post-process data generated by the Luna ODiSI System',
    'long_description': '# Python reader for exported ODiSI data\n\n## Description\n\nThis python package defines a reader and helper methods to handle data exported from the Luna ODiSI 6000 optical measuring system.\nIt allows for an easier retrieval of data corresponding to each segment, as well as the possibility to interpolate the results based on additional measurements, such as experimental load.\nDoing this manually requires some amount of python code, which can be avoided by using this package.\n\n## Installation\n\nInstall as usual:\n\n```bash\npip install python-odisi\n```\n\n## Usage\n\n### Retrieve data from a \\*.tsv file\n\nThe library can be used to read files in the following manner:\n\n```python\nfrom odisi import read_tsv\n\nd = read_tsv("data_gages.tsv")\n\n# List all gages\ngages = d.gages\n# List all segments\nsegments = d.segments\n# Get the data for a specific gage, e.g. with the label \'A\'\nd_gage = d.gage("A")\n# Get the data for a specific segment, e.g. with the label \'Seg-1\'\nd_seg, x_seg = d.segment("Seg-1")\n```\n\n### Interpolation of data\n\nThe package allows to easily interpolate an external signal (e.g. the load during the test).\nFor this, two strategies can be followed:\n\n#### 1. Interpolate the data from the sensors using the timestamps from the external signal\n\n```python\nimport polars as pl\n\nload = pl.read_csv("load_data.csv")\n# Assume that the timestamp is in the column \'time\'\nd.interpolate(load.select(pl.col("time")))\n```\n\nThen you should be able to plot your data against the measured load:\n\n```python\nimport matplotlib.pyplot as plt\n\nd_gage = d.gage("A")\n# Assume that the load data is in column \'load\'\na_load = load.select(pl.col("load")).to_series()\n\nplt.plot(d_gage, a_load)\n```\n\n#### 2. Interpolate the data from the external signal to match the timestamp from the sensor data\n\n```python\nimport polars as pl\n\nload = pl.read_csv("load_data.csv")\n# Assume that the timestamp is in the column \'time\'\nnew_load = d.interpolate_signal(data=load, time="time")\n```\n\nThen you should be able to plot your data against the measured load:\n\n```python\nimport matplotlib.pyplot as plt\n\nd_gage = d.gage("A")\n# Assume that the load data is in column \'load\'\na_load = new_load.select(pl.col("load")).to_series()\n\nplt.plot(d_gage, a_load)\n```\n\nIn both cases it is assumed that the timestamps from both files are synchronized, i.e. that both measuring computers have synchronized clocks.\n\n### Clip data during interpolation\n\nIt is probable that the measurements from both data sources (ODiSI and additional system) were started at different times.\nThis produces some annoyances during the processing of the data due to the mismatch in datapoints.\nTo remedy this, the option `clip=True` can be passed to both interpolation methods (`interpolate(...)` and `interpolate_signal(...)`), which will clip the data to the common time interval between both signals.\n\n```python\nimport polars as pl\n\nload = pl.read_csv("load_data.csv")\n# Assume that the timestamp is in the column \'time\'\nd.interpolate(load.select(pl.col("time")), clip=True)\n```\n\n## Tests\n\nThe package includes a test suite which should be run with pytest:\n\n```bash\npoetry run pytest\n```\n\n## Citation\n\n```bib\n@software{Tapia_2023,\n    author = {Tapia Camú, Cristóbal},\n    title = {{python-odisi: Import data generated by the Luna ODiSI System}},\n    url = {https://github.com/cristobaltapia/python-odisi},\n    version = {v0.3},\n    year = {2023},\n}\n```\n',
    'author': 'Cristóbal Tapia Camú',
    'author_email': 'crtapia@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/cristobaltapia/python-odisi',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
