#
# parakeet.dqe.py
#
# Copyright (C) 2019 Diamond Light Source and Rosalind Franklin Institute
#
# Author: James Parkhurst
#
# This code is distributed under the GPLv3 license, a copy of
# which is included in the root directory of this package.
#
import numpy as np
import scipy.interpolate


class DQETable(object):
    """
    A class to compute DQE function in Fourier space

    """

    def __init__(self):
        """
        Initialise the DQE table table

        """

        # The spatial frequency
        self.spatial_freq = np.arange(65) / (65.0 - 1.0)

        # The electrons per second
        self.eps = np.array((1.5, 5))

        # The DQE, shape = (spatial_freq, eps)
        self.dqe_300kev = np.array(
            [
                (0.95, 0.8),
                (0.947707152938843, 0.795177001953125),
                (0.944999899291992, 0.790087890625),
                (0.94188673286438, 0.784739990234375),
                (0.938376147460938, 0.779140625),
                (0.934476636886597, 0.773297119140625),
                (0.930196694946289, 0.767216796875),
                (0.925544815444946, 0.760906982421875),
                (0.9205294921875, 0.754375),
                (0.915159218978882, 0.747628173828125),
                (0.909442489624023, 0.740673828125),
                (0.903387797927856, 0.733519287109375),
                (0.897003637695313, 0.726171875),
                (0.890298502731323, 0.718638916015625),
                (0.88328088684082, 0.710927734375),
                (0.875959283828735, 0.703045654296875),
                (0.8683421875, 0.695),
                (0.860438091659546, 0.686798095703125),
                (0.852255490112305, 0.678447265625),
                (0.843802876663208, 0.669954833984375),
                (0.835088745117188, 0.661328125),
                (0.826121589279175, 0.652574462890625),
                (0.816909902954102, 0.643701171875),
                (0.807462179946899, 0.634715576171875),
                (0.7977869140625, 0.625625),
                (0.787892599105835, 0.616436767578125),
                (0.777787728881836, 0.607158203125),
                (0.767480797195435, 0.597796630859375),
                (0.756980297851563, 0.588359375),
                (0.746294724655151, 0.578853759765625),
                (0.735432571411133, 0.569287109375),
                (0.724402331924439, 0.559666748046875),
                (0.7132125, 0.55),
                (0.701871569442749, 0.540294189453125),
                (0.690388034057617, 0.530556640625),
                (0.678770387649536, 0.520794677734375),
                (0.667027124023438, 0.511015625),
                (0.655166736984253, 0.501226806640625),
                (0.643197720336914, 0.491435546875),
                (0.631128567886353, 0.481649169921875),
                (0.6189677734375, 0.471875),
                (0.606723830795288, 0.462120361328125),
                (0.594405233764648, 0.452392578125),
                (0.582020476150513, 0.442698974609375),
                (0.569578051757812, 0.433046875),
                (0.55708645439148, 0.423443603515625),
                (0.544554177856445, 0.413896484375),
                (0.531989715957642, 0.404412841796875),
                (0.5194015625, 0.395),
                (0.506798211288452, 0.385665283203125),
                (0.49418815612793, 0.376416015625),
                (0.481579890823364, 0.367259521484375),
                (0.468981909179688, 0.358203125),
                (0.456402705001831, 0.349254150390625),
                (0.443850772094726, 0.340419921875),
                (0.431334604263306, 0.331707763671875),
                (0.4188626953125, 0.323125),
                (0.406443539047241, 0.314678955078125),
                (0.394085629272461, 0.306376953125),
                (0.381797459793091, 0.298226318359375),
                (0.369587524414062, 0.290234375),
                (0.357464316940308, 0.282408447265625),
                (0.345436331176758, 0.274755859375),
                (0.333512060928345, 0.267283935546875),
                (0.3217, 0.26),
            ]
        )

    def dqe_table(self, energy, eps):
        """
        Return a DQE table for the specified electrons per second

        """
        table = []
        for i in range(self.dqe_300kev.shape[0]):
            table.append(np.interp(eps, self.eps, self.dqe_300kev[i, :]))
        return np.array(table)

    def dqe_fs(self, energy, eps, shape):
        """
        Return a DQE image in fourier space

        """

        # Compute the DQE table for a given eps
        dqe_table = self.dqe_table(energy, eps)

        # Compute the spatial frequencies (in terms of nyquist)
        Y, X = np.mgrid[0 : shape[0], 0 : shape[1]]
        F = np.sqrt(
            ((Y - shape[0] // 2) / shape[0]) ** 2
            + ((X - shape[1] // 2) / shape[1]) ** 2
        )

        # Compute the DQE array
        return scipy.interp(F, self.spatial_freq, dqe_table).reshape(shape)
