import math
import time
import threading
from typing import Tuple
from dataclasses import dataclass

import mmh3
import pyarrow as pa
import pyarrow.fs as fs
import pyarrow.ipc as ipc


@dataclass
class Snapshot:
    """
    Snapshot of the filter state for serialization/deserialization.
    """
    k: int
    l: int
    g: int
    r: int        # refresh interval in nanoseconds
    base: int
    count: int
    buffer: bytearray


def snapshot_to_arrow(snapshot: Snapshot) -> pa.Table:
    """
    Convert a Snapshot dataclass to an Apache Arrow table, with a single row
    containing all fields. The 'buffer' is stored as a binary column.
    """
    return pa.table({
        "k": [snapshot.k],
        "l": [snapshot.l],
        "g": [snapshot.g],
        "r": [snapshot.r],
        "base": [snapshot.base],
        "count": [snapshot.count],
        "buffer": [bytes(snapshot.buffer)],  # store as bytes
    })


def arrow_to_snapshot(table: pa.Table) -> Snapshot:
    """
    Convert an Apache Arrow table (with the columns generated by `snapshot_to_arrow`)
    back into a Snapshot dataclass.
    """
    # We expect exactly 1 row in the table.
    row = table.slice(0, 1)  # ensure we only grab the first row
    k = row["k"][0].as_py()
    l_ = row["l"][0].as_py()
    g_ = row["g"][0].as_py()
    r_ = row["r"][0].as_py()
    base = row["base"][0].as_py()
    count = row["count"][0].as_py()
    buf = row["buffer"][0].as_py()  # this is bytes
    return Snapshot(k, l_, g_, r_, base, count, bytearray(buf))


def write_snapshot_to_s3(snapshot: Snapshot,
                         bucket: str,
                         key: str,
                         access_key: str,
                         secret_key: str) -> None:
    """
    Serialize the given snapshot to an Arrow IPC stream and write it to S3 at s3://bucket/key.
    """
    # Create an Arrow table from the snapshot
    table = snapshot_to_arrow(snapshot)
    
    # Open an S3 filesystem using PyArrow
    s3 = fs.S3FileSystem(access_key=access_key, secret_key=secret_key)
    
    # Write the table as an IPC (Arrow Stream) to the S3 object
    full_path = f"{bucket}/{key}"
    with s3.open_output_stream(full_path) as sink:
        ipc.write_table(table, sink)


def read_snapshot_from_s3(bucket: str,
                          key: str,
                          access_key: str,
                          secret_key: str) -> Snapshot:
    """
    Read an Arrow IPC stream from s3://bucket/key and deserialize it into a Snapshot.
    """
    s3 = fs.S3FileSystem(access_key=access_key, secret_key=secret_key)
    full_path = f"{bucket}/{key}"
    with s3.open_input_file(full_path) as source:
        table = ipc.read_table(source)
    return arrow_to_snapshot(table)


class AgePartitionedBloomFilter:
    """
    A thread-safe Age-Partitioned Bloom Filter (APBF) in Python,
    with the ability to snapshot state to S3 (via Apache Arrow)
    and restore itself from S3.
    """

    def __init__(self, k: int, l: int, g: int, refresh_interval: float = 0.0):
        """
        Initialize an Age-Partitioned Bloom Filter.

        Args:
            k (int): Number of hash slices used per insertion.
            l (int): Number of 'extra' slices, so total slices = k + l.
            g (int): Maximum number of items per generation before forcing a shift.
            refresh_interval (float): Optional time-based refresh interval in seconds.
                                      If > 0, the filter shifts generations
                                      automatically based on elapsed time.
        Raises:
            ValueError: If parameters are invalid (non-positive or negative).
        """
        if k <= 0 or l <= 0 or g <= 0 or refresh_interval < 0:
            raise ValueError("Invalid parameter(s). Ensure k, l, g > 0 and refresh_interval >= 0.")

        self.k = k
        self.l = l
        self.h = k + l  # total slices in the ring
        self.g = g
        self.r = int(refresh_interval * 1e9)  # store interval in nanoseconds

        # Calculate the number of bits per slice (heuristic).
        self.m = math.ceil(1.442695 * (k * g))

        # Total bits = (k + l) * m
        total_bits = self.h * self.m
        self.buffer_len = (total_bits + 7) // 8

        # Thread lock for all operations (simple approach).
        self._lock = threading.Lock()

        # Start slice for the current generation
        self.base = 1

        # Bit array + how many items we inserted in the current generation
        self.buffer = bytearray(self.buffer_len)
        self.count = 0

        # For time-based refreshing
        self._last_refresh = time.time_ns()

        # For optional periodic snapshotting
        self._snapshot_thread = None
        self._snapshot_stop_event = None

    @classmethod
    def from_s3(cls, bucket: str, key: str, access_key: str, secret_key: str) -> "AgePartitionedBloomFilter":
        """
        Initialize a new APBF by loading its snapshot from S3 (Arrow IPC file).
        Then reconstruct the filter via from_snapshot().
        """
        snap = read_snapshot_from_s3(bucket, key, access_key, secret_key)
        return cls.from_snapshot(snap)

    @classmethod
    def from_snapshot(cls, snap: Snapshot) -> "AgePartitionedBloomFilter":
        """
        Reconstruct an APBF from a Snapshot.
        """
        if snap.k <= 0 or snap.l <= 0 or snap.g <= 0 or snap.r < 0:
            raise ValueError("Invalid snapshot parameters (k, l, g, r).")

        # Build a new instance
        filter_obj = cls(
            k=snap.k,
            l=snap.l,
            g=snap.g,
            refresh_interval=snap.r / 1e9
        )

        # Validate
        if len(snap.buffer) != filter_obj.buffer_len:
            raise ValueError("Snapshot buffer size mismatch.")
        if not (1 <= snap.base <= filter_obj.h):
            raise ValueError("Snapshot base slice out of range.")
        if snap.count > snap.g:
            raise ValueError("Snapshot count cannot exceed generation capacity.")

        # Restore internal fields
        filter_obj.base = snap.base
        filter_obj.count = snap.count
        filter_obj.buffer = bytearray(snap.buffer)
        filter_obj._last_refresh = time.time_ns()

        return filter_obj

    def snapshot_to_s3(self, bucket: str, key: str, access_key: str, secret_key: str) -> None:
        """
        Acquire a snapshot of the current APBF state and write it to S3 as Arrow IPC.
        """
        with self._lock:
            snap = Snapshot(
                k=self.k,
                l=self.l,
                g=self.g,
                r=self.r,
                base=self.base,
                count=self.count,
                buffer=bytearray(self.buffer)  # copy
            )
        write_snapshot_to_s3(snap, bucket, key, access_key, secret_key)

    def start_periodic_snapshot_thread(self,
                                       interval_sec: float,
                                       bucket: str,
                                       key: str,
                                       access_key: str,
                                       secret_key: str) -> None:
        """
        Start a background thread that periodically snapshots this filter to S3
        every `interval_sec` seconds.
        """
        if self._snapshot_thread is not None:
            raise RuntimeError("Periodic snapshot thread already running.")

        self._snapshot_stop_event = threading.Event()

        def _snapshot_loop():
            while not self._snapshot_stop_event.is_set():
                time.sleep(interval_sec)
                if self._snapshot_stop_event.is_set():
                    break
                # snapshot to S3
                self.snapshot_to_s3(bucket, key, access_key, secret_key)

        self._snapshot_thread = threading.Thread(target=_snapshot_loop, daemon=True)
        self._snapshot_thread.start()

    def stop_periodic_snapshot_thread(self) -> None:
        """
        Stop the background snapshot thread if running.
        """
        if self._snapshot_thread is not None:
            self._snapshot_stop_event.set()
            self._snapshot_thread.join()
            self._snapshot_thread = None

    # ------------------------------------------------------------------------
    # Core Bloom Filter Logic
    # ------------------------------------------------------------------------

    def _get_hashes(self, item: bytes) -> Tuple[int, int]:
        """
        Generate two 64-bit hash values (h1, h2) via MurmurHash3.
        """
        return mmh3.hash64(item, seed=0)

    def _bit_index(self, slice_idx: int, h1_32: int, h2_32: int, i: int) -> int:
        """
        Double-hashing approach to find the bit offset for the i-th hash in the slice.
        """
        t = (i * i * i - i) // 6
        slice_offset = (h1_32 + i * h2_32 + t) % self.m
        return (slice_idx - 1) * self.m + slice_offset

    def _set_bit(self, bit_idx: int) -> None:
        self.buffer[bit_idx >> 3] |= (1 << (bit_idx & 7))

    def _has_bit(self, bit_idx: int) -> bool:
        return bool(self.buffer[bit_idx >> 3] & (1 << (bit_idx & 7)))

    def _clear_slice_bits(self, slice_idx: int) -> None:
        """
        Clear all bits belonging to slice_idx.
        """
        start_bit = (slice_idx - 1) * self.m
        end_bit = start_bit + self.m

        start_byte = start_bit >> 3
        end_byte = (end_bit + 7) >> 3

        mv = memoryview(self.buffer)
        mv[start_byte:end_byte] = b'\x00' * (end_byte - start_byte)

    def _next_slice(self, i: int) -> int:
        return 1 if i == self.h else i + 1

    def _prev_slice(self, i: int) -> int:
        return self.h if i == 1 else i - 1

    def _refresh_if_needed(self) -> None:
        """
        Perform a time-based refresh if refresh_interval > 0 and enough time has passed.
        """
        if self.r == 0:
            return
        now = time.time_ns()
        while self._last_refresh + self.r <= now:
            # shift once
            self._shift_generation()
            self._last_refresh += self.r

    def _shift_generation(self) -> None:
        """
        Shift to the next generation by moving 'base' backward in the ring and clearing
        that slice for reuse.
        """
        with self._lock:
            # Reset count in the new generation
            self.count = 0

            # Move base to the previous slice in the ring
            self.base = self._prev_slice(self.base)

            # Clear that slice
            self._clear_slice_bits(self.base)

    # ------------------------------------------------------------------------
    # Public Methods
    # ------------------------------------------------------------------------

    def add(self, item: bytes) -> None:
        """
        Insert 'item' into the current generation. 
        If the generation is at capacity (count == g), we shift first.
        """
        self._refresh_if_needed()
        with self._lock:
            if self.count == self.g:
                # Force shift
                self.count = 0
                self.base = self._prev_slice(self.base)
                self._clear_slice_bits(self.base)

            h1, h2 = self._get_hashes(item)
            h1_32 = h1 & 0xFFFFFFFF
            h2_32 = h2 & 0xFFFFFFFF

            slice_idx = self.base
            for i in range(1, self.k + 1):
                bit_idx = self._bit_index(slice_idx, h1_32, h2_32, i)
                self._set_bit(bit_idx)
                slice_idx = self._next_slice(slice_idx)

            self.count += 1

    def query(self, item: bytes) -> bool:
        """
        Check if 'item' might be in the filter. Returns True if possibly present
        (could be a false positive), or False if definitely not present.
        """
        self._refresh_if_needed()
        with self._lock:
            h1, h2 = self._get_hashes(item)
            h1_32 = h1 & 0xFFFFFFFF
            h2_32 = h2 & 0xFFFFFFFF

            matched = 0
            slice_idx = self.base

            # Up to h steps, looking for k consecutive slices with set bits
            for _ in range(self.h):
                i = matched + 1
                bit_idx = self._bit_index(slice_idx, h1_32, h2_32, i)
                if self._has_bit(bit_idx):
                    matched += 1
                    if matched == self.k:
                        return True
                else:
                    matched = 0

                slice_idx = self._next_slice(slice_idx)

            return False

    def next_generation(self) -> None:
        """
        Manually force a shift to the next generation (clearing one slice).
        """
        self._shift_generation()
        # Also update last_refresh so we don't immediately shift again
        with self._lock:
            self._last_refresh = time.time_ns()

    def max_capacity(self) -> int:
        """
        Return the maximum capacity across all slices/generations: (l+1)*g.
        """
        return self.g * (self.l + 1)

    @staticmethod
    def calculate_false_positive_rate(k: int, l: int) -> float:
        """
        A theoretical false positive rate calculation (heuristic) for the age-partitioned
        filter. This approach or formula might be approximate; use for reference only.
        """
        if k <= 0 or l <= 0:
            raise ValueError("k and l must be positive.")

        cache = {}

        def dfs(a: int, i: int) -> float:
            if a == k:
                return 1.0
            if i > l + a:
                return 0.0
            key = (a, i)
            if key in cache:
                return cache[key]

            # Probability that slice i is "hit." This is a simplistic assumption.
            ri = 0.5 if i >= k else float(i + 1) / (2.0 * k)

            val = ri * dfs(a + 1, i + 1) + (1 - ri) * dfs(0, i + 1)
            cache[key] = val
            return val

        return dfs(0, 0)