"""
Key handling utilities for EC keys (ANSI X.62/RFC3279), domain parameter and
signatures.

These utilities depend on :mod:`pyasn1` and :mod:`pyasn1_modules`.
"""

from ..constants import Attribute, ObjectClass
from ..mechanisms import KeyType

from pyasn1.type.univ import BitString, OctetString
from pyasn1.codec.der import encoder, decoder
from pyasn1_modules import rfc3280
from pyasn1_modules.rfc3279 import (
    ECDSA_Sig_Value,
    EcpkParameters,
    id_ecPublicKey,
)


def encode_named_curve_parameters(oid):
    """
    Return DER-encoded ANSI X.62 EC parameters for a named curve.

    Curve names are given by object identifier and can be found in
    :mod:`pyasn1_modules.rfc3279`.

    :param oid:
        Object identifier for a named curve
    :type oid: pyasn1.type.univ.ObjectIdentifier, str or tuple
    :rtype: bytes
    """
    ecParams = EcpkParameters()
    ecParams['namedCurve'] = oid
    return encoder.encode(ecParams)


def decode_ec_public_key(der, encode_ec_point=True):
    """
    Decode a DER-encoded EC public key as stored by OpenSSL into a dictionary
    of attributes able to be passed to :meth:`pkcs11.Session.create_object`.

    .. note:: **encode_ec_point**

        For use as an attribute `EC_POINT` should be DER-encoded (True).

        For key derivation implementations can vary.  Since v2.30 the
        specification says implementations MUST accept a raw `EC_POINT` for
        ECDH (False), however not all implementations follow this yet.

    :param bytes der: DER-encoded key
    :param encode_ec_point: See text.
    :rtype: dict(Attribute,*)
    """
    asn1, _ = decoder.decode(der, asn1Spec=rfc3280.SubjectPublicKeyInfo())

    assert asn1['algorithm']['algorithm'] == id_ecPublicKey, \
        "Wrong algorithm, not an EC key!"

    ecpoint = asn1['subjectPublicKey'].asOctets()

    if encode_ec_point:
        ecpoint = encoder.encode(OctetString(value=ecpoint))

    return {
        Attribute.KEY_TYPE: KeyType.EC,
        Attribute.CLASS: ObjectClass.PUBLIC_KEY,
        Attribute.EC_PARAMS: asn1['algorithm']['parameters'],
        Attribute.EC_POINT: ecpoint,
    }


def encode_ec_public_key(key):
    """
    Encode a DER-encoded EC public key as stored by OpenSSL.


    :param PublicKey key: RSA public key
    :rtype: bytes
    """

    asn1 = rfc3280.SubjectPublicKeyInfo()
    ecparams, _ = decoder.decode(key[Attribute.EC_PARAMS],
                                 asn1Spec=EcpkParameters())

    ecpoint, _ = decoder.decode(key[Attribute.EC_POINT],
                                asn1Spec=OctetString())

    asn1['algorithm'] = algo = rfc3280.AlgorithmIdentifier()
    algo['algorithm'] = id_ecPublicKey
    algo['parameters'] = ecparams

    asn1['subjectPublicKey'] = BitString.fromOctetString(ecpoint)

    return encoder.encode(asn1)


def encode_ecdsa_signature(signature):
    """
    Encode a signature (generated by :meth:`pkcs11.SignMixin.sign`) into
    DER-encoded ASN.1 (ECDSA_Sig_Value) format.

    :param bytes signature: signature as bytes
    :rtype: bytes
    """

    part = len(signature) // 2
    r, s = signature[:part], signature[part:]

    asn1 = ECDSA_Sig_Value()
    asn1['r'] = int.from_bytes(r, byteorder='big')
    asn1['s'] = int.from_bytes(s, byteorder='big')

    return encoder.encode(asn1)


def decode_ecdsa_signature(der):
    """
    Decode a DER-encoded ASN.1 (ECDSA_Sig_Value) signature (as generated by
    OpenSSL/X.509) into PKCS #11 format.

    :param bytes der: DER-encoded signature
    :rtype bytes:
    """

    asn1, _ = decoder.decode(der, asn1Spec=ECDSA_Sig_Value())

    r = int(asn1['r'])
    s = int(asn1['s'])

    # r and s must be the same length
    length = (max(r.bit_length(), s.bit_length()) + 7) // 8
    return b''.join((
        r.to_bytes(length, byteorder='big'),
        s.to_bytes(length, byteorder='big'),
    ))
