# -*- coding: utf-8 -*-
"""
# ---------------------------------------------------------------------------------------------------------
# ProjectName:  playwright-helper
# FileName:     base_po.py
# Description:  po对象基础类
# Author:       ASUS
# CreateDate:   2025/12/13
# Copyright ©2011-2025. Hunan xxxxxxx Company limited. All rights reserved.
# ---------------------------------------------------------------------------------------------------------
"""
import asyncio
from logging import Logger
from typing import List, Any, cast
from playwright.async_api import Page, Locator, TimeoutError as PlaywrightTimeoutError


class BasePo(object):
    __page: Page

    def __init__(self, page: Page, url: str):
        self.url = url
        self.__page = page

    def get_page(self) -> Page:
        return self.__page

    def is_current_page(self) -> bool:
        return self.iss_current_page(self.__page, self.url)

    def get_url_domain(self) -> str:
        if isinstance(self.__page, Page):
            page_slice: List[str] = self.__page.url.split("/")
            return f"{page_slice[0]}://{page_slice[2]}"
        else:
            raise AttributeError("PO对象中的page属性未被初始化")

    def get_url(self) -> str:
        if self.__page.url.find("://") != -1:
            return self.__page.url.split("?")[0]
        else:
            return self.__page.url

    @staticmethod
    def iss_current_page(page: Page, url: str) -> bool:
        if isinstance(page, Page):
            page_url_prefix = page.url.split("?")[0]
            url_prefix = url.split("?")[0]
            if page_url_prefix.endswith(url_prefix):
                return True
            else:
                return False
        else:
            return False

    @staticmethod
    async def exists(locator):
        return await locator.count() > 0

    @staticmethod
    async def exists_one(locator):
        return await locator.count() == 1

    async def get_locator(self, selector: str, timeout: float = 3.0) -> Locator:
        """
        获取页面元素locator
        :param selector: 选择器表达式
        :param timeout: 超时时间（秒）
        :return: 元素对象
        :return:
        """
        locator = self.__page.locator(selector)
        try:
            await locator.first.wait_for(state='visible', timeout=timeout * 1000)
            return locator
        except (PlaywrightTimeoutError,):
            raise PlaywrightTimeoutError(f"元素 '{selector}' 未在 {timeout} 秒内找到")
        except Exception as e:
            raise RuntimeError(f"检查元素时发生错误: {str(e)}")

    @staticmethod
    async def get_sub_locator(locator: Locator, selector: str, timeout: float = 3.0) -> Locator:
        """
        获取页面locator的子locator
        :param locator: 页面Locator对象
        :param selector: 选择器表达式
        :param timeout: 超时时间（秒）
        :return: 元素对象
        :return:
        """
        locator_inner = locator.locator(selector)
        try:
            await locator_inner.first.wait_for(state='visible', timeout=timeout * 1000)
            return locator_inner
        except (PlaywrightTimeoutError,):
            raise PlaywrightTimeoutError(f"元素 '{selector}' 未在 {timeout} 秒内找到")
        except Exception as e:
            raise RuntimeError(f"检查元素时发生错误: {str(e)}")

    @classmethod
    async def handle_po_cookie_tip(cls, page: Any, logger: Logger, timeout: float = 3.0,
                                   selectors: List[str] = None) -> None:
        selectors_inner: List[str] = [
            '//div[@id="isReadedCookie"]/button',
            '//button[@id="continue-btn"]/span[normalize-space(text())="同意"]'
        ]
        if selectors:
            selectors_inner.extend(selectors)
        for selector in selectors_inner:
            try:
                page_inner = cast(cls, page)
                cookie: Locator = await cls.get_locator(self=page_inner, selector=selector, timeout=timeout)
                logger.info(
                    f'找到页面中存在cookie提示：[本网站使用cookie，用于在您的电脑中储存信息。这些cookie可以使网站正常运行，以及帮助我们改进用户体验。使用本网站，即表示您接受放置这些cookie。]')
                await cookie.click(button="left")
                logger.info("【同意】按钮点击完成")
                await asyncio.sleep(1)
                return
            except (Exception,):
                pass
