# -*- coding: utf-8 -*-
"""
# ---------------------------------------------------------------------------------------------------------
# ProjectName:  playwright-helper
# FileName:     executor.py
# Description:  执行器模块
# Author:       ASUS
# CreateDate:   2025/12/13
# Copyright ©2011-2025. Hunan xxxxxxx Company limited. All rights reserved.
# ---------------------------------------------------------------------------------------------------------
"""
import os
import time
import uuid
import asyncio
from logging import Logger
from contextlib import asynccontextmanager
from playwright_helper.utils.type_utils import RunResult
from playwright._impl._api_structures import ViewportSize
from playwright_helper.utils.log_utils import logger as log
from playwright_helper.libs.browser_pool import BrowserPool
from playwright_helper.utils.file_handle import get_caller_dir
from typing import Any, List, Optional, cast, Callable, Literal, Dict
from playwright.async_api import Page, Browser, BrowserContext, TimeoutError as PlaywrightTimeoutError, \
    Error as PlaywrightError, async_playwright


class PlaywrightBrowserExecutor:
    def __init__(
            self,
            *,
            logger: Logger = log,
            browser_pool: Optional[BrowserPool] = None,
            mode: Literal["persistent", "storage"] = "storage",
            middlewares: Optional[List[Callable]] = None,
            retries: int = 1,
            record_video: bool = False,
            record_trace: bool = False,
            video_dir: str = None,
            trace_dir: str = None,
            screenshot_dir: str = None,
            viewport: Optional[ViewportSize] = None,
            user_agent: Optional[str] = None,
            proxy: Optional[Dict[str, Any]] = None,
            storage_state: Optional[Dict[str, Any]] = None,
            **browser_config: Any,
    ):
        self.mode = mode
        self.logger = logger
        self.proxy = proxy or None
        # 代理格式：proxy = {
        #     "server": "http://ip:port",       # HTTP 代理
        #     # 或
        #     "server": "https://ip:port",      # HTTPS 代理（较少见）
        #     # 或
        #     "server": "socks5://ip:port",     # SOCKS5 代理
        #     # 或
        #     "server": "socks4://ip:port",     # SOCKS4 代理
        #
        #     # 可选：如果代理需要认证
        #     "username": "your_user",
        #     "password": "your_pass"
        # }
        self.browser_pool = browser_pool
        self.middlewares = middlewares or []
        self.retries = retries
        self.user_agent = user_agent
        self.viewport = viewport

        self.record_video = record_video
        self.record_trace = record_trace
        self.video_dir = video_dir or get_caller_dir()
        self.trace_dir = trace_dir or get_caller_dir()
        self.screenshot_dir = screenshot_dir or get_caller_dir()
        self.storage_state = storage_state
        self.browser_config = browser_config

        self._playwright = None

        if self.mode == "storage" and not self.browser_pool:
            raise ValueError("storage 模式必须提供 browser_pool")

    async def _safe_screenshot(self, page: Page, name: str = None):
        try:
            os.makedirs(self.screenshot_dir, exist_ok=True)
            if name is None or "unknown" in name:
                name = f"error_{int(time.time())}"
            path = os.path.join(self.screenshot_dir, f"{name}.png")
            await page.screenshot(path=path)
            self.logger.debug(f"[Screenshot Saved] {path}")
        except Exception as e:
            self.logger.error(f"[Screenshot Failed] {e}")

    async def start(self):
        """Executor 生命周期开始（进程级调用一次）"""
        if not self._playwright:
            self._playwright = await async_playwright().start()

            # storage 模式：BrowserPool 需要 playwright
            if self.mode == "storage" and self.browser_pool:
                await self.browser_pool.start(self._playwright)

    async def stop(self):
        """Executor 生命周期结束"""
        if self.mode == "persistent" and self._playwright:
            await self._playwright.stop()
            self._playwright = None

    @asynccontextmanager
    async def session(self, storage_state: Optional[Dict[str, Any]] = None):
        context: BrowserContext = cast(BrowserContext, None)
        try:
            context = await self._create_context(storage_state=storage_state or self.storage_state)
            page: Page = await context.new_page()
            yield page
        finally:
            if context:
                await self._cleanup_context(context)

    async def _create_context(self, storage_state: Optional[Dict[str, Any]] = None) -> BrowserContext:
        task_id = str(uuid.uuid4())

        if self.mode == "persistent":
            self.logger.debug("[Executor] mode=persistent")
            # persistent 模式：不走 pool
            context = await self._playwright.chromium.launch_persistent_context(
                record_video_dir=self.video_dir if self.record_video else None,
                **self.browser_config
            )
        else:
            # 并发模式：BrowserPool
            self.logger.debug("[Executor] mode=storage")

            browser: Browser = await self.browser_pool.acquire()

            # 如果有 peoxy 代理配置，则在这里应用
            context = await browser.new_context(
                storage_state=storage_state or self.storage_state,
                viewport=self.viewport,
                user_agent=self.user_agent,
                proxy=self.proxy,
                record_video_dir=self.video_dir if self.record_video else None,
            )

            # 关键：挂载资源，供 cleanup 使用
            context._browser = browser  # type: ignore[attr-defined]

        # task_id 给 trace / video 用
        context._task_id = task_id  # type: ignore[attr-defined]

        if self.record_trace:
            await context.tracing.start(
                screenshots=True,
                snapshots=True,
                sources=True,
            )

        return context

    async def _cleanup_context(self, context: BrowserContext):
        if getattr(context, "_closed", False):
            return

        context._closed = True  # type: ignore

        try:
            await context.close()
        except Exception as e:
            self.logger.warning(f"[Cleanup] ignore close error: {e}")

        browser = getattr(context, "_browser", None)
        if browser:
            try:
                self.logger.debug("[Executor] Release browser to pool")
                await self.browser_pool.release(browser)
            except Exception as e:
                self.logger.error(f"[Cleanup] ignore release error: {e}")

    async def _run_callback_chain(self, *, callback: Callable, page: Page, context: BrowserContext, **kwargs) -> Any:
        # Run middlewares before callback
        for mw in self.middlewares:
            await mw(page=page, logger=self.logger, context=context, **kwargs)

        # Main callback
        return await callback(page=page, logger=self.logger, context=context, **kwargs)

    async def run(
            self, *, callback: Callable, storage_state: Optional[Dict[str, Any]] = None, **kwargs: Any
    ) -> RunResult:
        attempt = 0
        last_error: Optional[Exception] = None
        task_id: Optional[str] = "unknown"
        result: Any = None

        while attempt <= self.retries:
            page = None
            context: BrowserContext = cast(BrowserContext, None)
            try:
                context = await self._create_context(storage_state=storage_state)
                task_id = getattr(context, "_task_id", "unknown")
                page = await context.new_page()

                result = await self._run_callback_chain(
                    callback=callback, page=page, context=context, **kwargs
                )
                self.logger.info(f"[Task<{task_id}> Success]")
                return RunResult(
                    success=True,
                    attempts=attempt + 1,
                    task_id=task_id,
                    error=last_error,
                    result=result
                )
            except (asyncio.CancelledError, KeyboardInterrupt):
                # ⚠️ 不要当成错误
                self.logger.warning(f"[Task<{task_id}> Cancelled]")

                # 清理资源
                if context and self.record_trace:
                    os.makedirs(self.trace_dir, exist_ok=True)
                    trace_path = os.path.join(self.trace_dir, f"{task_id}.zip")
                    await context.tracing.stop(path=trace_path)
                    self.logger.debug(f"[Trace Saved] {trace_path}")

                # ❗关键：不要 raise
                return RunResult(
                    success=False,
                    attempts=attempt + 1,
                    error=asyncio.CancelledError(),
                    task_id=task_id,
                    result=result
                )
            except (PlaywrightTimeoutError, PlaywrightError, RuntimeError, EnvironmentError, Exception) as e:
                last_error = e
                self.logger.error(f"[Task<{task_id}> Attempt {attempt + 1} Failed] {e}")
                if page:
                    await self._safe_screenshot(page=page, name=task_id)

                if context and self.record_trace:
                    os.makedirs(self.trace_dir, exist_ok=True)
                    trace_path = os.path.join(self.trace_dir, f"{task_id}.zip")
                    await context.tracing.stop(path=trace_path)
                    self.logger.debug(f"[Trace Saved] {trace_path}")

                attempt += 1
                if attempt <= self.retries:
                    await asyncio.sleep(1)
            finally:
                if context:
                    await self._cleanup_context(context)
        # 所有重试结束，仍然失败
        self.logger.error(f"[Task<{task_id}> Final Failure]")

        return RunResult(
            success=False,
            attempts=attempt,
            error=last_error,
            task_id=task_id,
            result=result
        )
