##########################################################################
# XXX - Copyright (C) XXX, 2017
# Distributed under the terms of the CeCILL-B license, as published by
# the CEA-CNRS-INRIA. Refer to the LICENSE file or to
# http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
# for details.
##########################################################################

"""
Overload the proximity class from modopt.
"""

import numpy as np


class Threshold(object):
    """Threshold proximity operator

    This class defines the threshold proximity operator

    Parameters
    ----------
    weights : np.ndarray
        Input array of weights
    thresh_type : str {'hard', 'soft'}, optional
        Threshold type (default is 'soft')

    """
    def __init__(self, weights):
        self.weights = weights

    def op(self, data, extra_factor=1.0):
        """ Operator

        This method returns the input data thresholded by the weights

        Parameters
        ----------
        data : DictionaryBase
            Input data array
        extra_factor : float
            Additional multiplication factor

        Returns
        -------
        DictionaryBase thresholded data

        """
        threshold = self.weights * extra_factor
        return data * np.maximum(1 - threshold / np.maximum(
                                                np.finfo(np.float32).eps,
                                                np.abs(np.copy(data))), 0)

    def get_cost(self, x):
        """Cost function
        This method calculate the cost function of the proximable part.

        Parameters:
        ----------
        x: np.ndarray
            Input array of the sparse code.
        Return:
        ------
        The cost of this sparse code
        """
        return np.sum(np.abs(self.weights * x))
