#!/bin/bash

# This script generates text or html documentation from all scripts,
# from xml/help.xml and javadoc and places that in one directory.
#
# use -h for all options
#
# $Revision: 1.5 $
# FracPete

# prints the usage for this script
function usage()
{
   echo
   echo "usage: ${0##*/} -o <output-dir> -t <type> -d [-h]"
   echo
   echo "generates text or html documentation from all scripts and xml/help.xml"
   echo
   echo " -h   this help"
   echo " -o   <output-dir>"
   echo "      the directory where to store the documentation"
   echo "      (it is automatically created if non-existing)"
   echo "      default: $OUTPUT"
   echo " -t   <type>"
   echo "      the type of output, either 'html' or 'text'"
   echo "      default: $TYPE"
   echo " -d   deletes the output dir first"
   echo
}

# creates the dir TMP
function create_dir()
{
   if [ ! -d $TMP ]
   then
      mkdir -p $TMP
      echo "$TMP created."
   fi
}

# creates the output dirs
function create_dirs()
{
   TMP="$OUTPUT/gui";create_dir
   TMP="$OUTPUT/gui/images";create_dir
   TMP="$OUTPUT/classes";create_dir
   TMP="$OUTPUT/scripts";create_dir
   TMP="$OUTPUT/javadoc";create_dir
}

# builds the filename FILE from TMP and FILENAME and deletes it
function build_filename()
{
   FILE=$TMP/$FILENAME

   # remove file if existing
   if [ -f $FILE ]
   then
      rm $FILE
      echo "removed '$FILE'."
   fi
}

# creats the TOC HTML page for all the documentation
function create_toc()
{
   TMP="$OUTPUT/index.html"

   rm -f $TMP
   
   echo "<html>" >> $TMP
   echo "<header><title>WEKA Proper Doc</title></header>" >> $TMP
   echo "<body>" >> $TMP
   echo "<table width=\"100%\" height=\"100%\" cellpadding=\"10\">" >> $TMP
   echo "<tr><td align=\"center\" valign=\"bottom\"><h1>WEKA Proper Doc</h1></td></tr>" >> $TMP
   echo "<tr><td align=\"center\" valign=\"top\">" >> $TMP
   echo "<table border=\"0\" height=\"100\">" >> $TMP
   echo "<tr><td><a href=\"gui/\">GUI (Manual and First Steps)</a></td></tr>" >> $TMP
   echo "<tr><td><a href=\"classes/\">Classes</a></td></tr>" >> $TMP
   echo "<tr><td><a href=\"javadoc/\">Javadoc</a></td></tr>" >> $TMP
   echo "<tr><td><a href=\"scripts/\">Scripts</a></td></tr>" >> $TMP
   echo "<tr><td><a href=\"http://www.cs.waikato.ac.nz/ml/proper/\">WEKA Proper Homepage</a></td></tr>" >> $TMP
   echo "</table>" >> $TMP
   echo "<tr><td><p align=\"right\"><font size=\"-1\"><i>Last Updated: $DATE</i></font></p></td></tr>" >> $TMP
   echo "</td></tr></table>" >> $TMP
   echo "</body>" >> $TMP
   echo "</html>" >> $TMP
}

# creates the header of the output
function scripts_create_header()
{
   # header
   if [ "$TYPE" = "html" ]
   then
      echo "<html>" >> $FILE
      echo "<head><title>Scripts</title></head>" >> $FILE
      echo "<body>" >> $FILE
      echo "<center>" >> $FILE
      echo "<h1>Scripts</h1>" >> $FILE
      echo "<table cellpadding=\"5\">" >> $FILE
      echo "<tr><td valign=\"top\">Available Scripts</td><td><table cellpadding=\"0\">" >> $FILE
   elif [ "$TYPE" = "text" ]
   then
      echo "Scripts" >> $FILE
      echo "=======" >> $FILE
      echo >> $FILE
      echo "Available scripts:" >> $FILE
   fi

   # overview
   for i in $ROOT/*.sh
   do
      TMP=`echo ${i##*/}`

      # exclude classpath.sh
      COUNT=`echo $i | grep "classpath" | wc -l | sed s/" "*//g`
      if [ $COUNT -gt 0 ]
      then
         continue
      fi
      
      if [ "$TYPE" = "html" ]
      then
         echo "<tr><td></td><td><a href=\"#$TMP\">$TMP</a></td></tr>" >> $FILE
      elif [ "$TYPE" = "text" ]
      then
         echo " - $TMP" >> $FILE
      fi

   done

   # end overview
   if [ "$TYPE" = "html" ]
   then
      echo "</table></td></tr></table>" >> $FILE
      echo "<br>" >> $FILE
      echo "<table>" >> $FILE
   elif [ "$TYPE" = "text" ]
   then
      echo >> $FILE
   fi
}

# creates the body of the output
function scripts_create_body()
{
   for i in $ROOT/*.sh
   do
      TMP=`echo ${i##*/}`
      # make it HTML conformant: no <> and hyperlinks
      HELP=`$i -h | sed s/"<"/"\&lt;"/g | sed s/">"/"\&gt;"/g | sed s/"http:"[^\ ]*/"<a href=\""\&"\">"\&"<\/a>"/g`
      
      # exclude classpath.sh
      COUNT=`echo $i | grep "classpath" | wc -l | sed s/" "*//g`
      if [ $COUNT -gt 0 ]
      then
         continue
      fi
      
      if [ "$TYPE" = "html" ]
      then
         echo "<tr><td><b><a name=\"$TMP\">$TMP</a></b></td></tr>" >> $FILE
         echo "<tr><td><pre>$HELP</pre><br></td></tr>" >> $FILE
      elif [ "$TYPE" = "text" ]
      then
         echo "$TMP" >> $FILE
         echo `echo $TMP | sed s/./"-"/g` >> $FILE
         echo "$HELP" >> $FILE
         echo >> $FILE
      fi

      echo -n "."
   done

   echo
}

# creates the footer of the output
function scripts_create_footer()
{
   if [ "$TYPE" = "html" ]
   then
      echo "</table>" >> $FILE
      echo "</center>" >> $FILE
      echo "<p align=\"right\"><i><font size=\"-1\">Last updated: $DATE</font></i></p>" >> $FILE
      echo "</body>" >> $FILE
      echo "</html>" >> $FILE
   elif [ "$TYPE" = "text" ]
   then
      echo >> $FILE
      echo "Last updated: $DATE" >> $FILE
   fi
}

# creates the header of the output
function classes_create_header()
{
   if [ "$TYPE" = "html" ]
   then
      echo "<html>" >> $FILE
      echo "<head><title>Classes</title></head>" >> $FILE
      echo "<body>" >> $FILE
      echo "<center>" >> $FILE
      echo "<h1>Classes</h1>" >> $FILE
      echo "<table cellpadding=\"5\">" >> $FILE
      echo "<tr><td valign=\"top\">Available Classes</td><td><table cellpadding=\"0\">" >> $FILE
      echo "<tr><td>" >> $FILE
      echo `cat $HELP | grep -v "\[java\]\|Buildfile\|all\|gui:\|commandline\|Total time\|BUILD" | sed s/\{r\}//g | grep -v ^" "*$ | sed s/.*/"\<a href=\"#&\">&\<\/a>\<br>"/g | sed s/":\""/\"/g | sed s/":<"/"<"/g | sed s/.*"\[echo\] "/"\<i>"/g | sed s/")".*/"):\<\/i>\<br>"/g` >> $FILE
      echo "</td></tr>" >> $FILE
      echo "</table></td></tr></table>" >> $FILE
      echo "<br>" >> $FILE
      echo "<table>" >> $FILE
   elif [ "$TYPE" = "text" ]
   then
      echo "Classes" >> $FILE
      echo "=======" >> $FILE
      echo >> $FILE
      cat $HELP | grep -v "\[java\]\|Buildfile\|all\|gui:\|commandline\|Total time\|BUILD" | sed s/\{r\}//g | grep -v ^" "*$ | sed s/":"//g | sed s/^/" - "/g | sed s/.*"\[echo\] "//g | sed s/")"/"):"/g >> $FILE
      echo >> $FILE
   fi
}

# creates the body of the output
function classes_create_body()
{
   TMP=`echo ${i##*/}`
   
   if [ "$TYPE" = "html" ]
   then
      echo "<tr><td><pre>" >> $FILE
      echo "<pre>" >> $FILE
      cat $HELP | grep "\[java\]\|\|.*:" | grep -v "^commandline\|^gui:\|^Buildfile\|BUILD \|Command line\|GUI\|[0-9]* seconds" | sed s/.*"\[java\]\|\[echo\]"//g | sed s/"<"/"\&lt;"/g | sed s/">"/"\&gt;"/g | sed s/^[a-zA-Z].*:/"\<a name=\"&\">\<\/a>"/g | sed s/":\""/\"/g | sed s/":<"/"<"/g >> $FILE
      echo "</pre><br></td></tr>" >> $FILE
   elif [ "$TYPE" = "text" ]
   then
      cat $HELP | grep "\[java\]\|\|.*:" | grep -v "^commandline\|^gui:\|^Buildfile\|BUILD \|Command line\|GUI\|[0-9]* seconds" | sed s/.*"\[java\]\|\[echo\]"//g | sed s/"<"/"\&lt;"/g | sed s/">"/"\&gt;"/g | sed s/^[a-zA-Z].*://g >> $FILE
      echo >> $FILE
   fi
}

# creates the footer of the output
function classes_create_footer()
{
   if [ "$TYPE" = "html" ]
   then
      echo "</table>" >> $FILE
      echo "</center>" >> $FILE
      echo "<p align=\"right\"><i><font size=\"-1\">Last updated: $DATE</font></i></p>" >> $FILE
      echo "</body>" >> $FILE
      echo "</html>" >> $FILE
   elif [ "$TYPE" = "text" ]
   then
      echo >> $FILE
      echo "Last updated: $DATE" >> $FILE
   fi
}

#########################
# interprete parameters #
#########################

ANT=$ANT_HOME/bin/ant
JAVA="$JAVA_HOME/bin/java"
JAR="$JAVA_HOME/bin/java -jar"
ROOT=`expr "$0" : '\(.*\)/'`
XMLDIR="$ROOT/../xml"
OUTPUT="$ROOT/../doc"
TYPE="html"
DELETE="no"
DATE=`date`

while getopts ":ho:t:d" flag
do
   case $flag in
      o) OUTPUT=$OPTARG
         ;;
      t) if [ ! "$OPTARG" = "html" ] && [ ! "$OPTARG" = "text" ]
         then
            echo "Type '$OPTARG' is not recognized!"
            usage
            exit 2
         else
            TYPE=$OPTARG
         fi
         ;;
      d) DELETE="yes"
         ;;
      h) usage
         exit 0
         ;;
      *) usage
         exit 1
         ;;
   esac
done

####################
# start processing #
####################

# delete output?
if [ "$DELETE" = "yes" ]
then
   rm -R $OUTPUT
fi

# create output dirs
create_dirs

#######
# TOC #
#######

echo
echo "creating TOC..."
create_toc

###########
# SCRIPTS #
###########

echo
echo "Scripts:"

# build filename with path
if [ "$TYPE" = "html" ]
then
   FILENAME="index.html"
elif [ "$TYPE" = "text" ]
then
   FILENAME="scripts.txt"
fi
TMP="$OUTPUT/scripts";build_filename

echo "creating header...";scripts_create_header
echo "creating body...";scripts_create_body
echo "creating footer...";scripts_create_footer
echo "created '$FILE'!"

###########
# CLASSES #
###########

echo
echo "Classes:"
HELP=$ROOT/../tmp/help.txt
"$ANT" -f $XMLDIR/help.xml > $HELP

# build filename with path
if [ "$TYPE" = "html" ]
then
   FILENAME="index.html"
elif [ "$TYPE" = "text" ]
then
   FILENAME="classes.txt"
fi
TMP="$OUTPUT/classes";build_filename

echo "creating header...";classes_create_header
echo "creating body...";classes_create_body
echo "creating footer...";classes_create_footer
echo "created '$FILE'!"

rm $HELP


##################
# Manual/JAVADOC #
##################

echo
echo "Manual/Javadoc:"
"$ANT" -Djavadoc=$OUTPUT/javadoc doc

############
# finished #
############

echo "finished!"

