# -*- coding: utf-8 -*-
"""
# ---------------------------------------------------------------------------------------------------------
# ProjectName:  redis-helper
# FileName:     lock.py
# Description:  支持自动续期的互斥任务锁
# Author:       ASUS
# CreateDate:   2026/01/12
# Copyright ©2011-2026. Hunan xxxxxxx Company limited. All rights reserved.
# ---------------------------------------------------------------------------------------------------------
"""
import asyncio
from typing import Optional
import redis.asyncio as redis


class MutexTaskLock:
    def __init__(self, redis_client: redis.Redis, lock_key: str, ttl: int = 300):
        self.redis = redis_client
        self.lock_key = lock_key
        self.ttl = ttl  # 锁的基础 TTL（秒）
        self._renew_task: Optional[asyncio.Task] = None
        self._stop_renew = asyncio.Event()

    async def acquire(self) -> bool:
        """尝试获取锁"""
        return await self.redis.set(self.lock_key, "1", nx=True, ex=self.ttl)

    async def release(self):
        """释放锁"""
        await self.redis.delete(self.lock_key)

    async def _renew_lock_periodically(self):
        """后台协程：定期续期锁"""
        try:
            while not self._stop_renew.is_set():
                # 每隔 ttl/3 时间续一次期
                await asyncio.wait_for(self._stop_renew.wait(), timeout=self.ttl // 3)
                if self._stop_renew.is_set():
                    break
                # 续期：只有当前持有者才能续（通过 Lua 脚本更安全，此处简化）
                await self.redis.expire(self.lock_key, self.ttl)
        except asyncio.TimeoutError:
            # 正常超时，继续循环
            pass
        except (Exception,):
            # 忽略续期异常，不影响主任务
            pass

    async def __aenter__(self):
        acquired = await self.acquire()
        if acquired:
            # 启动续期协程
            self._stop_renew.clear()
            self._renew_task = asyncio.create_task(self._renew_lock_periodically())
        return acquired

    async def __aexit__(self, exc_type, exc_val, exc_tb):
        if self._renew_task:
            self._stop_renew.set()
            try:
                await self._renew_task
            except (Exception,):
                pass
        # 释放锁（即使续期失败也要删）
        await self.release()
