"""
Gestion du stockage des clés (/storage/keys).
"""

from __future__ import annotations

from typing import Any

from rundeck.base import RundeckObject, RundeckObjectManager


class StorageKey(RundeckObject):
    """Ressource de stockage (fichier ou dossier)."""

    _id_attr = "path"
    _repr_attr = "name"


class StorageKeyManager(RundeckObjectManager[StorageKey]):
    """Manager pour les clés/ressources du storage."""

    _path = "/storage/keys"
    _obj_cls = StorageKey

    def _dir_path(self, path: str | None = None) -> str:
        base = self._build_path()
        if path:
            cleaned = path.strip("/")
            return f"{base}/{cleaned}/"
        return base

    def _file_path(self, path: str) -> str:
        base = self._build_path()
        cleaned = path.strip("/")
        return f"{base}/{cleaned}"

    def list(self, path: str | None = None) -> list[StorageKey]:
        """Liste les ressources (fichiers/dossiers) sous le chemin donné."""
        target = self._dir_path(path)
        result = self.rd.http_get(target)
        resources = []
        if isinstance(result, dict):
            resources = result.get("resources") or []
        return self._wrap_list(resources)

    def get_metadata(self, path: str) -> StorageKey:
        """Récupère les métadonnées d'une clé/fichier."""
        target = self._file_path(path)
        result = self.rd.http_get(target)
        if isinstance(result, dict):
            return self._wrap(result)
        return self._wrap({"path": path, "raw": result})

    def get_content(self, path: str, accept: str = "application/pgp-keys") -> bytes:
        """Récupère le contenu d'une clé (ex: clé publique)."""
        target = self._file_path(path)
        headers = {"Accept": accept} if accept else None
        response = self.rd.http_get(target, headers=headers, raw=True)
        return response.content

    def create(self, path: str, content: bytes | str, content_type: str) -> Any:
        """Crée une nouvelle clé/fichier à l'emplacement donné."""
        target = self._file_path(path)
        headers = {"Content-Type": content_type, "Accept": "application/json"}
        return self.rd.http_post(target, data=content, headers=headers)

    def update(self, path: str, content: bytes | str, content_type: str) -> Any:
        """Met à jour une clé/fichier existant."""
        target = self._file_path(path)
        headers = {"Content-Type": content_type, "Accept": "application/json"}
        return self.rd.http_put(target, data=content, headers=headers)

    def delete(self, path: str) -> None:
        """Supprime une clé/fichier."""
        target = self._file_path(path)
        self.rd.http_delete(target)


__all__ = ["StorageKey", "StorageKeyManager"]
