"""Module to handle client registry url."""

import typing
import urllib
from collections import defaultdict


class UrlManager:
    """Instantiate url related and Path related to the registry client."""

    def __init__(self, base_url: str, paths: typing.List[typing.Tuple[str, str, str]]) -> None:
        parsed_url = urllib.parse.urlparse(base_url)

        assert parsed_url.scheme in (  # noqa: S101
            "http",
            "https",
        ), f"The url {base_url} has invalid schema. Use http or https. For example http://{base_url}"

        # this is the absolute url to the server
        # make sure that url ends with /
        if not base_url.endswith("/"):
            base_url += "/"

        self.base_url = base_url

        self.paths = {path.name: path for path in (Path(path) for path in paths)}

    @property
    def url(self) -> str:
        """Return base url."""
        return self.base_url

    def url_for(self, func: str, **kwargs: typing.Any) -> tuple:
        """Generate a url for a given function."""
        path = self.paths[func]
        url = path.generate_url(**kwargs)

        return urllib.parse.urljoin(self.base_url, url), path.method


class Path:
    """Associate an action related to a path & method."""

    def __init__(self, path: typing.Tuple[str, str, str]) -> None:
        self.func = path[0]
        self.url = path[1]
        self.method = path[2]

    @property
    def name(self) -> str:
        return self.func

    def generate_url(self, **kwargs: typing.Any) -> str:
        parameters = {key: value for key, value in kwargs.items() if value}

        return self.url.format_map(defaultdict(str, **parameters))
