# Copyright 2022 David Harcombe
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
from __future__ import annotations

import dataclasses
from typing import Optional

import aenum as enum
import dataclasses_json
import immutabledict

from . import camel_field, lazy_property


@dataclasses_json.dataclass_json
@dataclasses.dataclass
class ServiceDefinition(object):
  """Defines a Google Service for the builder."""
  service_name: Optional[str] = camel_field()
  version: Optional[str] = camel_field()
  discovery_service_url: Optional[str] = camel_field()


class S(enum.Enum):
  """Defines the generic Enum for any service.

  Raises:
      ValueError: if no enum is defined.
  """
  @lazy_property
  def definition(self) -> ServiceDefinition:
    """Fetch the ServiceDefinition.

    Lazily returns the dataclass containing the service definition
    details. It has to be lazy, as it can't be defined at
    initialization time.

    Returns:
        ServiceDefinition: the service definition
    """
    (service_name, version, url) = SERVICE_DEFINITIONS.get(self.name)
    return ServiceDefinition(
        service_name=service_name,
        version=version,
        discovery_service_url=url
        # (
        #     f'https://{service_name}.googleapis.com/$discovery/rest'
        #     f'?version={version}')
    )

  @classmethod
  def from_value(cls, value: str) -> S:
    """Creates a service enum from the name of the service.

    Args:
        value (str): the service name

    Raises:
        ValueError: no service found

    Returns:
        S: the service definition
    """
    for k, v in cls.__members__.items():
      if k == value.upper():
        return v
    else:
      raise ValueError(f"'{cls.__name__}' enum not found for '{value}'")


SERVICE_DEFINITIONS = \
    immutabledict.immutabledict(
        {'ABUSIVEEXPERIENCEREPORT': ('abusiveexperiencereport',
                                     'v1',
                                     'https://abusiveexperiencereport.googleapis.com/$discovery/rest?version=v1'),
         'ACCELERATEDMOBILEPAGEURL': ('acceleratedmobilepageurl',
                                      'v1',
                                      'https://acceleratedmobilepageurl.googleapis.com/$discovery/rest?version=v1'),
         'ACCESSAPPROVAL': ('accessapproval',
                            'v1',
                            'https://accessapproval.googleapis.com/$discovery/rest?version=v1'),
         'ACCESSCONTEXTMANAGER': ('accesscontextmanager',
                                  'v1',
                                  'https://accesscontextmanager.googleapis.com/$discovery/rest?version=v1'),
         'ADDRESSVALIDATION': ('addressvalidation',
                               'v1',
                               'https://addressvalidation.googleapis.com/$discovery/rest?version=v1'),
         'ADEXCHANGEBUYER2': ('adexchangebuyer2',
                              'v2beta1',
                              'https://adexchangebuyer.googleapis.com/$discovery/rest?version=v2beta1'),
         'ADEXPERIENCEREPORT': ('adexperiencereport',
                                'v1',
                                'https://adexperiencereport.googleapis.com/$discovery/rest?version=v1'),
         'ADMIN': ('admin',
                   'reports_v1',
                   'https://admin.googleapis.com/$discovery/rest?version=reports_v1'),
         'ADMOB': ('admob',
                   'v1',
                   'https://admob.googleapis.com/$discovery/rest?version=v1'),
         'ADSENSE': ('adsense',
                     'v2',
                     'https://adsense.googleapis.com/$discovery/rest?version=v2'),
         'ADSENSEPLATFORM': ('adsenseplatform',
                             'v1',
                             'https://adsenseplatform.googleapis.com/$discovery/rest?version=v1'),
         'ADVISORYNOTIFICATIONS': ('advisorynotifications',
                                   'v1',
                                   'https://advisorynotifications.googleapis.com/$discovery/rest?version=v1'),
         'AIPLATFORM': ('aiplatform',
                        'v1',
                        'https://aiplatform.googleapis.com/$discovery/rest?version=v1'),
         'AIRQUALITY': ('airquality',
                        'v1',
                        'https://airquality.googleapis.com/$discovery/rest?version=v1'),
         'ALERTCENTER': ('alertcenter',
                         'v1beta1',
                         'https://alertcenter.googleapis.com/$discovery/rest?version=v1beta1'),
         'ALLOYDB': ('alloydb',
                     'v1',
                     'https://alloydb.googleapis.com/$discovery/rest?version=v1'),
         'ANALYTICS': ('analytics',
                       'v3',
                       'https://analytics.googleapis.com/$discovery/rest?version=v3'),
         'ANALYTICSADMIN': ('analyticsadmin',
                            'v1beta',
                            'https://analyticsadmin.googleapis.com/$discovery/rest?version=v1beta'),
         'ANALYTICSDATA': ('analyticsdata',
                           'v1beta',
                           'https://analyticsdata.googleapis.com/$discovery/rest?version=v1beta'),
         'ANALYTICSHUB': ('analyticshub',
                          'v1',
                          'https://analyticshub.googleapis.com/$discovery/rest?version=v1'),
         'ANDROIDDEVICEPROVISIONING': ('androiddeviceprovisioning',
                                       'v1',
                                       'https://androiddeviceprovisioning.googleapis.com/$discovery/rest?version=v1'),
         'ANDROIDENTERPRISE': ('androidenterprise',
                               'v1',
                               'https://androidenterprise.googleapis.com/$discovery/rest?version=v1'),
         'ANDROIDMANAGEMENT': ('androidmanagement',
                               'v1',
                               'https://androidmanagement.googleapis.com/$discovery/rest?version=v1'),
         'ANDROIDPUBLISHER': ('androidpublisher',
                              'v3',
                              'https://androidpublisher.googleapis.com/$discovery/rest?version=v3'),
         'APIGATEWAY': ('apigateway',
                        'v1',
                        'https://apigateway.googleapis.com/$discovery/rest?version=v1'),
         'APIGEE': ('apigee',
                    'v1',
                    'https://apigee.googleapis.com/$discovery/rest?version=v1'),
         'APIGEEREGISTRY': ('apigeeregistry',
                            'v1',
                            'https://apigeeregistry.googleapis.com/$discovery/rest?version=v1'),
         'APIHUB': ('apihub',
                    'v1',
                    'https://apihub.googleapis.com/$discovery/rest?version=v1'),
         'APIKEYS': ('apikeys',
                     'v2',
                     'https://apikeys.googleapis.com/$discovery/rest?version=v2'),
         'APIM': ('apim',
                  'v1alpha',
                  'https://apim.googleapis.com/$discovery/rest?version=v1alpha'),
         'APPENGINE': ('appengine',
                       'v1',
                       'https://appengine.googleapis.com/$discovery/rest?version=v1'),
         'APPHUB': ('apphub',
                    'v1',
                    'https://apphub.googleapis.com/$discovery/rest?version=v1'),
         'AREA120TABLES': ('area120tables',
                           'v1alpha1',
                           'https://area120tables.googleapis.com/$discovery/rest?version=v1alpha1'),
         'AREAINSIGHTS': ('areainsights',
                          'v1',
                          'https://areainsights.googleapis.com/$discovery/rest?version=v1'),
         'ARTIFACTREGISTRY': ('artifactregistry',
                              'v1',
                              'https://artifactregistry.googleapis.com/$discovery/rest?version=v1'),
         'ASSUREDWORKLOADS': ('assuredworkloads',
                              'v1',
                              'https://assuredworkloads.googleapis.com/$discovery/rest?version=v1'),
         'AUTHORIZEDBUYERSMARKETPLACE': ('authorizedbuyersmarketplace',
                                         'v1',
                                         'https://authorizedbuyersmarketplace.googleapis.com/$discovery/rest?version=v1'),
         'BACKUPDR': ('backupdr',
                      'v1',
                      'https://backupdr.googleapis.com/$discovery/rest?version=v1'),
         'BAREMETALSOLUTION': ('baremetalsolution',
                               'v2',
                               'https://baremetalsolution.googleapis.com/$discovery/rest?version=v2'),
         'BATCH': ('batch',
                   'v1',
                   'https://batch.googleapis.com/$discovery/rest?version=v1'),
         'BEYONDCORP': ('beyondcorp',
                        'v1',
                        'https://beyondcorp.googleapis.com/$discovery/rest?version=v1'),
         'BIGLAKE': ('biglake',
                     'v1',
                     'https://biglake.googleapis.com/$discovery/rest?version=v1'),
         'BIGQUERY': ('bigquery',
                      'v2',
                      'https://bigquery.googleapis.com/$discovery/rest?version=v2'),
         'BIGQUERYCONNECTION': ('bigqueryconnection',
                                'v1',
                                'https://bigqueryconnection.googleapis.com/$discovery/rest?version=v1'),
         'BIGQUERYDATAPOLICY': ('bigquerydatapolicy',
                                'v1',
                                'https://bigquerydatapolicy.googleapis.com/$discovery/rest?version=v1'),
         'BIGQUERYDATATRANSFER': ('bigquerydatatransfer',
                                  'v1',
                                  'https://bigquerydatatransfer.googleapis.com/$discovery/rest?version=v1'),
         'BIGQUERYRESERVATION': ('bigqueryreservation',
                                 'v1',
                                 'https://bigqueryreservation.googleapis.com/$discovery/rest?version=v1'),
         'BIGTABLEADMIN': ('bigtableadmin',
                           'v2',
                           'https://bigtableadmin.googleapis.com/$discovery/rest?version=v2'),
         'BILLINGBUDGETS': ('billingbudgets',
                            'v1',
                            'https://billingbudgets.googleapis.com/$discovery/rest?version=v1'),
         'BINARYAUTHORIZATION': ('binaryauthorization',
                                 'v1',
                                 'https://binaryauthorization.googleapis.com/$discovery/rest?version=v1'),
         'BLOCKCHAINNODEENGINE': ('blockchainnodeengine',
                                  'v1',
                                  'https://blockchainnodeengine.googleapis.com/$discovery/rest?version=v1'),
         'BLOGGER': ('blogger',
                     'v3',
                     'https://blogger.googleapis.com/$discovery/rest?version=v3'),
         'BOOKS': ('books',
                   'v1',
                   'https://books.googleapis.com/$discovery/rest?version=v1'),
         'BUSINESSPROFILEPERFORMANCE': ('businessprofileperformance',
                                        'v1',
                                        'https://businessprofileperformance.googleapis.com/$discovery/rest?version=v1'),
         'CALENDAR': ('calendar',
                      'v3',
                      'https://calendar-json.googleapis.com/$discovery/rest?version=v3'),
         'CERTIFICATEMANAGER': ('certificatemanager',
                                'v1',
                                'https://certificatemanager.googleapis.com/$discovery/rest?version=v1'),
         'CHAT': ('chat',
                  'v1',
                  'https://chat.googleapis.com/$discovery/rest?version=v1'),
         'CHECKS': ('checks',
                    'v1alpha',
                    'https://checks.googleapis.com/$discovery/rest?version=v1alpha'),
         'CHROMEMANAGEMENT': ('chromemanagement',
                              'v1',
                              'https://chromemanagement.googleapis.com/$discovery/rest?version=v1'),
         'CHROMEPOLICY': ('chromepolicy',
                          'v1',
                          'https://chromepolicy.googleapis.com/$discovery/rest?version=v1'),
         'CHROMEUXREPORT': ('chromeuxreport',
                            'v1',
                            'https://chromeuxreport.googleapis.com/$discovery/rest?version=v1'),
         'CIVICINFO': ('civicinfo',
                       'v2',
                       'https://civicinfo.googleapis.com/$discovery/rest?version=v2'),
         'CLASSROOM': ('classroom',
                       'v1',
                       'https://classroom.googleapis.com/$discovery/rest?version=v1'),
         'CLOUDASSET': ('cloudasset',
                        'v1',
                        'https://cloudasset.googleapis.com/$discovery/rest?version=v1'),
         'CLOUDBILLING': ('cloudbilling',
                          'v1',
                          'https://cloudbilling.googleapis.com/$discovery/rest?version=v1'),
         'CLOUDBUILD': ('cloudbuild',
                        'v2',
                        'https://cloudbuild.googleapis.com/$discovery/rest?version=v2'),
         'CLOUDCHANNEL': ('cloudchannel',
                          'v1',
                          'https://cloudchannel.googleapis.com/$discovery/rest?version=v1'),
         'CLOUDCONTROLSPARTNER': ('cloudcontrolspartner',
                                  'v1',
                                  'https://cloudcontrolspartner.googleapis.com/$discovery/rest?version=v1'),
         'CLOUDDEPLOY': ('clouddeploy',
                         'v1',
                         'https://clouddeploy.googleapis.com/$discovery/rest?version=v1'),
         'CLOUDERRORREPORTING': ('clouderrorreporting',
                                 'v1beta1',
                                 'https://clouderrorreporting.googleapis.com/$discovery/rest?version=v1beta1'),
         'CLOUDFUNCTIONS': ('cloudfunctions',
                            'v2',
                            'https://cloudfunctions.googleapis.com/$discovery/rest?version=v2'),
         'CLOUDIDENTITY': ('cloudidentity',
                           'v1',
                           'https://cloudidentity.googleapis.com/$discovery/rest?version=v1'),
         'CLOUDKMS': ('cloudkms',
                      'v1',
                      'https://cloudkms.googleapis.com/$discovery/rest?version=v1'),
         'CLOUDLOCATIONFINDER': ('cloudlocationfinder',
                                 'v1alpha',
                                 'https://cloudlocationfinder.googleapis.com/$discovery/rest?version=v1alpha'),
         'CLOUDPROFILER': ('cloudprofiler',
                           'v2',
                           'https://cloudprofiler.googleapis.com/$discovery/rest?version=v2'),
         'CLOUDRESOURCEMANAGER': ('cloudresourcemanager',
                                  'v3',
                                  'https://cloudresourcemanager.googleapis.com/$discovery/rest?version=v3'),
         'CLOUDSCHEDULER': ('cloudscheduler',
                            'v1',
                            'https://cloudscheduler.googleapis.com/$discovery/rest?version=v1'),
         'CLOUDSEARCH': ('cloudsearch',
                         'v1',
                         'https://cloudsearch.googleapis.com/$discovery/rest?version=v1'),
         'CLOUDSHELL': ('cloudshell',
                        'v1',
                        'https://cloudshell.googleapis.com/$discovery/rest?version=v1'),
         'CLOUDSUPPORT': ('cloudsupport',
                          'v2',
                          'https://cloudsupport.googleapis.com/$discovery/rest?version=v2'),
         'CLOUDTASKS': ('cloudtasks',
                        'v2',
                        'https://cloudtasks.googleapis.com/$discovery/rest?version=v2'),
         'CLOUDTRACE': ('cloudtrace',
                        'v2',
                        'https://cloudtrace.googleapis.com/$discovery/rest?version=v2'),
         'COMPOSER': ('composer',
                      'v1',
                      'https://composer.googleapis.com/$discovery/rest?version=v1'),
         'COMPUTE': ('compute',
                     'v1',
                     'https://www.googleapis.com/discovery/v1/apis/compute/v1/rest'),
         'CONFIG': ('config',
                    'v1',
                    'https://config.googleapis.com/$discovery/rest?version=v1'),
         'CONNECTORS': ('connectors',
                        'v2',
                        'https://connectors.googleapis.com/$discovery/rest?version=v2'),
         'CONTACTCENTERAIPLATFORM': ('contactcenteraiplatform',
                                     'v1alpha1',
                                     'https://contactcenteraiplatform.googleapis.com/$discovery/rest?version=v1alpha1'),
         'CONTACTCENTERINSIGHTS': ('contactcenterinsights',
                                   'v1',
                                   'https://contactcenterinsights.googleapis.com/$discovery/rest?version=v1'),
         'CONTAINER': ('container',
                       'v1',
                       'https://container.googleapis.com/$discovery/rest?version=v1'),
         'CONTAINERANALYSIS': ('containeranalysis',
                               'v1',
                               'https://containeranalysis.googleapis.com/$discovery/rest?version=v1'),
         'CONTENT': ('content',
                     'v2.1',
                     'https://shoppingcontent.googleapis.com/$discovery/rest?version=v2.1'),
         'CONTENTWAREHOUSE': ('contentwarehouse',
                              'v1',
                              'https://contentwarehouse.googleapis.com/$discovery/rest?version=v1'),
         'CSS': ('css', 'v1', 'https://css.googleapis.com/$discovery/rest?version=v1'),
         'CUSTOMSEARCH': ('customsearch',
                          'v1',
                          'https://customsearch.googleapis.com/$discovery/rest?version=v1'),
         'DATACATALOG': ('datacatalog',
                         'v1',
                         'https://datacatalog.googleapis.com/$discovery/rest?version=v1'),
         'DATAFLOW': ('dataflow',
                      'v1b3',
                      'https://dataflow.googleapis.com/$discovery/rest?version=v1b3'),
         'DATAFORM': ('dataform',
                      'v1beta1',
                      'https://dataform.googleapis.com/$discovery/rest?version=v1beta1'),
         'DATAFUSION': ('datafusion',
                        'v1',
                        'https://datafusion.googleapis.com/$discovery/rest?version=v1'),
         'DATALABELING': ('datalabeling',
                          'v1beta1',
                          'https://datalabeling.googleapis.com/$discovery/rest?version=v1beta1'),
         'DATALINEAGE': ('datalineage',
                         'v1',
                         'https://datalineage.googleapis.com/$discovery/rest?version=v1'),
         'DATAMIGRATION': ('datamigration',
                           'v1',
                           'https://datamigration.googleapis.com/$discovery/rest?version=v1'),
         'DATAPIPELINES': ('datapipelines',
                           'v1',
                           'https://datapipelines.googleapis.com/$discovery/rest?version=v1'),
         'DATAPLEX': ('dataplex',
                      'v1',
                      'https://dataplex.googleapis.com/$discovery/rest?version=v1'),
         'DATAPORTABILITY': ('dataportability',
                             'v1',
                             'https://dataportability.googleapis.com/$discovery/rest?version=v1'),
         'DATAPROC': ('dataproc',
                      'v1',
                      'https://dataproc.googleapis.com/$discovery/rest?version=v1'),
         'DATASTORE': ('datastore',
                       'v1',
                       'https://datastore.googleapis.com/$discovery/rest?version=v1'),
         'DATASTREAM': ('datastream',
                        'v1',
                        'https://datastream.googleapis.com/$discovery/rest?version=v1'),
         'DEPLOYMENTMANAGER': ('deploymentmanager',
                               'v2',
                               'https://deploymentmanager.googleapis.com/$discovery/rest?version=v2'),
         'DEVELOPERCONNECT': ('developerconnect',
                              'v1',
                              'https://developerconnect.googleapis.com/$discovery/rest?version=v1'),
         'DFAREPORTING': ('dfareporting',
                          'v4',
                          'https://dfareporting.googleapis.com/$discovery/rest?version=v4'),
         'DIALOGFLOW': ('dialogflow',
                        'v3',
                        'https://dialogflow.googleapis.com/$discovery/rest?version=v3'),
         'DIGITALASSETLINKS': ('digitalassetlinks',
                               'v1',
                               'https://digitalassetlinks.googleapis.com/$discovery/rest?version=v1'),
         'DISCOVERY': ('discovery',
                       'v1',
                       'https://discovery.googleapis.com/$discovery/rest?version=v1'),
         'DISCOVERYENGINE': ('discoveryengine',
                             'v1',
                             'https://discoveryengine.googleapis.com/$discovery/rest?version=v1'),
         'DISPLAYVIDEO': ('displayvideo',
                          'v4',
                          'https://displayvideo.googleapis.com/$discovery/rest?version=v4'),
         'DLP': ('dlp', 'v2', 'https://dlp.googleapis.com/$discovery/rest?version=v2'),
         'DNS': ('dns', 'v1', 'https://dns.googleapis.com/$discovery/rest?version=v1'),
         'DOCS': ('docs',
                  'v1',
                  'https://docs.googleapis.com/$discovery/rest?version=v1'),
         'DOCUMENTAI': ('documentai',
                        'v1',
                        'https://documentai.googleapis.com/$discovery/rest?version=v1'),
         'DOMAINS': ('domains',
                     'v1',
                     'https://domains.googleapis.com/$discovery/rest?version=v1'),
         'DOUBLECLICKBIDMANAGER': ('doubleclickbidmanager',
                                   'v2',
                                   'https://doubleclickbidmanager.googleapis.com/$discovery/rest?version=v2'),
         'DOUBLECLICKSEARCH': ('doubleclicksearch',
                               'v2',
                               'https://doubleclicksearch.googleapis.com/$discovery/rest?version=v2'),
         'DRIVE': ('drive',
                   'v3',
                   'https://www.googleapis.com/discovery/v1/apis/drive/v3/rest'),
         'DRIVEACTIVITY': ('driveactivity',
                           'v2',
                           'https://driveactivity.googleapis.com/$discovery/rest?version=v2'),
         'DRIVELABELS': ('drivelabels',
                         'v2',
                         'https://drivelabels.googleapis.com/$discovery/rest?version=v2'),
         'ESSENTIALCONTACTS': ('essentialcontacts',
                               'v1',
                               'https://essentialcontacts.googleapis.com/$discovery/rest?version=v1'),
         'EVENTARC': ('eventarc',
                      'v1',
                      'https://eventarc.googleapis.com/$discovery/rest?version=v1'),
         'FACTCHECKTOOLS': ('factchecktools',
                            'v1alpha1',
                            'https://factchecktools.googleapis.com/$discovery/rest?version=v1alpha1'),
         'FCM': ('fcm', 'v1', 'https://fcm.googleapis.com/$discovery/rest?version=v1'),
         'FCMDATA': ('fcmdata',
                     'v1beta1',
                     'https://fcmdata.googleapis.com/$discovery/rest?version=v1beta1'),
         'FILE': ('file',
                  'v1',
                  'https://file.googleapis.com/$discovery/rest?version=v1'),
         'FIREBASE': ('firebase',
                      'v1beta1',
                      'https://firebase.googleapis.com/$discovery/rest?version=v1beta1'),
         'FIREBASEAPPCHECK': ('firebaseappcheck',
                              'v1',
                              'https://firebaseappcheck.googleapis.com/$discovery/rest?version=v1'),
         'FIREBASEAPPDISTRIBUTION': ('firebaseappdistribution',
                                     'v1',
                                     'https://firebaseappdistribution.googleapis.com/$discovery/rest?version=v1'),
         'FIREBASEAPPHOSTING': ('firebaseapphosting',
                                'v1',
                                'https://firebaseapphosting.googleapis.com/$discovery/rest?version=v1'),
         'FIREBASEDATABASE': ('firebasedatabase',
                              'v1beta',
                              'https://firebasedatabase.googleapis.com/$discovery/rest?version=v1beta'),
         'FIREBASEDATACONNECT': ('firebasedataconnect',
                                 'v1',
                                 'https://firebasedataconnect.googleapis.com/$discovery/rest?version=v1'),
         'FIREBASEDYNAMICLINKS': ('firebasedynamiclinks',
                                  'v1',
                                  'https://firebasedynamiclinks.googleapis.com/$discovery/rest?version=v1'),
         'FIREBASEHOSTING': ('firebasehosting',
                             'v1',
                             'https://firebasehosting.googleapis.com/$discovery/rest?version=v1'),
         'FIREBASEML': ('firebaseml',
                        'v1',
                        'https://firebaseml.googleapis.com/$discovery/rest?version=v1'),
         'FIREBASERULES': ('firebaserules',
                           'v1',
                           'https://firebaserules.googleapis.com/$discovery/rest?version=v1'),
         'FIREBASESTORAGE': ('firebasestorage',
                             'v1beta',
                             'https://firebasestorage.googleapis.com/$discovery/rest?version=v1beta'),
         'FIRESTORE': ('firestore',
                       'v1',
                       'https://firestore.googleapis.com/$discovery/rest?version=v1'),
         'FITNESS': ('fitness',
                     'v1',
                     'https://fitness.googleapis.com/$discovery/rest?version=v1'),
         'FORMS': ('forms',
                   'v1',
                   'https://forms.googleapis.com/$discovery/rest?version=v1'),
         'GAMES': ('games',
                   'v1',
                   'https://games.googleapis.com/$discovery/rest?version=v1'),
         'GAMESCONFIGURATION': ('gamesConfiguration',
                                'v1configuration',
                                'https://gamesconfiguration.googleapis.com/$discovery/rest?version=v1configuration'),
         'GAMESMANAGEMENT': ('gamesManagement',
                             'v1management',
                             'https://gamesmanagement.googleapis.com/$discovery/rest?version=v1management'),
         'GKEBACKUP': ('gkebackup',
                       'v1',
                       'https://gkebackup.googleapis.com/$discovery/rest?version=v1'),
         'GKEHUB': ('gkehub',
                    'v2',
                    'https://gkehub.googleapis.com/$discovery/rest?version=v2'),
         'GKEONPREM': ('gkeonprem',
                       'v1',
                       'https://gkeonprem.googleapis.com/$discovery/rest?version=v1'),
         'GMAIL': ('gmail',
                   'v1',
                   'https://gmail.googleapis.com/$discovery/rest?version=v1'),
         'GMAILPOSTMASTERTOOLS': ('gmailpostmastertools',
                                  'v1',
                                  'https://gmailpostmastertools.googleapis.com/$discovery/rest?version=v1'),
         'GROUPSMIGRATION': ('groupsmigration',
                             'v1',
                             'https://groupsmigration.googleapis.com/$discovery/rest?version=v1'),
         'GROUPSSETTINGS': ('groupssettings',
                            'v1',
                            'https://groupssettings.googleapis.com/$discovery/rest?version=v1'),
         'HEALTHCARE': ('healthcare',
                        'v1',
                        'https://healthcare.googleapis.com/$discovery/rest?version=v1'),
         'HOMEGRAPH': ('homegraph',
                       'v1',
                       'https://homegraph.googleapis.com/$discovery/rest?version=v1'),
         'IAM': ('iam', 'v2', 'https://iam.googleapis.com/$discovery/rest?version=v2'),
         'IAMCREDENTIALS': ('iamcredentials',
                            'v1',
                            'https://iamcredentials.googleapis.com/$discovery/rest?version=v1'),
         'IAP': ('iap', 'v1', 'https://iap.googleapis.com/$discovery/rest?version=v1'),
         'IDENTITYTOOLKIT': ('identitytoolkit',
                             'v3',
                             'https://identitytoolkit.googleapis.com/$discovery/rest?version=v3'),
         'IDS': ('ids', 'v1', 'https://ids.googleapis.com/$discovery/rest?version=v1'),
         'INDEXING': ('indexing',
                      'v3',
                      'https://indexing.googleapis.com/$discovery/rest?version=v3'),
         'INTEGRATIONS': ('integrations',
                          'v1',
                          'https://integrations.googleapis.com/$discovery/rest?version=v1'),
         'JOBS': ('jobs',
                  'v4',
                  'https://jobs.googleapis.com/$discovery/rest?version=v4'),
         'KEEP': ('keep',
                  'v1',
                  'https://keep.googleapis.com/$discovery/rest?version=v1'),
         'KGSEARCH': ('kgsearch',
                      'v1',
                      'https://kgsearch.googleapis.com/$discovery/rest?version=v1'),
         'KMSINVENTORY': ('kmsinventory',
                          'v1',
                          'https://kmsinventory.googleapis.com/$discovery/rest?version=v1'),
         'LANGUAGE': ('language',
                      'v2',
                      'https://language.googleapis.com/$discovery/rest?version=v2'),
         'LIBRARYAGENT': ('libraryagent',
                          'v1',
                          'https://libraryagent.googleapis.com/$discovery/rest?version=v1'),
         'LICENSING': ('licensing',
                       'v1',
                       'https://licensing.googleapis.com/$discovery/rest?version=v1'),
         'LIFESCIENCES': ('lifesciences',
                          'v2beta',
                          'https://lifesciences.googleapis.com/$discovery/rest?version=v2beta'),
         'LOCALSERVICES': ('localservices',
                           'v1',
                           'https://localservices.googleapis.com/$discovery/rest?version=v1'),
         'LOGGING': ('logging',
                     'v2',
                     'https://logging.googleapis.com/$discovery/rest?version=v2'),
         'LOOKER': ('looker',
                    'v1',
                    'https://looker.googleapis.com/$discovery/rest?version=v1'),
         'MANAGEDIDENTITIES': ('managedidentities',
                               'v1',
                               'https://managedidentities.googleapis.com/$discovery/rest?version=v1'),
         'MANAGEDKAFKA': ('managedkafka',
                          'v1',
                          'https://managedkafka.googleapis.com/$discovery/rest?version=v1'),
         'MANUFACTURERS': ('manufacturers',
                           'v1',
                           'https://manufacturers.googleapis.com/$discovery/rest?version=v1'),
         'MARKETINGPLATFORMADMIN': ('marketingplatformadmin',
                                    'v1alpha',
                                    'https://marketingplatformadmin.googleapis.com/$discovery/rest?version=v1alpha'),
         'MEET': ('meet',
                  'v2',
                  'https://meet.googleapis.com/$discovery/rest?version=v2'),
         'MEMCACHE': ('memcache',
                      'v1',
                      'https://memcache.googleapis.com/$discovery/rest?version=v1'),
         'MERCHANTAPI': ('merchantapi',
                         'reviews_v1beta',
                         'https://merchantapi.googleapis.com/$discovery/rest?version=reviews_v1beta'),
         'METASTORE': ('metastore',
                       'v2',
                       'https://metastore.googleapis.com/$discovery/rest?version=v2'),
         'MIGRATIONCENTER': ('migrationcenter',
                             'v1',
                             'https://migrationcenter.googleapis.com/$discovery/rest?version=v1'),
         'ML': ('ml', 'v1', 'https://ml.googleapis.com/$discovery/rest?version=v1'),
         'MONITORING': ('monitoring',
                        'v3',
                        'https://monitoring.googleapis.com/$discovery/rest?version=v3'),
         'MYBUSINESSACCOUNTMANAGEMENT': ('mybusinessaccountmanagement',
                                         'v1',
                                         'https://mybusinessaccountmanagement.googleapis.com/$discovery/rest?version=v1'),
         'MYBUSINESSBUSINESSINFORMATION': ('mybusinessbusinessinformation',
                                           'v1',
                                           'https://mybusinessbusinessinformation.googleapis.com/$discovery/rest?version=v1'),
         'MYBUSINESSLODGING': ('mybusinesslodging',
                               'v1',
                               'https://mybusinesslodging.googleapis.com/$discovery/rest?version=v1'),
         'MYBUSINESSNOTIFICATIONS': ('mybusinessnotifications',
                                     'v1',
                                     'https://mybusinessnotifications.googleapis.com/$discovery/rest?version=v1'),
         'MYBUSINESSPLACEACTIONS': ('mybusinessplaceactions',
                                    'v1',
                                    'https://mybusinessplaceactions.googleapis.com/$discovery/rest?version=v1'),
         'MYBUSINESSQANDA': ('mybusinessqanda',
                             'v1',
                             'https://mybusinessqanda.googleapis.com/$discovery/rest?version=v1'),
         'MYBUSINESSVERIFICATIONS': ('mybusinessverifications',
                                     'v1',
                                     'https://mybusinessverifications.googleapis.com/$discovery/rest?version=v1'),
         'NETAPP': ('netapp',
                    'v1',
                    'https://netapp.googleapis.com/$discovery/rest?version=v1'),
         'NETWORKCONNECTIVITY': ('networkconnectivity',
                                 'v1',
                                 'https://networkconnectivity.googleapis.com/$discovery/rest?version=v1'),
         'NETWORKMANAGEMENT': ('networkmanagement',
                               'v1',
                               'https://networkmanagement.googleapis.com/$discovery/rest?version=v1'),
         'NETWORKSECURITY': ('networksecurity',
                             'v1',
                             'https://networksecurity.googleapis.com/$discovery/rest?version=v1'),
         'NETWORKSERVICES': ('networkservices',
                             'v1',
                             'https://networkservices.googleapis.com/$discovery/rest?version=v1'),
         'NOTEBOOKS': ('notebooks',
                       'v2',
                       'https://notebooks.googleapis.com/$discovery/rest?version=v2'),
         'OAUTH2': ('oauth2',
                    'v2',
                    'https://www.googleapis.com/discovery/v1/apis/oauth2/v2/rest'),
         'OBSERVABILITY': ('observability',
                           'v1',
                           'https://observability.googleapis.com/$discovery/rest?version=v1'),
         'ONDEMANDSCANNING': ('ondemandscanning',
                              'v1',
                              'https://ondemandscanning.googleapis.com/$discovery/rest?version=v1'),
         'ORACLEDATABASE': ('oracledatabase',
                            'v1',
                            'https://oracledatabase.googleapis.com/$discovery/rest?version=v1'),
         'ORGPOLICY': ('orgpolicy',
                       'v2',
                       'https://orgpolicy.googleapis.com/$discovery/rest?version=v2'),
         'OSCONFIG': ('osconfig',
                      'v2',
                      'https://osconfig.googleapis.com/$discovery/rest?version=v2'),
         'OSLOGIN': ('oslogin',
                     'v1',
                     'https://oslogin.googleapis.com/$discovery/rest?version=v1'),
         'PAGESPEEDONLINE': ('pagespeedonline',
                             'v5',
                             'https://pagespeedonline.googleapis.com/$discovery/rest?version=v5'),
         'PARALLELSTORE': ('parallelstore',
                           'v1',
                           'https://parallelstore.googleapis.com/$discovery/rest?version=v1'),
         'PARAMETERMANAGER': ('parametermanager',
                              'v1',
                              'https://parametermanager.googleapis.com/$discovery/rest?version=v1'),
         'PAYMENTSRESELLERSUBSCRIPTION': ('paymentsresellersubscription',
                                          'v1',
                                          'https://paymentsresellersubscription.googleapis.com/$discovery/rest?version=v1'),
         'PEOPLE': ('people',
                    'v1',
                    'https://people.googleapis.com/$discovery/rest?version=v1'),
         'PLACES': ('places',
                    'v1',
                    'https://places.googleapis.com/$discovery/rest?version=v1'),
         'PLAYCUSTOMAPP': ('playcustomapp',
                           'v1',
                           'https://playcustomapp.googleapis.com/$discovery/rest?version=v1'),
         'PLAYDEVELOPERREPORTING': ('playdeveloperreporting',
                                    'v1beta1',
                                    'https://playdeveloperreporting.googleapis.com/$discovery/rest?version=v1beta1'),
         'PLAYGROUPING': ('playgrouping',
                          'v1alpha1',
                          'https://playgrouping.googleapis.com/$discovery/rest?version=v1alpha1'),
         'PLAYINTEGRITY': ('playintegrity',
                           'v1',
                           'https://playintegrity.googleapis.com/$discovery/rest?version=v1'),
         'POLICYANALYZER': ('policyanalyzer',
                            'v1',
                            'https://policyanalyzer.googleapis.com/$discovery/rest?version=v1'),
         'POLICYSIMULATOR': ('policysimulator',
                             'v1',
                             'https://policysimulator.googleapis.com/$discovery/rest?version=v1'),
         'POLICYTROUBLESHOOTER': ('policytroubleshooter',
                                  'v1',
                                  'https://policytroubleshooter.googleapis.com/$discovery/rest?version=v1'),
         'POLLEN': ('pollen',
                    'v1',
                    'https://pollen.googleapis.com/$discovery/rest?version=v1'),
         'POLY': ('poly',
                  'v1',
                  'https://poly.googleapis.com/$discovery/rest?version=v1'),
         'PRIVATECA': ('privateca',
                       'v1',
                       'https://privateca.googleapis.com/$discovery/rest?version=v1'),
         'PROD_TT_SASPORTAL': ('prod_tt_sasportal',
                               'v1alpha1',
                               'https://prod-tt-sasportal.googleapis.com/$discovery/rest?version=v1alpha1'),
         'PUBLICCA': ('publicca',
                      'v1',
                      'https://publicca.googleapis.com/$discovery/rest?version=v1'),
         'PUBSUB': ('pubsub',
                    'v1',
                    'https://pubsub.googleapis.com/$discovery/rest?version=v1'),
         'PUBSUBLITE': ('pubsublite',
                        'v1',
                        'https://pubsublite.googleapis.com/$discovery/rest?version=v1'),
         'RAPIDMIGRATIONASSESSMENT': ('rapidmigrationassessment',
                                      'v1',
                                      'https://rapidmigrationassessment.googleapis.com/$discovery/rest?version=v1'),
         'READERREVENUESUBSCRIPTIONLINKING': ('readerrevenuesubscriptionlinking',
                                              'v1',
                                              'https://readerrevenuesubscriptionlinking.googleapis.com/$discovery/rest?version=v1'),
         'REALTIMEBIDDING': ('realtimebidding',
                             'v1',
                             'https://realtimebidding.googleapis.com/$discovery/rest?version=v1'),
         'RECAPTCHAENTERPRISE': ('recaptchaenterprise',
                                 'v1',
                                 'https://recaptchaenterprise.googleapis.com/$discovery/rest?version=v1'),
         'RECOMMENDATIONENGINE': ('recommendationengine',
                                  'v1beta1',
                                  'https://recommendationengine.googleapis.com/$discovery/rest?version=v1beta1'),
         'RECOMMENDER': ('recommender',
                         'v1',
                         'https://recommender.googleapis.com/$discovery/rest?version=v1'),
         'REDIS': ('redis',
                   'v1',
                   'https://redis.googleapis.com/$discovery/rest?version=v1'),
         'RESELLER': ('reseller',
                      'v1',
                      'https://reseller.googleapis.com/$discovery/rest?version=v1'),
         'RETAIL': ('retail',
                    'v2',
                    'https://retail.googleapis.com/$discovery/rest?version=v2'),
         'RUN': ('run', 'v2', 'https://run.googleapis.com/$discovery/rest?version=v2'),
         'RUNTIMECONFIG': ('runtimeconfig',
                           'v1',
                           'https://runtimeconfig.googleapis.com/$discovery/rest?version=v1'),
         'SAASSERVICEMGMT': ('saasservicemgmt',
                             'v1beta1',
                             'https://saasservicemgmt.googleapis.com/$discovery/rest?version=v1beta1'),
         'SAFEBROWSING': ('safebrowsing',
                          'v5',
                          'https://safebrowsing.googleapis.com/$discovery/rest?version=v5'),
         'SASPORTAL': ('sasportal',
                       'v1alpha1',
                       'https://sasportal.googleapis.com/$discovery/rest?version=v1alpha1'),
         'SCRIPT': ('script',
                    'v1',
                    'https://script.googleapis.com/$discovery/rest?version=v1'),
         'SEARCHADS360': ('searchads360',
                          'v0',
                          'https://searchads360.googleapis.com/$discovery/rest?version=v0'),
         'SEARCHCONSOLE': ('searchconsole',
                           'v1',
                           'https://searchconsole.googleapis.com/$discovery/rest?version=v1'),
         'SECRETMANAGER': ('secretmanager',
                           'v1',
                           'https://secretmanager.googleapis.com/$discovery/rest?version=v1'),
         'SECURITYCENTER': ('securitycenter',
                            'v1',
                            'https://securitycenter.googleapis.com/$discovery/rest?version=v1'),
         'SECURITYPOSTURE': ('securityposture',
                             'v1',
                             'https://securityposture.googleapis.com/$discovery/rest?version=v1'),
         'SERVICECONSUMERMANAGEMENT': ('serviceconsumermanagement',
                                       'v1',
                                       'https://serviceconsumermanagement.googleapis.com/$discovery/rest?version=v1'),
         'SERVICECONTROL': ('servicecontrol',
                            'v2',
                            'https://servicecontrol.googleapis.com/$discovery/rest?version=v2'),
         'SERVICEDIRECTORY': ('servicedirectory',
                              'v1',
                              'https://servicedirectory.googleapis.com/$discovery/rest?version=v1'),
         'SERVICEMANAGEMENT': ('servicemanagement',
                               'v1',
                               'https://servicemanagement.googleapis.com/$discovery/rest?version=v1'),
         'SERVICENETWORKING': ('servicenetworking',
                               'v1',
                               'https://servicenetworking.googleapis.com/$discovery/rest?version=v1'),
         'SERVICEUSAGE': ('serviceusage',
                          'v1',
                          'https://serviceusage.googleapis.com/$discovery/rest?version=v1'),
         'SHEETS': ('sheets',
                    'v4',
                    'https://sheets.googleapis.com/$discovery/rest?version=v4'),
         'SITEVERIFICATION': ('siteVerification',
                              'v1',
                              'https://siteverification.googleapis.com/$discovery/rest?version=v1'),
         'SLIDES': ('slides',
                    'v1',
                    'https://slides.googleapis.com/$discovery/rest?version=v1'),
         'SMARTDEVICEMANAGEMENT': ('smartdevicemanagement',
                                   'v1',
                                   'https://smartdevicemanagement.googleapis.com/$discovery/rest?version=v1'),
         'SOLAR': ('solar',
                   'v1',
                   'https://solar.googleapis.com/$discovery/rest?version=v1'),
         'SPANNER': ('spanner',
                     'v1',
                     'https://spanner.googleapis.com/$discovery/rest?version=v1'),
         'SPEECH': ('speech',
                    'v1',
                    'https://speech.googleapis.com/$discovery/rest?version=v1'),
         'SQLADMIN': ('sqladmin',
                      'v1',
                      'https://sqladmin.googleapis.com/$discovery/rest?version=v1'),
         'STORAGE': ('storage',
                     'v1',
                     'https://storage.googleapis.com/$discovery/rest?version=v1'),
         'STORAGEBATCHOPERATIONS': ('storagebatchoperations',
                                    'v1',
                                    'https://storagebatchoperations.googleapis.com/$discovery/rest?version=v1'),
         'STORAGETRANSFER': ('storagetransfer',
                             'v1',
                             'https://storagetransfer.googleapis.com/$discovery/rest?version=v1'),
         'STREETVIEWPUBLISH': ('streetviewpublish',
                               'v1',
                               'https://streetviewpublish.googleapis.com/$discovery/rest?version=v1'),
         'STS': ('sts', 'v1', 'https://sts.googleapis.com/$discovery/rest?version=v1'),
         'TAGMANAGER': ('tagmanager',
                        'v2',
                        'https://tagmanager.googleapis.com/$discovery/rest?version=v2'),
         'TASKS': ('tasks',
                   'v1',
                   'https://tasks.googleapis.com/$discovery/rest?version=v1'),
         'TESTING': ('testing',
                     'v1',
                     'https://testing.googleapis.com/$discovery/rest?version=v1'),
         'TEXTTOSPEECH': ('texttospeech',
                          'v1',
                          'https://texttospeech.googleapis.com/$discovery/rest?version=v1'),
         'TOOLRESULTS': ('toolresults',
                         'v1beta3',
                         'https://toolresults.googleapis.com/$discovery/rest?version=v1beta3'),
         'TPU': ('tpu', 'v2', 'https://tpu.googleapis.com/$discovery/rest?version=v2'),
         'TRAFFICDIRECTOR': ('trafficdirector',
                             'v3',
                             'https://trafficdirector.googleapis.com/$discovery/rest?version=v3'),
         'TRANSCODER': ('transcoder',
                        'v1',
                        'https://transcoder.googleapis.com/$discovery/rest?version=v1'),
         'TRANSLATE': ('translate',
                       'v3',
                       'https://translation.googleapis.com/$discovery/rest?version=v3'),
         'TRAVELIMPACTMODEL': ('travelimpactmodel',
                               'v1',
                               'https://travelimpactmodel.googleapis.com/$discovery/rest?version=v1'),
         'VAULT': ('vault',
                   'v1',
                   'https://vault.googleapis.com/$discovery/rest?version=v1'),
         'VERIFIEDACCESS': ('verifiedaccess',
                            'v2',
                            'https://verifiedaccess.googleapis.com/$discovery/rest?version=v2'),
         'VERSIONHISTORY': ('versionhistory',
                            'v1',
                            'https://versionhistory.googleapis.com/$discovery/rest?version=v1'),
         'VIDEOINTELLIGENCE': ('videointelligence',
                               'v1',
                               'https://videointelligence.googleapis.com/$discovery/rest?version=v1'),
         'VISION': ('vision',
                    'v1',
                    'https://vision.googleapis.com/$discovery/rest?version=v1'),
         'VMMIGRATION': ('vmmigration',
                         'v1',
                         'https://vmmigration.googleapis.com/$discovery/rest?version=v1'),
         'VMWAREENGINE': ('vmwareengine',
                          'v1',
                          'https://vmwareengine.googleapis.com/$discovery/rest?version=v1'),
         'VPCACCESS': ('vpcaccess',
                       'v1',
                       'https://vpcaccess.googleapis.com/$discovery/rest?version=v1'),
         'WALLETOBJECTS': ('walletobjects',
                           'v1',
                           'https://walletobjects.googleapis.com/$discovery/rest?version=v1'),
         'WEBFONTS': ('webfonts',
                      'v1',
                      'https://webfonts.googleapis.com/$discovery/rest?version=v1'),
         'WEBRISK': ('webrisk',
                     'v1',
                     'https://webrisk.googleapis.com/$discovery/rest?version=v1'),
         'WEBSECURITYSCANNER': ('websecurityscanner',
                                'v1',
                                'https://websecurityscanner.googleapis.com/$discovery/rest?version=v1'),
         'WORKFLOWEXECUTIONS': ('workflowexecutions',
                                'v1',
                                'https://workflowexecutions.googleapis.com/$discovery/rest?version=v1'),
         'WORKFLOWS': ('workflows',
                       'v1',
                       'https://workflows.googleapis.com/$discovery/rest?version=v1'),
         'WORKLOADMANAGER': ('workloadmanager',
                             'v1',
                             'https://workloadmanager.googleapis.com/$discovery/rest?version=v1'),
         'WORKSPACEEVENTS': ('workspaceevents',
                             'v1',
                             'https://workspaceevents.googleapis.com/$discovery/rest?version=v1'),
         'WORKSTATIONS': ('workstations',
                          'v1',
                          'https://workstations.googleapis.com/$discovery/rest?version=v1'),
         'YOUTUBE': ('youtube',
                     'v3',
                     'https://youtube.googleapis.com/$discovery/rest?version=v3'),
         'YOUTUBEANALYTICS': ('youtubeAnalytics',
                              'v2',
                              'https://youtubeanalytics.googleapis.com/$discovery/rest?version=v2'),
         'YOUTUBEREPORTING': ('youtubereporting',
                              'v1',
                              'https://youtubereporting.googleapis.com/$discovery/rest?version=v1')})


Service = S('Service', list(SERVICE_DEFINITIONS.keys()))
