from __future__ import print_function
from __future__ import division

# Import the modules to read the dynamical matrix
import cellconstructor as CC
import cellconstructor.Phonons

# Import the SCHA modules
import sscha, sscha.Ensemble


# Here the input information
DATA_DIR = "data"
N_RANDOM = 10
DYN_PREFIX = "../ensemble_data_test/dyn"
NQIRR = 1
T = 0
POPULATION = 1


# An explanation of what this script does
INFO = """
GENERATE A NEW ENSEMBLE
=======================

In this example we generate a new ensemble using the python-sscha package.

The ensemble is generated starting from the dynamical matrix located at {},
and will be saved into the data directory {}.

Several files will be produced.
The id of the esnemble (required to save more in the same directory)
will be {}.

We will generate {} configurations at temperature {} K.

FILES:
{}/scf_population{}_X.dat

where X is the configuration ID, 
it is a file the atomic type and coordinates in alat units. 
in cartesian space (Quantum Espresso format) of the configuration.
This is the only human user-friendly file, and what you need to compute
energy, forces and stresses.

FILES:
{}/u_population{}_X.dat

The displacements of the atoms with respect to the central nuclear position.
Do not edit this file (in bohr)

FILES:
{}/forces_population{}_X.dat
For each atom of configuration X, 
the cartesian force expressed in Ry/Bohr.
You must overwrite this file with the result of a calculation.

FILES:
{}/pressures_population{}_X.dat
The stress tensor of the X configuration. Expressed in Ry/Bohr^3
Also this file is empty, you must use a calculator and fill this file.
Delete these files if you do not want to calculate the stress.

FILE:
{}/energies_supercell_population{}.dat
The list of energies for each configuration X, expressed in Ry.
Also this file is empty and must be filled with your calculator.


Note, you can also configure an ASE (Atomic Simulation Environment)
calculator, and pass it to the ensemble as explained in this script.
In this way it will take care of automatically computing everything
and saving it in the correct way.
If you use an ASE calculator, you can also save the ensemble
in compressed binary format, to save a lot of disk space.
This is also explained in a comment to this script.


""".format(DYN_PREFIX, DATA_DIR, POPULATION, N_RANDOM, T,
           DATA_DIR, POPULATION,
           DATA_DIR, POPULATION,
           DATA_DIR, POPULATION,
           DATA_DIR, POPULATION,
           DATA_DIR, POPULATION)

# Print the info in the screen
print(INFO)
print()
print(" ========== RUNNING =========== ")
print()


print("Loading the dynamical matrix...")
dyn = CC.Phonons.Phonons(DYN_PREFIX, NQIRR)
print("The loaded dynamical matrix has a supercell of", dyn.GetSupercell())

print("")
print("Generating the ensemble of {} configurations".format(N_RANDOM))
ens = sscha.Ensemble.Ensemble(dyn, T, dyn.GetSupercell())
# Evenodd keyword is used to reduce the stochastic noise, by generating symmetric configurations
# Around the centroid positions. It requires an even ensemble or the code will complain.
ens.generate(N_RANDOM, evenodd = True)


print("")
print("Saving the ensemble into {}, with id = {}...".format(DATA_DIR, POPULATION))
ens.save(DATA_DIR, POPULATION)

print("Done.")

# The compuation of forces, energies and stresses can be done with:

#ens.compute_ensemble(calc, cluster = None)

# Where calc is an ASE Calculator object. (Look ASE documentation)
# You can also provide a sscha.Cluster.Cluster object to the cluster keyword
# To submit automatically the calculation to a remote server
# through a queue system.
# Detailed explanation on this class can be found in the cluster example.

# After forces and energies have been computed,
# the ensemble could be saved in binary to save space.
# With the command:

#ens.save_bin(DATA_DIR, POPULATION)





